<?php
/**
 * النتائج السنوية (فصلين دراسيين)
 * Annual Results (Two Semesters)
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';
require_once __DIR__ . '/classes/ResultsArchive.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$resultsArchive = new ResultsArchive($db);

$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$academic_year = isset($_GET['academic_year']) ? $_GET['academic_year'] : '';
$message = '';

// اسم الجامعة
$university_name = 'جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology';

// دالة لتحويل الدرجة الرقمية إلى تقدير حرفي
// التقدير الجديد: A (80-100), B (70-79), C (60-69), D (50-59), F (<50)
function getLetterGradeFromMarks($marks) {
    if ($marks >= 80 && $marks <= 100) {
        return 'A';
    } elseif ($marks >= 70 && $marks <= 79) {
        return 'B';
    } elseif ($marks >= 60 && $marks <= 69) {
        return 'C';
    } elseif ($marks >= 50 && $marks <= 59) {
        return 'D';
    } else {
        return 'F';
    }
}

// دالة لتحويل المعدل إلى تقدير حرفي (للتوافق مع الكود القديم)
function getLetterGradeFromGPA($gpa) {
    // تحويل GPA من 5 إلى درجة رقمية تقريبية (GPA * 20)
    $estimated_marks = $gpa * 20;
    return getLetterGradeFromMarks($estimated_marks);
}

// دالة لحساب التقدير النصي بناءً على المعدل السنوي (المعدل من 5)
// التقدير: ممتاز (>= 4)، جيد جداً (>= 3.5)، جيد (>= 3)، مقبول (>= 2.5)، إعادة السنة (>= 2)، فصل (>= 1)
function getTextGrade($gpa, $failed_credits = 0) {
    // ممتاز: >= 4
    if ($gpa >= 4) {
        return 'ممتاز';
    }
    // جيد جداً: >= 3.5
    elseif ($gpa >= 3.5) {
        return 'جيد جداً';
    }
    // جيد: >= 3
    elseif ($gpa >= 3) {
        return 'جيد';
    }
    // مقبول: >= 2.5
    elseif ($gpa >= 2.5) {
        return 'مقبول';
    }
    // إعادة السنة: >= 2
    elseif ($gpa >= 2) {
        return 'إعادة السنة';
    }
    // فصل: >= 1
    elseif ($gpa >= 1) {
        return 'فصل';
    }
    // أقل من 1
    else {
        return 'فصل';
    }
}

// دالة لحساب حالة الطالب السنوية بناءً على المعدل السنوي والمواد الراسبة
// القواعد: فصل إذا المعدل < 1.5، إعادة السنة إذا المعدل < 2.5، ملاحق إذا رسب في مواد ومعدله >= 2.5
function calculateAnnualStudentStatus($student_grades, $all_courses, $total_credits, $annual_gpa = 0) {
    $failed_credits = 0;
    $failed_courses = [];
    $no_grade_courses = [];
    $is_frozen = false;
    
    // حساب الساعات المعتمدة الراسبة
    foreach ($all_courses as $course) {
        $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
        
        if ($grade) {
            $letter_grade = isset($grade['letter_grade']) ? trim($grade['letter_grade']) : '';
            $marks = isset($grade['marks']) ? $grade['marks'] : null;
            
            $is_failed = false;
            
            // التحقق من وجود "ت" (تجميد)
            $marks_str = $marks !== null ? (string)$marks : '';
            $letter_grade_str = $letter_grade !== null ? (string)$letter_grade : '';
            if (strpos($marks_str, 'ت') !== false || strpos($letter_grade_str, 'ت') !== false) {
                $is_frozen = true;
            }
            
            // التحقق من الرسوب
            if ($marks !== null && !is_numeric($marks)) {
                $marks_text = trim($marks);
                if ($marks_text == 'ب') $marks_text = 'بد';
                
                if ($marks_text == 'بد') {
                    $is_failed = false;
                    $no_grade_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code']];
                } elseif ($marks_text == 'غ' || $marks_text == 'غياب' || 
                         $marks_text == 'غش' || $marks_text == 'غشش') {
                    $is_failed = true;
                } elseif (strpos($marks_text, 'ت') !== false) {
                    $is_failed = false;
                    $is_frozen = true;
                } elseif ($marks_text == 'مع' || strpos($marks_text, 'مع') !== false) {
                    $is_failed = false;
                }
            } elseif ($letter_grade) {
                $letter_grade_trimmed = trim($letter_grade);
                if ($letter_grade_trimmed == 'ب') $letter_grade_trimmed = 'بد';
                
                if ($letter_grade_trimmed == 'بد') {
                    $is_failed = false;
                    $no_grade_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code']];
                } elseif ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' ||
                         $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش') {
                    $is_failed = true;
                } elseif (strpos($letter_grade_trimmed, 'ت') !== false) {
                    $is_failed = false;
                    $is_frozen = true;
                } elseif ($letter_grade_trimmed == 'مع' || strpos($letter_grade_trimmed, 'مع') !== false) {
                    $is_failed = false;
                } elseif ($letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                    $is_failed = true;
                }
            } elseif ($marks !== null && is_numeric($marks)) {
                $is_failed = ((float)$marks < 50);
            }
            
            if ($is_failed) {
                $failed_credits += (float)$course['credit_hours'];
                $failed_courses[] = [
                    'name_ar' => $course['name_ar'],
                    'code' => $course['code'],
                    'credits' => (float)$course['credit_hours']
                ];
            }
        }
    }
    
    $notes = '';
    $status = 'ناجح';
    
    // التحقق من تجميد
    if ($is_frozen) {
        $status = 'تجميد';
        $notes = 'الطالب مجمد';
        return ['status' => $status, 'notes' => $notes, 'failed_credits' => $failed_credits, 'failed_courses' => $failed_courses];
    }
    
    // القواعد الجديدة: بناءً على المعدل السنوي والمواد الراسبة
    // فصل: إذا كان المعدل < 1.5
    if ($annual_gpa < 1.5 && $annual_gpa > 0) {
        $status = 'فصل';
        $notes = 'المعدل أقل من 1.5 (' . number_format($annual_gpa, 2) . ')';
        if ($failed_credits > 0) {
            $failed_course_names = array_map(function($course) {
                return $course['name_ar'];
            }, $failed_courses);
            $notes .= ' - راسب في: ' . implode('، ', $failed_course_names);
        }
    }
    // إعادة السنة: إذا كان المعدل < 2.5
    elseif ($annual_gpa < 2.5) {
        $status = 'إعادة السنة';
        if ($failed_credits > 0) {
            $failed_course_names = array_map(function($course) {
                return $course['name_ar'];
            }, $failed_courses);
            $notes = 'المعدل أقل من 2.5 (' . number_format($annual_gpa, 2) . ') - راسب في: ' . implode('، ', $failed_course_names);
        } else {
            $notes = 'المعدل أقل من 2.5 (' . number_format($annual_gpa, 2) . ')';
        }
        if (!empty($no_grade_courses)) {
            $no_grade_names = array_map(function($course) {
                return $course['name_ar'];
            }, $no_grade_courses);
            if ($notes) $notes .= ' | ';
            $notes .= 'بديل: ' . implode('، ', $no_grade_names);
        }
    }
    // ملاحق: إذا رسب في مواد ومعدله >= 2.5
    elseif ($failed_credits > 0 && $annual_gpa >= 2.5) {
        $status = 'ملاحق';
        $failed_course_names = array_map(function($course) {
            return $course['name_ar'];
        }, $failed_courses);
        
        $notes_parts = [];
        if (!empty($no_grade_courses)) {
            $no_grade_names = array_map(function($course) {
                return $course['name_ar'];
            }, $no_grade_courses);
            $notes_parts[] = 'بديل: ' . implode('، ', $no_grade_names);
        }
        if (!empty($failed_course_names)) {
            $notes_parts[] = 'ملاحق في: ' . implode('، ', $failed_course_names);
        }
        $notes = implode(' + ', $notes_parts);
    }
    // ناجح: إذا لم يرسب في أي مادة
    else {
        $status = 'ناجح';
        if (!empty($no_grade_courses)) {
            $no_grade_names = array_map(function($course) {
                return $course['name_ar'];
            }, $no_grade_courses);
            $notes = 'بديل: ' . implode('، ', $no_grade_names);
        }
    }
    
    return [
        'status' => $status,
        'notes' => $notes,
        'failed_credits' => $failed_credits,
        'failed_courses' => $failed_courses
    ];
}

// جلب الكليات
$colleges = [];
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $user = $auth->getCurrentUser();
    if ($user['college_id']) {
        $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges WHERE id = ?", [$user['college_id']]);
    }
}

// جلب التخصصات
$majors = [];
if ($college_id) {
    $majors = $db->fetchAll("SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar", [$college_id]);
}

// جلب السنوات الأكاديمية
$academic_years = $db->fetchAll(
    "SELECT DISTINCT academic_year FROM semesters ORDER BY academic_year DESC"
);

$semester1_id = 0;
$semester2_id = 0;
$college_info = null;
$major_info = null;
$students_data = [];
$courses_list_sem1 = [];
$courses_list_sem2 = [];

// إذا تم اختيار جميع المعايير، جلب معرفات الفصلين الدراسيين
if ($college_id && $major_id && $academic_year) {
    // جلب الفصل الأول (semester_number = 1)
    $semester1 = $db->fetchOne(
        "SELECT id FROM semesters 
         WHERE major_id = ? AND semester_number = 1 AND academic_year = ?",
        [$major_id, $academic_year]
    );
    if ($semester1) {
        $semester1_id = $semester1['id'];
    }
    
    // جلب الفصل الثاني (semester_number = 2)
    $semester2 = $db->fetchOne(
        "SELECT id FROM semesters 
         WHERE major_id = ? AND semester_number = 2 AND academic_year = ?",
        [$major_id, $academic_year]
    );
    if ($semester2) {
        $semester2_id = $semester2['id'];
    }
    
    // جلب معلومات الكلية والتخصص
    $college_info = $db->fetchOne("SELECT id, name_ar FROM colleges WHERE id = ?", [$college_id]);
    $major_info = $db->fetchOne("SELECT id, name_ar FROM majors WHERE id = ?", [$major_id]);
    
    if ($semester1_id || $semester2_id) {
        // جلب المواد للفصل الأول
        if ($semester1_id) {
            $courses_list_sem1 = $db->fetchAll(
                "SELECT c.id, c.code, c.name_ar, c.credit_hours, c.is_alternative
                 FROM courses c
                 WHERE c.semester_id = ?
                 ORDER BY c.code",
                [$semester1_id]
            );
        }
        
        // جلب المواد للفصل الثاني
        if ($semester2_id) {
            $courses_list_sem2 = $db->fetchAll(
                "SELECT c.id, c.code, c.name_ar, c.credit_hours, c.is_alternative
                 FROM courses c
                 WHERE c.semester_id = ?
                 ORDER BY c.code",
                [$semester2_id]
            );
        }
        
        // دمج قائمة المواد (جميع المواد من الفصلين)
        $all_courses = array_merge($courses_list_sem1, $courses_list_sem2);
        
        // حساب إجمالي الساعات المعتمدة
        $total_credits = 0;
        foreach ($all_courses as $course) {
            $total_credits += (float)$course['credit_hours'];
        }
        
        // جلب جميع الطلاب المسجلين في أي من الفصلين
        $student_ids = [];
        if ($semester1_id) {
            $students_sem1 = $db->fetchAll(
                "SELECT DISTINCT student_id FROM student_courses WHERE semester_id = ?",
                [$semester1_id]
            );
            foreach ($students_sem1 as $s) {
                $student_ids[$s['student_id']] = true;
            }
        }
        if ($semester2_id) {
            $students_sem2 = $db->fetchAll(
                "SELECT DISTINCT student_id FROM student_courses WHERE semester_id = ?",
                [$semester2_id]
            );
            foreach ($students_sem2 as $s) {
                $student_ids[$s['student_id']] = true;
            }
        }
        
        // جلب معلومات الطلاب
        if (!empty($student_ids)) {
            $student_ids_list = array_keys($student_ids);
            $placeholders = implode(',', array_fill(0, count($student_ids_list), '?'));
            $students = $db->fetchAll(
                "SELECT id, student_number, full_name_ar
                 FROM students
                 WHERE id IN ($placeholders)
                 ORDER BY student_number",
                $student_ids_list
            );
            
            // جلب الدرجات والمعدلات لكل طالب
            foreach ($students as $student) {
                $student_grades = [];
                $total_points = 0;
                $total_credit_hours = 0;
                
                // جلب درجات الفصل الأول
                if ($semester1_id) {
                    foreach ($courses_list_sem1 as $course) {
                        $grade = $db->fetchOne(
                            "SELECT g.marks, g.letter_grade, g.points
                             FROM grades g
                             WHERE g.student_id = ? AND g.course_id = ? AND g.semester_id = ?",
                            [$student['id'], $course['id'], $semester1_id]
                        );
                        
                        $student_grades[$course['id']] = $grade ? [
                            'marks' => $grade['marks'],
                            'letter_grade' => $grade['letter_grade'],
                            'points' => $grade['points'],
                            'semester' => 1
                        ] : null;
                        
                        // حساب المعدل السنوي (تجاهل مواد البديل والمواد المعفى منها)
                        // الصيغة الجديدة: (الدرجة/20) * الساعات المعتمدة
                        if ($grade && $grade['marks'] !== null && is_numeric($grade['marks'])) {
                            if (!isset($course['is_alternative']) || !$course['is_alternative']) {
                                $marks_str = (string)($grade['marks'] ?? '');
                                $letter_str = (string)($grade['letter_grade'] ?? '');
                                if (strpos($marks_str, 'مع') === false && strpos($letter_str, 'مع') === false) {
                                    $marks_value = (float)$grade['marks'];
                                    $credit_hours = (float)$course['credit_hours'];
                                    $total_points += ($marks_value / 20.0) * $credit_hours;
                                    $total_credit_hours += $credit_hours;
                                }
                            }
                        }
                    }
                }
                
                // جلب درجات الفصل الثاني
                if ($semester2_id) {
                    foreach ($courses_list_sem2 as $course) {
                        $grade = $db->fetchOne(
                            "SELECT g.marks, g.letter_grade, g.points
                             FROM grades g
                             WHERE g.student_id = ? AND g.course_id = ? AND g.semester_id = ?",
                            [$student['id'], $course['id'], $semester2_id]
                        );
                        
                        $student_grades[$course['id']] = $grade ? [
                            'marks' => $grade['marks'],
                            'letter_grade' => $grade['letter_grade'],
                            'points' => $grade['points'],
                            'semester' => 2
                        ] : null;
                        
                        // حساب المعدل السنوي
                        // الصيغة الجديدة: (الدرجة/20) * الساعات المعتمدة
                        if ($grade && $grade['marks'] !== null && is_numeric($grade['marks'])) {
                            if (!isset($course['is_alternative']) || !$course['is_alternative']) {
                                $marks_str = (string)($grade['marks'] ?? '');
                                $letter_str = (string)($grade['letter_grade'] ?? '');
                                if (strpos($marks_str, 'مع') === false && strpos($letter_str, 'مع') === false) {
                                    $marks_value = (float)$grade['marks'];
                                    $credit_hours = (float)$course['credit_hours'];
                                    $total_points += ($marks_value / 20.0) * $credit_hours;
                                    $total_credit_hours += $credit_hours;
                                }
                            }
                        }
                    }
                }
                
                // حساب المعدل السنوي
                $annual_gpa = $total_credit_hours > 0 ? round($total_points / $total_credit_hours, 2) : 0;
                $letter_grade = $annual_gpa > 0 ? getLetterGradeFromGPA($annual_gpa) : '-';
                
                // جلب المعدلات الفصلية
                $gpa_sem1 = 0;
                $gpa_sem2 = 0;
                if ($semester1_id) {
                    $sem1_gpa = $db->fetchOne(
                        "SELECT gpa FROM semester_gpas WHERE student_id = ? AND semester_id = ?",
                        [$student['id'], $semester1_id]
                    );
                    $gpa_sem1 = $sem1_gpa ? (float)$sem1_gpa['gpa'] : 0;
                }
                if ($semester2_id) {
                    $sem2_gpa = $db->fetchOne(
                        "SELECT gpa FROM semester_gpas WHERE student_id = ? AND semester_id = ?",
                        [$student['id'], $semester2_id]
                    );
                    $gpa_sem2 = $sem2_gpa ? (float)$sem2_gpa['gpa'] : 0;
                }
                
                // حساب حالة الطالب السنوية بناءً على المعدل السنوي والمواد الراسبة
                $annual_status = calculateAnnualStudentStatus($student_grades, $all_courses, $total_credits, $annual_gpa);
                $failed_credits = isset($annual_status['failed_credits']) ? $annual_status['failed_credits'] : 0;
                
                // حساب التقدير النصي بناءً على المعدل السنوي والساعات الراسبة
                $text_grade = $annual_gpa > 0 ? getTextGrade($annual_gpa, $failed_credits) : '-';
                
                $students_data[] = [
                    'id' => $student['id'],
                    'student_number' => $student['student_number'],
                    'full_name_ar' => $student['full_name_ar'],
                    'grades' => $student_grades,
                    'gpa_sem1' => $gpa_sem1,
                    'gpa_sem2' => $gpa_sem2,
                    'annual_gpa' => $annual_gpa,
                    'letter_grade' => $letter_grade,
                    'text_grade' => $text_grade,
                    'total_credits' => $total_credit_hours,
                    'status' => $annual_status['status'],
                    'notes' => $annual_status['notes'],
                    'failed_credits' => $annual_status['failed_credits']
                ];
            }
            
            // ترتيب الطلاب حسب المعدل السنوي (تنازلي)
            usort($students_data, function($a, $b) {
                return $b['annual_gpa'] <=> $a['annual_gpa'];
            });
        }
    }
}

// معالجة حفظ النتائج في الأرشيف (بعد جلب البيانات)
$save_success = false;
$save_error = null;
if (isset($_POST['save_results'])) {
    // الحصول على المعاملات من POST أو GET
    $post_college_id = isset($_POST['college_id']) ? (int)$_POST['college_id'] : $college_id;
    $post_major_id = isset($_POST['major_id']) ? (int)$_POST['major_id'] : $major_id;
    $post_academic_year = isset($_POST['academic_year']) ? $_POST['academic_year'] : $academic_year;
    
    // إعادة جلب البيانات إذا كانت المعاملات مختلفة
    if ($post_college_id != $college_id || $post_major_id != $major_id || $post_academic_year != $academic_year) {
        $college_id = $post_college_id;
        $major_id = $post_major_id;
        $academic_year = $post_academic_year;
        // إعادة توجيه للحصول على البيانات
        header("Location: annual_results.php?college_id=" . $college_id . "&major_id=" . $major_id . "&academic_year=" . urlencode($academic_year) . "&save=1");
        exit;
    }
}

if (isset($_POST['save_results']) && $semester1_id > 0 && !empty($students_data)) {
    $notes = isset($_POST['notes']) ? trim($_POST['notes']) : null;
    try {
        // إعداد بيانات التقرير للحفظ
        $report_data = [
            'students' => [],
            'summary' => [
                'total_students' => count($students_data),
                'academic_year' => $academic_year,
                'college_name' => $college_info ? $college_info['name_ar'] : null,
                'major_name' => $major_info ? $major_info['name_ar'] : null
            ]
        ];
        
        // إضافة بيانات الطلاب
        foreach ($students_data as $student) {
            $report_data['students'][] = [
                'student_id' => $student['id'],
                'student_number' => $student['student_number'],
                'full_name_ar' => $student['full_name_ar'],
                'gpa_sem1' => $student['gpa_sem1'],
                'gpa_sem2' => $student['gpa_sem2'],
                'annual_gpa' => $student['annual_gpa'],
                'text_grade' => $student['text_grade'],
                'status' => $student['status'],
                'notes' => $student['notes'],
                'failed_credits' => $student['failed_credits'],
                'grades' => $student['grades']
            ];
        }
        
        // حساب الإحصائيات
        $pass_count = 0;
        $repeat_year_count = 0;
        $removal_exam_count = 0;
        foreach ($students_data as $student) {
            if ($student['status'] == 'ناجح') {
                $pass_count++;
            } elseif ($student['status'] == 'إعادة السنة') {
                $repeat_year_count++;
            } elseif (strpos($student['status'], 'ملاحق') !== false) {
                $removal_exam_count++;
            }
        }
        
        $report_data['summary']['pass_count'] = $pass_count;
        $report_data['summary']['repeat_year_count'] = $repeat_year_count;
        $report_data['summary']['removal_exam_count'] = $removal_exam_count;
        
        $archive_id = $resultsArchive->saveAnnualResults(
            $semester1_id,
            $_SESSION['user_id'],
            $report_data,
            $college_id,
            $major_id,
            $notes
        );
        $save_success = true;
        // إعادة توجيه بعد الحفظ الناجح
        header("Location: annual_results.php?college_id=" . $college_id . "&major_id=" . $major_id . "&academic_year=" . urlencode($academic_year) . "&saved=1");
        exit;
    } catch (Exception $e) {
        $save_error = $e->getMessage();
        $message = '<div style="background: #f8d7da; color: #721c24; padding: 12px; border-radius: 4px; margin-bottom: 20px;">✗ حدث خطأ عند حفظ النتائج: ' . htmlspecialchars($save_error) . '</div>';
    }
}

// التحقق من رسالة الحفظ الناجح من URL
if (isset($_GET['saved']) && $_GET['saved'] == 1) {
    $save_success = true;
    $message = '<div style="background: #d4edda; color: #155724; padding: 12px; border-radius: 4px; margin-bottom: 20px;">✓ تم حفظ النتائج في الأرشيف بنجاح</div>';
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>النتائج السنوية - نظام نتائج جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology</title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .results-container {
            margin: 20px 0;
        }
        .results-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            font-size: 0.9em;
        }
        .results-table th,
        .results-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
        }
        .results-table th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            font-weight: bold;
        }
        .results-table tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .results-table tr:hover {
            background-color: #f5f5f5;
        }
        .semester-header {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%) !important;
            color: white;
        }
        .no-print {
            @media print {
                display: none;
            }
        }
        .print-button {
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <h1>النتائج السنوية (فصلين دراسيين)</h1>
        
        <?php if ($message): ?>
            <?php echo $message; ?>
        <?php endif; ?>
        
        <!-- نموذج اختيار المعايير -->
        <div class="card no-print" style="margin-bottom: 30px;">
            <h2>اختر المعايير</h2>
            <form method="GET" action="">
                <div class="form-group">
                    <label for="college_id">الكلية:</label>
                    <select name="college_id" id="college_id" required onchange="this.form.submit()">
                        <option value="">-- اختر الكلية --</option>
                        <?php foreach ($colleges as $college): ?>
                            <option value="<?php echo $college['id']; ?>" <?php echo $college_id == $college['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($college['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($college_id): ?>
                <div class="form-group">
                    <label for="major_id">التخصص:</label>
                    <select name="major_id" id="major_id" required onchange="this.form.submit()">
                        <option value="">-- اختر التخصص --</option>
                        <?php foreach ($majors as $major): ?>
                            <option value="<?php echo $major['id']; ?>" <?php echo $major_id == $major['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($major['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($major_id): ?>
                <div class="form-group">
                    <label for="academic_year">السنة الأكاديمية:</label>
                    <select name="academic_year" id="academic_year" required onchange="this.form.submit()">
                        <option value="">-- اختر السنة الأكاديمية --</option>
                        <?php foreach ($academic_years as $year): ?>
                            <option value="<?php echo htmlspecialchars($year['academic_year']); ?>" <?php echo $academic_year == $year['academic_year'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($year['academic_year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
            </form>
        </div>
        
        <!-- عرض النتائج -->
        <?php if (!empty($students_data)): ?>
            <div class="results-container">
                <div class="no-print" style="margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap;">
                    <button onclick="window.print()" class="btn btn-primary print-button">🖨️ طباعة</button>
                    <button type="button" onclick="document.getElementById('save-form').style.display='block';" class="btn btn-success">💾 حفظ في الأرشيف</button>
                </div>
                
                <!-- نموذج حفظ في الأرشيف -->
                <div id="save-form" class="no-print" style="display: none; background: #f9f9f9; padding: 20px; border-radius: 8px; margin-bottom: 20px; border: 2px solid #667eea;">
                    <h3 style="margin-top: 0;">حفظ النتائج في الأرشيف</h3>
                    <form method="POST" action="">
                        <input type="hidden" name="save_results" value="1">
                        <input type="hidden" name="college_id" value="<?php echo $college_id; ?>">
                        <input type="hidden" name="major_id" value="<?php echo $major_id; ?>">
                        <input type="hidden" name="academic_year" value="<?php echo htmlspecialchars($academic_year); ?>">
                        <div class="form-group">
                            <label for="notes">ملاحظات (اختياري):</label>
                            <textarea name="notes" id="notes" rows="3" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" placeholder="أضف ملاحظات إضافية عن هذه النتائج..."></textarea>
                        </div>
                        <div style="display: flex; gap: 10px; margin-top: 15px;">
                            <button type="submit" class="btn btn-success">💾 حفظ</button>
                            <button type="button" onclick="document.getElementById('save-form').style.display='none';" class="btn btn-secondary">إلغاء</button>
                        </div>
                    </form>
                </div>
                
                <div style="text-align: center; margin-bottom: 30px;">
                    <h2 style="font-size: 18px;"><?php echo htmlspecialchars($university_name); ?></h2>
                    <?php if ($college_info): ?>
                        <h3><?php echo htmlspecialchars($college_info['name_ar']); ?></h3>
                    <?php endif; ?>
                    <?php if ($major_info): ?>
                        <h3><?php echo htmlspecialchars($major_info['name_ar']); ?></h3>
                    <?php endif; ?>
                    <h3>النتائج السنوية للعام الدراسي: <?php echo htmlspecialchars($academic_year); ?></h3>
                </div>
                
                <table class="results-table">
                    <thead>
                        <tr>
                            <th rowspan="2">م</th>
                            <th rowspan="2">الرقم الجامعي</th>
                            <th rowspan="2">اسم الطالب</th>
                            <?php if ($semester1_id && !empty($courses_list_sem1)): ?>
                                <th colspan="<?php echo count($courses_list_sem1); ?>" class="semester-header">الفصل الأول</th>
                            <?php endif; ?>
                            <?php if ($semester2_id && !empty($courses_list_sem2)): ?>
                                <th colspan="<?php echo count($courses_list_sem2); ?>" class="semester-header">الفصل الثاني</th>
                            <?php endif; ?>
                            <th rowspan="2">معدل الفصل الأول</th>
                            <th rowspan="2">معدل الفصل الثاني</th>
                            <th rowspan="2">المعدل السنوي</th>
                            <th rowspan="2">التقدير</th>
                            <th rowspan="2">القرار</th>
                            <th rowspan="2">ملاحظات</th>
                        </tr>
                        <tr>
                            <?php if ($semester1_id && !empty($courses_list_sem1)): ?>
                                <?php foreach ($courses_list_sem1 as $course): ?>
                                    <th><?php echo htmlspecialchars($course['code']); ?></th>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            <?php if ($semester2_id && !empty($courses_list_sem2)): ?>
                                <?php foreach ($courses_list_sem2 as $course): ?>
                                    <th><?php echo htmlspecialchars($course['code']); ?></th>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $counter = 1; ?>
                        <?php foreach ($students_data as $student): ?>
                            <tr>
                                <td><?php echo $counter++; ?></td>
                                <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                <td style="text-align: right;"><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                
                                <!-- درجات الفصل الأول -->
                                <?php if ($semester1_id && !empty($courses_list_sem1)): ?>
                                    <?php foreach ($courses_list_sem1 as $course): ?>
                                        <td>
                                            <?php 
                                            $grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                                            if ($grade) {
                                                if ($grade['marks'] !== null && !is_numeric($grade['marks'])) {
                                                    echo htmlspecialchars($grade['marks']);
                                                } elseif ($grade['letter_grade']) {
                                                    echo htmlspecialchars(cleanLetterGrade($grade['letter_grade']));
                                                } elseif ($grade['marks'] !== null) {
                                                    echo htmlspecialchars($grade['marks']);
                                                } else {
                                                    echo '-';
                                                }
                                            } else {
                                                echo '-';
                                            }
                                            ?>
                                        </td>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                                
                                <!-- درجات الفصل الثاني -->
                                <?php if ($semester2_id && !empty($courses_list_sem2)): ?>
                                    <?php foreach ($courses_list_sem2 as $course): ?>
                                        <td>
                                            <?php 
                                            $grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                                            if ($grade) {
                                                if ($grade['marks'] !== null && !is_numeric($grade['marks'])) {
                                                    echo htmlspecialchars($grade['marks']);
                                                } elseif ($grade['letter_grade']) {
                                                    echo htmlspecialchars(cleanLetterGrade($grade['letter_grade']));
                                                } elseif ($grade['marks'] !== null) {
                                                    echo htmlspecialchars($grade['marks']);
                                                } else {
                                                    echo '-';
                                                }
                                            } else {
                                                echo '-';
                                            }
                                            ?>
                                        </td>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                                
                                <td><strong><?php echo $student['gpa_sem1'] > 0 ? number_format($student['gpa_sem1'], 2) : '-'; ?></strong></td>
                                <td><strong><?php echo $student['gpa_sem2'] > 0 ? number_format($student['gpa_sem2'], 2) : '-'; ?></strong></td>
                                <td><strong><?php echo $student['annual_gpa'] > 0 ? number_format($student['annual_gpa'], 2) : '-'; ?></strong></td>
                                <td><strong><?php echo htmlspecialchars($student['text_grade']); ?></strong></td>
                                <td style="text-align: right; font-size: 0.9em;">
                                    <?php 
                                    if (isset($student['status'])) {
                                        $status_color = '';
                                        if (strpos($student['status'], 'إعادة') !== false) {
                                            $status_color = 'color: #ff9800; font-weight: bold;';
                                        } elseif (strpos($student['status'], 'ملاحق') !== false) {
                                            $status_color = 'color: #2196F3; font-weight: bold;';
                                        } elseif (strpos($student['status'], 'تجميد') !== false) {
                                            $status_color = 'color: #9C27B0; font-weight: bold;';
                                        }
                                        echo '<div style="' . $status_color . '">' . htmlspecialchars($student['status']) . '</div>';
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                                <td style="text-align: right; font-size: 0.85em; color: #666;">
                                    <?php 
                                    if (!empty($student['notes'])) {
                                        echo htmlspecialchars($student['notes']);
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <!-- ملاحظات -->
                <div style="margin-top: 30px; padding: 15px; background: #f9f9f9; border-radius: 5px;">
                    <h4>ملاحظات:</h4>
                    <ul>
                        <li>المعدل السنوي = (مجموع نقاط الفصل الأول + مجموع نقاط الفصل الثاني) / (إجمالي الساعات المعتمدة للفصلين)</li>
                        <li>التقدير: ممتاز (>= 4)، جيد جداً (>= 3.5)، جيد (>= 3)، مقبول (>= 2.5)، إعادة السنة (>= 2)، فصل (>= 1)</li>
                        <li><strong>قواعد القرارات السنوية:</strong></li>
                        <li style="margin-right: 20px;">• <strong>ملاحق:</strong> إذا رسب الطالب في مواد وكان معدله >= 2.5</li>
                        <li style="margin-right: 20px;">• <strong>إعادة السنة:</strong> إذا كان المعدل < 2.5</li>
                        <li style="margin-right: 20px;">• <strong>فصل:</strong> إذا كان المعدل < 1.5</li>
                        <li style="margin-right: 20px;">• <strong>ناجح:</strong> إذا لم يرسب في أي مادة</li>
                    </ul>
                </div>
            </div>
        <?php elseif ($college_id && $major_id && $academic_year): ?>
            <div class="alert alert-warning">
                لا توجد نتائج للفصلين الدراسيين المحددين.
            </div>
        <?php endif; ?>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

