<?php
/**
 * صفحة تشخيصية للتحقق من وجود الطلاب
 * Diagnostic page to check for students
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;

$course = null;
$major = null;
$all_students = [];
$active_students = [];

if ($course_id) {
    $course = $db->fetchOne(
        "SELECT c.*, m.college_id, m.id as major_id, m.name_ar as major_name
         FROM courses c 
         JOIN majors m ON c.major_id = m.id 
         WHERE c.id = ?",
        [$course_id]
    );
    
    if ($course) {
        $major = ['id' => $course['major_id'], 'name_ar' => $course['major_name']];
        
        if ($auth->hasRole('admin')) {
            $all_students = $db->fetchAll(
                "SELECT s.* FROM students s 
                 WHERE s.major_id = ?
                 ORDER BY s.status DESC, s.student_number",
                [$course['major_id']]
            );
        } else {
            $all_students = $db->fetchAll(
                "SELECT s.* FROM students s 
                 WHERE s.major_id = ? AND s.college_id = ?
                 ORDER BY s.status DESC, s.student_number",
                [$course['major_id'], $_SESSION['college_id']]
            );
        }
        
        $active_students = array_filter($all_students, function($s) {
            return $s['status'] === 'active';
        });
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>فحص الطلاب</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .diagnostic-box {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            margin: 10px 0;
            border-right: 4px solid #2196F3;
        }
        .stat-box {
            display: inline-block;
            padding: 10px 20px;
            margin: 5px;
            background: white;
            border-radius: 5px;
            border: 1px solid #ddd;
        }
        .stat-number {
            font-size: 2em;
            font-weight: bold;
            color: #2196F3;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>فحص الطلاب - صفحة تشخيصية</h2>
            <?php if ($course_id): ?>
                <a href="register_students_to_course.php?course_id=<?php echo $course_id; ?>&semester_id=<?php echo $semester_id; ?>" class="btn btn-primary">العودة لتسجيل الطلاب</a>
            <?php endif; ?>
        </div>
        
        <?php if ($course_id && $course): ?>
            <div class="card">
                <div class="card-header">
                    <h3>معلومات المادة والتخصص</h3>
                </div>
                <div class="diagnostic-box">
                    <p><strong>المادة:</strong> <?php echo htmlspecialchars($course['name_ar']); ?> (<?php echo htmlspecialchars($course['code']); ?>)</p>
                    <p><strong>التخصص:</strong> <?php echo htmlspecialchars($course['major_name']); ?> (ID: <?php echo $course['major_id']; ?>)</p>
                    <p><strong>الكلية:</strong> ID: <?php echo $course['college_id']; ?></p>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>إحصائيات الطلاب</h3>
                </div>
                <div style="text-align: center;">
                    <div class="stat-box">
                        <div class="stat-number"><?php echo count($all_students); ?></div>
                        <div>إجمالي الطلاب</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-number" style="color: green;"><?php echo count($active_students); ?></div>
                        <div>طلاب نشطين</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-number" style="color: orange;"><?php echo count($all_students) - count($active_students); ?></div>
                        <div>طلاب غير نشطين</div>
                    </div>
                </div>
            </div>
            
            <?php if (empty($all_students)): ?>
                <div class="alert alert-error">
                    <h4>⚠️ لا يوجد طلاب في هذا التخصص</h4>
                    <p><strong>التخصص:</strong> <?php echo htmlspecialchars($course['major_name']); ?></p>
                    <p><strong>الكلية:</strong> ID <?php echo $course['college_id']; ?></p>
                    <hr>
                    <p><strong>التحقق:</strong></p>
                    <ul style="margin-right: 20px;">
                        <li>تأكد من وجود طلاب في قاعدة البيانات</li>
                        <li>تأكد من أن الطلاب مرتبطين بنفس التخصص (major_id = <?php echo $course['major_id']; ?>)</li>
                        <li>إذا كنت مدير كلية، تأكد من أن الطلاب في كليتك (college_id = <?php echo $course['college_id']; ?>)</li>
                    </ul>
                    <hr>
                    <p>
                        <a href="students.php?major_id=<?php echo $course['major_id']; ?>" class="btn btn-primary">➕ إضافة طلاب</a>
                        <a href="students.php" class="btn btn-secondary">عرض جميع الطلاب</a>
                    </p>
                </div>
            <?php else: ?>
                <div class="card">
                    <div class="card-header">
                        <h3>قائمة الطلاب (<?php echo count($active_students); ?> نشط)</h3>
                    </div>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>الرقم الجامعي</th>
                                    <th>الاسم</th>
                                    <th>الحالة</th>
                                    <th>التخصص</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($all_students as $idx => $student): ?>
                                    <tr style="<?php echo $student['status'] !== 'active' ? 'opacity: 0.6; background-color: #fff3cd;' : ''; ?>">
                                        <td><?php echo $idx + 1; ?></td>
                                        <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                        <td>
                                            <?php if ($student['status'] === 'active'): ?>
                                                <span style="color: green; font-weight: bold;">✓ نشط</span>
                                            <?php else: ?>
                                                <span style="color: orange;">⚠ <?php echo htmlspecialchars($student['status']); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>ID: <?php echo $student['major_id']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="alert alert-info">
                <h4>استخدام هذه الصفحة</h4>
                <p>هذه صفحة تشخيصية للتحقق من وجود الطلاب في تخصص معين.</p>
                <p>يمكنك الوصول إليها من صفحة تسجيل الطلاب عند عدم ظهور الطلاب.</p>
                <p><strong>الاستخدام:</strong> أضف <code>?course_id=X&semester_id=Y</code> إلى رابط الصفحة</p>
            </div>
        <?php endif; ?>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

