<?php
/**
 * نظام إصدار الشهادات
 * Certificate Generation System
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/GradeCalculator.php';

class CertificateGenerator {
    private $db;
    private $gradeCalculator;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->gradeCalculator = new GradeCalculator();
    }
    
    /**
     * التحقق من سداد الرسوم
     */
    public function checkFeesPaid($student_id, $semester_id = null) {
        try {
            // التحقق من وجود جدول الرسوم أولاً
            // إذا لم يكن الجدول موجوداً أو لم تكن هناك رسوم، نعتبرها مسددة
            if ($semester_id) {
                // التحقق من رسوم فصل دراسي محدد
                $fees = $this->db->fetchAll(
                    "SELECT * FROM fees 
                     WHERE student_id = ? AND semester_id = ? AND payment_status != 'paid'",
                    [$student_id, $semester_id]
                );
            } else {
                // التحقق من جميع الرسوم غير المسددة
                $fees = $this->db->fetchAll(
                    "SELECT * FROM fees 
                     WHERE student_id = ? AND payment_status != 'paid'",
                    [$student_id]
                );
            }
            
            // إذا لم تكن هناك رسوم غير مسددة، نعتبر الرسوم مسددة
            return empty($fees);
        } catch (Exception $e) {
            // في حالة وجود خطأ (مثل عدم وجود جدول fees)، نعتبر الرسوم مسددة للسماح بإنشاء الطلب
            error_log("CertificateGenerator::checkFeesPaid Error: " . $e->getMessage());
            return true; // نعتبرها مسددة للسماح بالعمل
        }
    }
    
    /**
     * إنشاء طلب شهادة
     */
    public function createRequest($student_id, $certificate_type) {
        try {
            // التحقق من صحة البيانات
            if (empty($student_id) || empty($certificate_type)) {
                return [
                    'success' => false,
                    'message' => 'بيانات الطلب غير صحيحة'
                ];
            }
            
            // التحقق من صحة نوع الشهادة
            $valid_types = ['detailed', 'university', 'transcript', 'enrollment', 'graduation'];
            if (!in_array($certificate_type, $valid_types)) {
                return [
                    'success' => false,
                    'message' => 'نوع الشهادة غير صحيح'
                ];
            }
            
            // التحقق من وجود الطالب
            $student = $this->db->fetchOne(
                "SELECT * FROM students WHERE id = ?",
                [$student_id]
            );
            
            if (!$student) {
                return [
                    'success' => false,
                    'message' => 'الطالب غير موجود'
                ];
            }
            
            // التحقق من سداد الرسوم
            if (!$this->checkFeesPaid($student_id)) {
                return [
                    'success' => false,
                    'message' => 'يجب سداد جميع الرسوم الدراسية قبل التقديم على الشهادة'
                ];
            }
            
            // التحقق من وجود طلب معلق
            $existing = $this->db->fetchOne(
                "SELECT * FROM certificate_requests 
                 WHERE student_id = ? AND certificate_type = ? AND status = 'pending'",
                [$student_id, $certificate_type]
            );
            
            if ($existing) {
                return [
                    'success' => false,
                    'message' => 'يوجد طلب معلق لهذه الشهادة'
                ];
            }
            
            // إنشاء الطلب
            $this->db->query(
                "INSERT INTO certificate_requests (student_id, certificate_type, request_date, status)
                 VALUES (?, ?, CURDATE(), 'pending')",
                [$student_id, $certificate_type]
            );
            
            return [
                'success' => true,
                'message' => 'تم تقديم الطلب بنجاح'
            ];
        } catch (Exception $e) {
            error_log("CertificateGenerator::createRequest Error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء إنشاء الطلب: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * الموافقة على طلب شهادة
     */
    public function approveRequest($request_id, $issued_by) {
        $request = $this->db->fetchOne(
            "SELECT * FROM certificate_requests WHERE id = ?",
            [$request_id]
        );
        
        if (!$request) {
            return ['success' => false, 'message' => 'الطلب غير موجود'];
        }
        
        // التحقق من سداد الرسوم مرة أخرى
        if (!$this->checkFeesPaid($request['student_id'])) {
            return [
                'success' => false,
                'message' => 'الطالب لم يسدد جميع الرسوم'
            ];
        }
        
        // إنشاء رقم الشهادة
        $certificate_number = $this->generateCertificateNumber($request['certificate_type'], $request['student_id']);
        
        // تحديث حالة الطلب
        $this->db->query(
            "UPDATE certificate_requests 
             SET status = 'approved', issued_date = CURDATE(), issued_by = ?, certificate_number = ?
             WHERE id = ?",
            [$issued_by, $certificate_number, $request_id]
        );
        
        // إنشاء الشهادة
        $certificate = $this->generateCertificate($request_id, $certificate_number, $issued_by);
        
        return [
            'success' => true,
            'message' => 'تم إصدار الشهادة بنجاح',
            'certificate' => $certificate
        ];
    }
    
    /**
     * إنشاء رقم الشهادة
     */
    private function generateCertificateNumber($type, $student_id) {
        $prefix = [
            'detailed' => 'DET',
            'university' => 'UNI',
            'transcript' => 'TRN',
            'enrollment' => 'ENR',
            'graduation' => 'GRD'
        ];
        
        $year = date('Y');
        $number = str_pad($student_id, 6, '0', STR_PAD_LEFT);
        
        return $prefix[$type] . '-' . $year . '-' . $number;
    }
    
    /**
     * إنشاء الشهادة (PDF)
     */
    public function generateCertificate($request_id, $certificate_number, $issued_by) {
        $request = $this->db->fetchOne(
            "SELECT cr.*, s.*, c.name_ar as college_name, m.name_ar as major_name
             FROM certificate_requests cr
             JOIN students s ON cr.student_id = s.id
             JOIN colleges c ON s.college_id = c.id
             JOIN majors m ON s.major_id = m.id
             WHERE cr.id = ?",
            [$request_id]
        );
        
        if (!$request) {
            return null;
        }
        
        // الحصول على المعدل التراكمي
        $cgpa = $this->db->fetchOne(
            "SELECT * FROM cumulative_gpas WHERE student_id = ?",
            [$request['student_id']]
        );
        
        // حفظ معلومات الشهادة في قاعدة البيانات
        $this->db->query(
            "INSERT INTO certificates (certificate_request_id, student_id, certificate_type, 
             certificate_number, issue_date, issued_by)
             VALUES (?, ?, ?, ?, CURDATE(), ?)",
            [
                $request_id,
                $request['student_id'],
                $request['certificate_type'],
                $certificate_number,
                $issued_by
            ]
        );
        
        // تحديث حالة الطلب
        $this->db->query(
            "UPDATE certificate_requests SET status = 'issued' WHERE id = ?",
            [$request_id]
        );
        
        // هنا يمكن إضافة كود لإنشاء PDF باستخدام مكتبة مثل TCPDF أو FPDF
        // $pdf_path = $this->createPDF($request, $cgpa, $certificate_number);
        
        return [
            'certificate_number' => $certificate_number,
            'issue_date' => date('Y-m-d'),
            'student_name' => $request['full_name_ar']
        ];
    }
    
    /**
     * الحصول على بيانات الشهادة
     */
    public function getCertificateData($certificate_id) {
        return $this->db->fetchOne(
            "SELECT c.*, s.full_name_ar, s.student_number, col.name_ar as college_name,
             m.name_ar as major_name, u.full_name_ar as issued_by_name
             FROM certificates c
             JOIN students s ON c.student_id = s.id
             JOIN colleges col ON s.college_id = col.id
             JOIN majors m ON s.major_id = m.id
             JOIN users u ON c.issued_by = u.id
             WHERE c.id = ?",
            [$certificate_id]
        );
    }
}

