<?php
/**
 * كلاس أرشيف النتائج
 * Results Archive Class
 * لحفظ واسترجاع نسخ من النتائج
 */

class ResultsArchive {
    
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
        
        // إنشاء الجدول إذا لم يكن موجوداً
        $this->createTableIfNotExists();
    }
    
    /**
     * إنشاء جدول أرشيف النتائج إذا لم يكن موجوداً
     */
    private function createTableIfNotExists() {
        try {
            $table_exists = $this->db->fetchOne("SHOW TABLES LIKE 'results_archive'");
            if (!$table_exists) {
                $this->db->query("
                    CREATE TABLE IF NOT EXISTS results_archive (
                        id INT PRIMARY KEY AUTO_INCREMENT,
                        semester_id INT NOT NULL,
                        college_id INT NULL,
                        major_id INT NULL,
                        result_type ENUM('first_term', 'supplementary', 'removal_exam') NOT NULL COMMENT 'نوع النتيجة: الدور الأول، الملاحق، إزالة الرسوب',
                        result_data TEXT NOT NULL COMMENT 'بيانات النتائج بصيغة JSON',
                        summary_data TEXT NULL COMMENT 'ملخص الإحصائيات بصيغة JSON',
                        saved_by INT NOT NULL COMMENT 'المستخدم الذي حفظ النتائج',
                        saved_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ ووقت الحفظ',
                        notes TEXT NULL COMMENT 'ملاحظات إضافية',
                        FOREIGN KEY (semester_id) REFERENCES semesters(id) ON DELETE CASCADE,
                        FOREIGN KEY (college_id) REFERENCES colleges(id) ON DELETE SET NULL,
                        FOREIGN KEY (major_id) REFERENCES majors(id) ON DELETE SET NULL,
                        FOREIGN KEY (saved_by) REFERENCES users(id) ON DELETE RESTRICT,
                        INDEX idx_semester_type (semester_id, result_type),
                        INDEX idx_saved_at (saved_at)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='أرشيف النتائج'
                ");
            }
        } catch (Exception $e) {
            error_log("Results Archive Table Creation Error: " . $e->getMessage());
        }
    }
    
    /**
     * حفظ نتيجة الملاحق
     * @param int $semester_id معرف الفصل الدراسي
     * @param int $user_id معرف المستخدم
     * @param array $report_data بيانات التقرير
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @param string|null $notes ملاحظات
     * @return int معرف السجل المحفوظ
     */
    public function saveSupplementaryResults($semester_id, $user_id, $report_data, $college_id = null, $major_id = null, $notes = null) {
        return $this->saveResults('supplementary', $semester_id, $user_id, $report_data, $college_id, $major_id, $notes);
    }
    
    /**
     * حفظ نتيجة إزالة الرسوب
     * @param int $semester_id معرف الفصل الدراسي
     * @param int $user_id معرف المستخدم
     * @param array $report_data بيانات التقرير
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @param string|null $notes ملاحظات
     * @return int معرف السجل المحفوظ
     */
    public function saveRemovalExamResults($semester_id, $user_id, $report_data, $college_id = null, $major_id = null, $notes = null) {
        return $this->saveResults('removal_exam', $semester_id, $user_id, $report_data, $college_id, $major_id, $notes);
    }
    
    /**
     * حفظ نتيجة الدور الأول
     * @param int $semester_id معرف الفصل الدراسي
     * @param int $user_id معرف المستخدم
     * @param array $report_data بيانات التقرير
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @param string|null $notes ملاحظات
     * @return int معرف السجل المحفوظ
     */
    public function saveFirstTermResults($semester_id, $user_id, $report_data, $college_id = null, $major_id = null, $notes = null) {
        return $this->saveResults('first_term', $semester_id, $user_id, $report_data, $college_id, $major_id, $notes);
    }
    
    /**
     * حفظ النتائج السنوية
     * @param int $semester_id معرف الفصل الدراسي الأول (يُستخدم كمرجع)
     * @param int $user_id معرف المستخدم
     * @param array $report_data بيانات التقرير
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @param string|null $notes ملاحظات
     * @return int معرف السجل المحفوظ
     */
    public function saveAnnualResults($semester_id, $user_id, $report_data, $college_id = null, $major_id = null, $notes = null) {
        // إضافة ملاحظة أن هذه نتائج سنوية
        $annual_note = 'نتائج سنوية (فصلين دراسيين)';
        if ($notes) {
            $notes = $annual_note . ' - ' . $notes;
        } else {
            $notes = $annual_note;
        }
        // استخدام first_term كـ result_type مع ملاحظة أنها نتائج سنوية
        return $this->saveResults('first_term', $semester_id, $user_id, $report_data, $college_id, $major_id, $notes);
    }
    
    /**
     * حفظ النتائج (دالة عامة)
     * @param string $result_type نوع النتيجة
     * @param int $semester_id معرف الفصل الدراسي
     * @param int $user_id معرف المستخدم
     * @param array $report_data بيانات التقرير
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @param string|null $notes ملاحظات
     * @return int معرف السجل المحفوظ
     */
    private function saveResults($result_type, $semester_id, $user_id, $report_data, $college_id = null, $major_id = null, $notes = null) {
        $result_json = json_encode($report_data, JSON_UNESCAPED_UNICODE);
        $summary_json = isset($report_data['summary']) ? json_encode($report_data['summary'], JSON_UNESCAPED_UNICODE) : null;
        
        $archive_id = $this->db->query(
            "INSERT INTO results_archive (semester_id, college_id, major_id, result_type, result_data, summary_data, saved_by, notes)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [$semester_id, $college_id, $major_id, $result_type, $result_json, $summary_json, $user_id, $notes]
        );
        
        return $archive_id;
    }
    
    /**
     * جلب النسخ المحفوظة حسب النوع والفصل
     * @param string $result_type نوع النتيجة
     * @param int $semester_id معرف الفصل الدراسي
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $major_id معرف التخصص (اختياري)
     * @return array قائمة النسخ المحفوظة
     */
    public function getArchivedResults($result_type, $semester_id, $college_id = null, $major_id = null) {
        $query = "SELECT ra.*, 
                         s.semester_number, s.academic_year,
                         c.name_ar as college_name,
                         m.name_ar as major_name,
                         u.full_name_ar as saved_by_name
                  FROM results_archive ra
                  JOIN semesters s ON ra.semester_id = s.id
                  LEFT JOIN colleges c ON ra.college_id = c.id
                  LEFT JOIN majors m ON ra.major_id = m.id
                  JOIN users u ON ra.saved_by = u.id
                  WHERE ra.result_type = ? AND ra.semester_id = ?";
        
        $params = [$result_type, $semester_id];
        
        if ($college_id !== null) {
            $query .= " AND (ra.college_id = ? OR ra.college_id IS NULL)";
            $params[] = $college_id;
        }
        
        if ($major_id !== null) {
            $query .= " AND (ra.major_id = ? OR ra.major_id IS NULL)";
            $params[] = $major_id;
        }
        
        $query .= " ORDER BY ra.saved_at DESC";
        
        $archives = $this->db->fetchAll($query, $params);
        
        // فك تشفير بيانات JSON
        foreach ($archives as &$archive) {
            $archive['result_data_decoded'] = json_decode($archive['result_data'], true);
            $archive['summary_data_decoded'] = $archive['summary_data'] ? json_decode($archive['summary_data'], true) : null;
        }
        
        return $archives;
    }
    
    /**
     * جلب نسخة محددة
     * @param int $archive_id معرف السجل
     * @return array|null بيانات النسخة
     */
    public function getArchiveById($archive_id) {
        $archive = $this->db->fetchOne(
            "SELECT ra.*, 
                    s.semester_number, s.academic_year,
                    c.name_ar as college_name,
                    m.name_ar as major_name,
                    u.full_name_ar as saved_by_name
             FROM results_archive ra
             JOIN semesters s ON ra.semester_id = s.id
             LEFT JOIN colleges c ON ra.college_id = c.id
             LEFT JOIN majors m ON ra.major_id = m.id
             JOIN users u ON ra.saved_by = u.id
             WHERE ra.id = ?",
            [$archive_id]
        );
        
        if ($archive) {
            $archive['result_data_decoded'] = json_decode($archive['result_data'], true);
            $archive['summary_data_decoded'] = $archive['summary_data'] ? json_decode($archive['summary_data'], true) : null;
        }
        
        return $archive;
    }
    
    /**
     * حذف نسخة محفوظة
     * @param int $archive_id معرف السجل
     * @return bool نجح الحذف أم لا
     */
    public function deleteArchive($archive_id) {
        try {
            $this->db->query("DELETE FROM results_archive WHERE id = ?", [$archive_id]);
            return true;
        } catch (Exception $e) {
            error_log("Delete Archive Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * جلب جميع النسخ المحفوظة (للإدارة)
     * @param int|null $college_id معرف الكلية (اختياري)
     * @param int|null $result_type نوع النتيجة (اختياري)
     * @return array قائمة النسخ
     */
    public function getAllArchives($college_id = null, $result_type = null) {
        $query = "SELECT ra.*, 
                         s.semester_number, s.academic_year,
                         c.name_ar as college_name,
                         m.name_ar as major_name,
                         u.full_name_ar as saved_by_name
                  FROM results_archive ra
                  JOIN semesters s ON ra.semester_id = s.id
                  LEFT JOIN colleges c ON ra.college_id = c.id
                  LEFT JOIN majors m ON ra.major_id = m.id
                  JOIN users u ON ra.saved_by = u.id
                  WHERE 1=1";
        
        $params = [];
        
        if ($college_id !== null) {
            $query .= " AND (ra.college_id = ? OR ra.college_id IS NULL)";
            $params[] = $college_id;
        }
        
        if ($result_type !== null) {
            $query .= " AND ra.result_type = ?";
            $params[] = $result_type;
        }
        
        $query .= " ORDER BY ra.saved_at DESC";
        
        $archives = $this->db->fetchAll($query, $params);
        
        // فك تشفير بيانات JSON
        foreach ($archives as &$archive) {
            $archive['summary_data_decoded'] = $archive['summary_data'] ? json_decode($archive['summary_data'], true) : null;
        }
        
        return $archives;
    }
    
    /**
     * الحصول على اسم نوع النتيجة بالعربية
     * @param string $result_type نوع النتيجة
     * @return string الاسم بالعربية
     */
    public function getResultTypeName($result_type) {
        $names = [
            'first_term' => 'نتيجة الدور الأول',
            'supplementary' => 'نتيجة الملاحق',
            'removal_exam' => 'نتيجة إزالة الرسوب'
        ];
        
        return $names[$result_type] ?? $result_type;
    }
    
    /**
     * التحقق من أن النتيجة سنوية بناءً على الملاحظات
     * @param string|null $notes الملاحظات
     * @return bool
     */
    public function isAnnualResult($notes) {
        return $notes && strpos($notes, 'نتائج سنوية') !== false;
    }
}

