<?php
/**
 * إعدادات قاعدة البيانات
 * Database Configuration
 */

// إعدادات الاتصال بقاعدة البيانات
// Database Connection Settings
//
// ملاحظة: في AppServ، كلمة مرور root الافتراضية عادة تكون:
// - فارغة (فقط إذا تم تثبيت MySQL بدون كلمة مرور)
// - "root" (في معظم حالات AppServ)
// - الكلمة التي قمت بتعيينها أثناء تثبيت AppServ
//
// Note: In AppServ, the default MySQL root password is usually:
// - Empty (only if MySQL was installed without password)
// - "root" (in most AppServ installations)
// - The password you set during AppServ installation

define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', ''); // قم بتغيير هذه القيمة إلى كلمة مرور MySQL الخاصة بك
define('DB_NAME', 'university_results');
define('DB_CHARSET', 'utf8mb4');

class Database {
    private static $instance = null;
    private $connection;

    private function __construct() {
        try {
            // أولاً: محاولة الاتصال بدون قاعدة البيانات للتحقق من كلمة المرور
            $dsnTest = "mysql:host=" . DB_HOST . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::ATTR_TIMEOUT => 5,
            ];

            $testConnection = new PDO($dsnTest, DB_USER, DB_PASS, $options);

            // ثانياً: محاولة الاتصال بقاعدة البيانات
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);

        } catch (PDOException $e) {
            $errorCode = $e->getCode();
            $errorMsg = "خطأ في الاتصال بقاعدة البيانات: " . $e->getMessage();
            $errorMsg .= "\n\n";

            // تحليل نوع الخطأ
            if ($errorCode == 1045 || strpos($e->getMessage(), 'Access denied') !== false) {
                $errorMsg .= "⚠ خطأ في المصادقة (كلمة المرور أو اسم المستخدم غير صحيح)\n\n";
                $errorMsg .= "الحلول:\n";
                $errorMsg .= "1. شغّل find_mysql_password.php للعثور على كلمة المرور الصحيحة\n";
                $errorMsg .= "2. أو تحقق من كلمة المرور من AppServ Control Panel\n";
                $errorMsg .= "3. أو جرب في config/database.php:\n";
                $errorMsg .= "   - define('DB_PASS', 'root');\n";
                $errorMsg .= "   - define('DB_PASS', ''); (فارغة)\n";
            } elseif ($errorCode == 1049 || strpos($e->getMessage(), "Unknown database") !== false) {
                $errorMsg .= "⚠ قاعدة البيانات '" . DB_NAME . "' غير موجودة\n\n";
                $errorMsg .= "الحل:\n";
                $errorMsg .= "1. افتح phpMyAdmin: http://localhost/phpmyadmin\n";
                $errorMsg .= "2. استورد ملف database/schema.sql\n";
                $errorMsg .= "أو من سطر الأوامر:\n";
                $errorMsg .= "   mysql -u root -p < database/schema.sql\n";
            } elseif ($errorCode == 2002 || strpos($e->getMessage(), "Connection refused") !== false) {
                $errorMsg .= "⚠ MySQL غير قيد التشغيل\n\n";
                $errorMsg .= "الحل:\n";
                $errorMsg .= "1. افتح AppServ Control Panel\n";
                $errorMsg .= "2. تأكد من تشغيل MySQL\n";
            } else {
                $errorMsg .= "يرجى التحقق من:\n";
                $errorMsg .= "1. أن MySQL يعمل (تحقق من AppServ Control Panel)\n";
                $errorMsg .= "2. أن كلمة مرور MySQL في ملف config/database.php صحيحة\n";
                $errorMsg .= "3. أن قاعدة البيانات '" . DB_NAME . "' موجودة\n";
            }

            $errorMsg .= "\nفي AppServ، كلمة مرور root الافتراضية عادة تكون 'root' أو فارغة\n";
            $errorMsg .= "يمكنك التحقق من كلمة المرور من خلال phpMyAdmin أو AppServ Control Panel";
            die($errorMsg);
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->connection;
    }

    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Database Error: " . $e->getMessage());
            throw $e;
        }
    }

    public function fetchAll($sql, $params = []) {
        return $this->query($sql, $params)->fetchAll();
    }

    public function fetchOne($sql, $params = []) {
        return $this->query($sql, $params)->fetch();
    }
}

