<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();

// فحص وإصلاح AUTO_INCREMENT إذا لزم الأمر
try {
    // التحقق من وجود سجل ب id=0 وحذفه
    $zero_record = $db->fetchOne("SELECT id FROM courses WHERE id = 0");
    if ($zero_record) {
        $db->query("DELETE FROM courses WHERE id = 0");
    }
    
    // التحقق من أن حقل id لديه خاصية AUTO_INCREMENT
    $columns = $db->fetchAll("SHOW COLUMNS FROM courses WHERE Field = 'id'");
    $has_auto_increment = false;
    if (!empty($columns)) {
        $has_auto_increment = (strpos($columns[0]['Extra'], 'auto_increment') !== false);
    }
    
    // إذا لم يكن لديه AUTO_INCREMENT، قم بإضافته
    if (!$has_auto_increment) {
        // جلب أعلى ID موجود
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM courses WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        $new_auto_increment = max($max_id + 1, 1);
        
        // إضافة AUTO_INCREMENT للحقل
        $db->getConnection()->exec("ALTER TABLE courses MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
        $db->getConnection()->exec("ALTER TABLE courses AUTO_INCREMENT = " . (int)$new_auto_increment);
    } else {
        // التحقق من أن AUTO_INCREMENT مضبوط بشكل صحيح
        $status = $db->fetchOne("SHOW TABLE STATUS LIKE 'courses'");
        $auto_increment = $status['Auto_increment'] ?? null;
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM courses WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        
        // إذا كان AUTO_INCREMENT أقل من أو يساوي أعلى ID موجود، قم بتحديثه
        if ($auto_increment !== null && $auto_increment <= $max_id) {
            $new_auto_increment = max($max_id + 1, 1);
            $db->getConnection()->exec("ALTER TABLE courses AUTO_INCREMENT = " . (int)$new_auto_increment);
        }
    }
} catch (PDOException $e) {
    // تجاهل الأخطاء في الفحص الاستباقي
    error_log("Auto-increment check error: " . $e->getMessage());
}

// معالجة AJAX requests
if (isset($_GET['action']) && $_GET['action'] == 'get_majors' && isset($_GET['college_id'])) {
    header('Content-Type: application/json');
    $college_id = (int)$_GET['college_id'];
    $majors = $db->fetchAll(
        "SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
    echo json_encode($majors);
    exit;
}

if (isset($_GET['action']) && $_GET['action'] == 'get_semesters' && isset($_GET['major_id'])) {
    header('Content-Type: application/json');
    $major_id = (int)$_GET['major_id'];
    $semesters = $db->fetchAll(
        "SELECT id, semester_number, academic_year FROM semesters WHERE major_id = ? ORDER BY academic_year, semester_number",
        [$major_id]
    );
    echo json_encode($semesters);
    exit;
}

$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$filter_college_id = isset($_GET['filter_college_id']) ? (int)$_GET['filter_college_id'] : 0;
$filter_major_id = isset($_GET['filter_major_id']) ? (int)$_GET['filter_major_id'] : 0;
$filter_semester_id = isset($_GET['filter_semester_id']) ? (int)$_GET['filter_semester_id'] : 0;
$filter_academic_year = isset($_GET['filter_academic_year']) ? $_GET['filter_academic_year'] : '';
$message = '';

if ($semester_id) {
    $semester = $db->fetchOne(
        "SELECT s.*, m.id as major_id, m.college_id FROM semesters s JOIN majors m ON s.major_id = m.id WHERE s.id = ?",
        [$semester_id]
    );
    if ($semester && (!$auth->belongsToCollege($semester['college_id']) && !$auth->hasRole('admin'))) {
        header('Location: semesters.php');
        exit;
    }
    $major_id = $semester['major_id'];
} elseif ($major_id) {
    $major = $db->fetchOne("SELECT * FROM majors WHERE id = ?", [$major_id]);
    if ($major && (!$auth->belongsToCollege($major['college_id']) && !$auth->hasRole('admin'))) {
        header('Location: majors.php');
        exit;
    }
} else {
    header('Location: majors.php');
    exit;
}

// جلب معلومات الكلية والتخصص للعرض
$major_info = $db->fetchOne(
    "SELECT m.*, c.name_ar as college_name, c.id as college_id 
     FROM majors m 
     JOIN colleges c ON m.college_id = c.id 
     WHERE m.id = ?",
    [$major_id]
);

// معالجة الإضافة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $code = $_POST['code'] ?? '';
    $name_ar = $_POST['name_ar'] ?? '';
    $name_en = $_POST['name_en'] ?? '';
    $credit_hours = (int)($_POST['credit_hours'] ?? 0);
    $is_required = isset($_POST['is_required']) ? 1 : 0;
    $semester_id_post = (int)($_POST['semester_id'] ?? 0);
    
    if ($code && $name_ar && $credit_hours) {
        try {
            // التحقق من الصلاحيات
            if (!$auth->belongsToCollege($major_info['college_id']) && !$auth->hasRole('admin')) {
                $message = '<div class="alert alert-error">ليس لديك صلاحية لإضافة مادة في هذه الكلية</div>';
            } else {
                // التأكد من عدم وجود سجل ب id=0 قبل الإدراج
                $zero_record = $db->fetchOne("SELECT id FROM courses WHERE id = 0");
                if ($zero_record) {
                    $db->query("DELETE FROM courses WHERE id = 0");
                }
                
                // التأكد من أن AUTO_INCREMENT يعمل بشكل صحيح
                $columns = $db->fetchAll("SHOW COLUMNS FROM courses WHERE Field = 'id'");
                $has_auto_increment = false;
                if (!empty($columns)) {
                    $has_auto_increment = (strpos($columns[0]['Extra'], 'auto_increment') !== false);
                }
                
                if (!$has_auto_increment) {
                    // إضافة AUTO_INCREMENT إذا لم يكن موجوداً
                    $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM courses WHERE id > 0");
                    $max_id = $max_id_result['max_id'] ?? 0;
                    $new_auto_increment = max($max_id + 1, 1);
                    $db->getConnection()->exec("ALTER TABLE courses MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
                    $db->getConnection()->exec("ALTER TABLE courses AUTO_INCREMENT = " . (int)$new_auto_increment);
                }
                
                // إدراج المادة (بدون تحديد id ليتم توليده تلقائياً)
                $db->query(
                    "INSERT INTO courses (major_id, semester_id, code, name_ar, name_en, credit_hours, is_required) 
                     VALUES (?, ?, ?, ?, ?, ?, ?)",
                    [$major_id, $semester_id_post ?: null, $code, $name_ar, $name_en, $credit_hours, $is_required]
                );
                
                // الحصول على ID المولّد تلقائياً
                $new_id = $db->getConnection()->lastInsertId();
                
                // إذا كان ID = 0 أو false، فهناك مشكلة، يجب إصلاحها
                if ($new_id == 0 || $new_id === false || $new_id === '0') {
                    // حذف أي سجل ب id=0
                    $db->query("DELETE FROM courses WHERE id = 0");
                    
                    // جلب أعلى ID موجود
                    $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM courses WHERE id > 0");
                    $max_id = $max_id_result['max_id'] ?? 0;
                    $new_auto_increment = max($max_id + 1, 1);
                    
                    // إصلاح AUTO_INCREMENT
                    $db->getConnection()->exec("ALTER TABLE courses MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
                    $db->getConnection()->exec("ALTER TABLE courses AUTO_INCREMENT = " . (int)$new_auto_increment);
                    
                    // إعادة إدراج السجل
                    $db->query(
                        "INSERT INTO courses (major_id, semester_id, code, name_ar, name_en, credit_hours, is_required) 
                         VALUES (?, ?, ?, ?, ?, ?, ?)",
                        [$major_id, $semester_id_post ?: null, $code, $name_ar, $name_en, $credit_hours, $is_required]
                    );
                    $new_id = $db->getConnection()->lastInsertId();
                    
                    // التحقق مرة أخرى
                    if ($new_id == 0 || $new_id === false || $new_id === '0') {
                        throw new Exception("فشل في توليد ID تلقائياً. يرجى التحقق من إعدادات قاعدة البيانات.");
                    }
                }
                
                $message = '<div class="alert alert-success">تم إضافة المادة بنجاح</div>';
            }
        } catch (PDOException $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// معالجة التعديل
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit') {
    $course_id = (int)($_POST['course_id'] ?? 0);
    $code = $_POST['code'] ?? '';
    $name_ar = $_POST['name_ar'] ?? '';
    $name_en = $_POST['name_en'] ?? '';
    $credit_hours = (int)($_POST['credit_hours'] ?? 0);
    $is_required = isset($_POST['is_required']) ? 1 : 0;
    $semester_id_post = (int)($_POST['semester_id'] ?? 0);
    
    if ($course_id > 0 && $code && $name_ar && $credit_hours) {
        try {
            // جلب معلومات المادة مع الكلية والتخصص والفصل
            $course = $db->fetchOne(
                "SELECT c.*, m.college_id, m.name_ar as major_name, s.academic_year, s.semester_number
                 FROM courses c 
                 JOIN majors m ON c.major_id = m.id 
                 LEFT JOIN semesters s ON c.semester_id = s.id
                 WHERE c.id = ?",
                [$course_id]
            );
            
            if (!$course) {
                $message = '<div class="alert alert-error">المادة غير موجودة</div>';
            } elseif (!$auth->belongsToCollege($course['college_id']) && !$auth->hasRole('admin')) {
                $message = '<div class="alert alert-error">ليس لديك صلاحية لتعديل هذه المادة. المادة تابعة لكلية: ' . htmlspecialchars($course['college_id']) . '</div>';
            } else {
                $db->query(
                    "UPDATE courses SET code = ?, name_ar = ?, name_en = ?, credit_hours = ?, is_required = ?, semester_id = ? 
                     WHERE id = ?",
                    [$code, $name_ar, $name_en, $credit_hours, $is_required, $semester_id_post ?: null, $course_id]
                );
                $message = '<div class="alert alert-success">تم تعديل المادة بنجاح</div>';
            }
        } catch (PDOException $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// معالجة الحذف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $course_id = (int)($_POST['course_id'] ?? 0);
    
    if ($course_id > 0) {
        try {
            // جلب معلومات المادة مع الكلية والتخصص والفصل
            $course = $db->fetchOne(
                "SELECT c.*, m.college_id, m.name_ar as major_name, s.academic_year, s.semester_number
                 FROM courses c 
                 JOIN majors m ON c.major_id = m.id 
                 LEFT JOIN semesters s ON c.semester_id = s.id
                 WHERE c.id = ?",
                [$course_id]
            );
            
            if (!$course) {
                $message = '<div class="alert alert-error">المادة غير موجودة</div>';
            } elseif (!$auth->belongsToCollege($course['college_id']) && !$auth->hasRole('admin')) {
                $message = '<div class="alert alert-error">ليس لديك صلاحية لحذف هذه المادة. المادة تابعة لكلية: ' . htmlspecialchars($course['college_id']) . '</div>';
            } else {
                // التحقق من وجود تسجيلات طلاب أو درجات مرتبطة بالمادة
                $student_courses = $db->fetchOne(
                    "SELECT COUNT(*) as count FROM student_courses WHERE course_id = ?",
                    [$course_id]
                );
                
                $registrations_count = $student_courses ? (int)$student_courses['count'] : 0;
                
                // بدء معاملة قاعدة البيانات
                $db->getConnection()->beginTransaction();
                
                try {
                    // حذف جميع الدرجات المرتبطة بالتسجيلات
                    $db->query(
                        "DELETE g FROM grades g 
                         INNER JOIN student_courses sc ON g.student_course_id = sc.id 
                         WHERE sc.course_id = ?",
                        [$course_id]
                    );
                    
                    // حذف جميع التسجيلات المرتبطة بالمادة
                    if ($registrations_count > 0) {
                        $db->query("DELETE FROM student_courses WHERE course_id = ?", [$course_id]);
                    }
                    
                    // حذف المادة نفسها
                    $db->query("DELETE FROM courses WHERE id = ?", [$course_id]);
                    
                    // تأكيد المعاملة
                    $db->getConnection()->commit();
                    
                    // بناء رسالة النجاح
                    $success_msg = 'تم حذف المادة بنجاح';
                    if ($registrations_count > 0) {
                        $success_msg .= ' (تم حذف ' . $registrations_count . ' تسجيل طالب مرتبط)';
                    }
                    
                    // بناء URL لإعادة التوجيه مع الحفاظ على الفلاتر
                    $redirect_url = 'courses.php';
                    $params = [];
                    if ($major_id) $params[] = 'major_id=' . $major_id;
                    if ($semester_id) $params[] = 'semester_id=' . $semester_id;
                    if ($filter_college_id) $params[] = 'filter_college_id=' . $filter_college_id;
                    if ($filter_major_id) $params[] = 'filter_major_id=' . $filter_major_id;
                    if ($filter_semester_id) $params[] = 'filter_semester_id=' . $filter_semester_id;
                    if ($filter_academic_year) $params[] = 'filter_academic_year=' . urlencode($filter_academic_year);
                    if (!empty($params)) {
                        $redirect_url .= '?' . implode('&', $params);
                    }
                    $redirect_url .= '&msg=' . urlencode($success_msg);
                    $redirect_url .= '&msg_type=success';
                    
                    header('Location: ' . $redirect_url);
                    exit;
                    
                } catch (Exception $e) {
                    // إلغاء المعاملة في حالة الخطأ
                    $db->getConnection()->rollBack();
                    throw $e;
                }
            }
        } catch (PDOException $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// معالجة حذف جميع المواد (فقط للـ admin)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_all') {
    // التحقق من الصلاحيات - فقط admin يمكنه حذف جميع المواد
    if (!$auth->hasRole('admin')) {
        $message = '<div class="alert alert-error">⚠️ خطأ: ليس لديك صلاحية لحذف جميع المواد. هذه العملية متاحة فقط للمدير العام.</div>';
    } else {
        // التحقق من تأكيد الحذف
        $confirm_text = trim($_POST['confirm_text'] ?? '');
        $expected_text = 'حذف جميع المواد';
        
        if ($confirm_text !== $expected_text) {
            $message = '<div class="alert alert-error">⚠️ خطأ: لم يتم تأكيد العملية. يجب كتابة "حذف جميع المواد" للتأكيد.</div>';
        } else {
            try {
                // جلب عدد المواد قبل الحذف
                $total_courses = $db->fetchOne("SELECT COUNT(*) as count FROM courses")['count'];
                
                // جلب عدد التسجيلات والدرجات المرتبطة
                $total_registrations = $db->fetchOne("SELECT COUNT(*) as count FROM student_courses")['count'];
                $total_grades = $db->fetchOne("SELECT COUNT(*) as count FROM grades")['count'];
                
                if ($total_courses == 0) {
                    $message = '<div class="alert alert-info">لا توجد مواد لحذفها</div>';
                } else {
                    // بدء معاملة قاعدة البيانات
                    $db->getConnection()->beginTransaction();
                    
                    try {
                        // حذف جميع الدرجات المرتبطة بالتسجيلات
                        if ($total_grades > 0) {
                            $db->query("DELETE FROM grades");
                        }
                        
                        // حذف جميع التسجيلات المرتبطة بالمواد
                        if ($total_registrations > 0) {
                            $db->query("DELETE FROM student_courses");
                        }
                        
                        // حذف جميع المواد
                        $db->query("DELETE FROM courses");
                        
                        // تسجيل نشاط حذف جميع المواد
                        require_once __DIR__ . '/classes/ActivityLogger.php';
                        $logger = new ActivityLogger();
                        $logger->log('delete', 'course', null, 'حذف جميع المواد (' . $total_courses . ' مادة) - تم حذف ' . $total_registrations . ' تسجيل و ' . $total_grades . ' درجة - تم بواسطة: ' . ($_SESSION['username'] ?? 'غير معروف'));
                        
                        // تأكيد المعاملة
                        $db->getConnection()->commit();
                        
                        $message = '<div class="alert alert-success">✓ تم حذف جميع المواد بنجاح (' . $total_courses . ' مادة)</div>';
                        if ($total_registrations > 0) {
                            $message .= '<div class="alert alert-info">⚠️ تم حذف ' . $total_registrations . ' تسجيل طالب مرتبط</div>';
                        }
                        if ($total_grades > 0) {
                            $message .= '<div class="alert alert-info">⚠️ تم حذف ' . $total_grades . ' درجة مرتبطة</div>';
                        }
                    } catch (Exception $e) {
                        // إلغاء المعاملة في حالة الخطأ
                        $db->getConnection()->rollBack();
                        throw $e;
                    }
                }
            } catch (PDOException $e) {
                $message = '<div class="alert alert-error">خطأ في حذف جميع المواد: ' . htmlspecialchars($e->getMessage()) . '</div>';
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

// جلب رسالة من URL (بعد إعادة التوجيه)
if (isset($_GET['msg']) && isset($_GET['msg_type'])) {
    $message = '<div class="alert alert-' . ($_GET['msg_type'] === 'success' ? 'success' : 'error') . '">' . htmlspecialchars(urldecode($_GET['msg'])) . '</div>';
}

// جلب الكليات (للفلاتر)
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");
} else {
    $college_id = $_SESSION['college_id'] ?? 0;
    $colleges = $db->fetchAll("SELECT * FROM colleges WHERE id = ? ORDER BY name_ar", [$college_id]);
}

// جلب التخصصات (للفلاتر)
$majors_for_filter = [];
if ($filter_college_id > 0) {
    $majors_for_filter = $db->fetchAll(
        "SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$filter_college_id]
    );
} elseif ($auth->hasRole('admin')) {
    $majors_for_filter = $db->fetchAll("SELECT * FROM majors ORDER BY name_ar");
} else {
    $college_id = $_SESSION['college_id'] ?? 0;
    $majors_for_filter = $db->fetchAll(
        "SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
}

// جلب الفصول الدراسية (للفلاتر)
$semesters_for_filter = [];
if ($filter_major_id > 0) {
    $semesters_for_filter = $db->fetchAll(
        "SELECT * FROM semesters WHERE major_id = ? ORDER BY academic_year, semester_number",
        [$filter_major_id]
    );
} elseif ($major_id > 0) {
    $semesters_for_filter = $db->fetchAll(
        "SELECT * FROM semesters WHERE major_id = ? ORDER BY academic_year, semester_number",
        [$major_id]
    );
}

// جلب السنوات الدراسية المتاحة
$academic_years = $db->fetchAll(
    "SELECT DISTINCT academic_year FROM semesters ORDER BY academic_year DESC"
);

// جلب المواد مع معلومات الكلية والتخصص والفصل (بناءً على الفلاتر)
$where_conditions = [];
$where_params = [];

// التحقق من وجود فلاتر نشطة
$has_filters = $filter_college_id > 0 || $filter_major_id > 0 || $filter_semester_id > 0 || $filter_academic_year;

if ($has_filters) {
    // استخدام الفلاتر المحددة
    if ($filter_college_id > 0) {
        $where_conditions[] = "col.id = ?";
        $where_params[] = $filter_college_id;
    } elseif (!$auth->hasRole('admin')) {
        // إذا لم يكن admin ولم يحدد كلية، استخدم كلية المستخدم
        $college_id = $_SESSION['college_id'] ?? 0;
        if ($college_id > 0) {
            $where_conditions[] = "col.id = ?";
            $where_params[] = $college_id;
        }
    }
    
    if ($filter_major_id > 0) {
        $where_conditions[] = "m.id = ?";
        $where_params[] = $filter_major_id;
    }
    
    if ($filter_semester_id > 0) {
        // عرض مواد الفصل المحدد فقط (بدون المواد العامة)
        $where_conditions[] = "c.semester_id = ?";
        $where_params[] = $filter_semester_id;
    }
    
    if ($filter_academic_year) {
        // عرض مواد السنة المحددة فقط (بدون المواد العامة)
        $where_conditions[] = "s.academic_year = ?";
        $where_params[] = $filter_academic_year;
    }
    
    $where = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
} else {
    // إذا لم تكن هناك فلاتر، استخدم المنطق القديم
    $where = "WHERE c.major_id = ?";
    $where_params[] = $major_id;
    if ($semester_id) {
        // عرض مواد الفصل المحدد فقط (بدون المواد العامة)
        $where .= " AND c.semester_id = ?";
        $where_params[] = $semester_id;
    }
}

$courses = $db->fetchAll(
    "SELECT c.*, s.academic_year, s.semester_number, m.name_ar as major_name, m.college_id, col.name_ar as college_name
     FROM courses c 
     JOIN majors m ON c.major_id = m.id
     JOIN colleges col ON m.college_id = col.id
     LEFT JOIN semesters s ON c.semester_id = s.id 
     $where 
     ORDER BY col.name_ar, m.name_ar, s.academic_year, s.semester_number, c.code",
    $where_params
);

$semesters = $db->fetchAll(
    "SELECT * FROM semesters WHERE major_id = ? ORDER BY academic_year, semester_number",
    [$major_id]
);

// فتح نموذج التعديل تلقائياً عند الوصول برابط edit_course (من add_courses_bulk مثلاً)
$edit_course_to_open = null;
if (isset($_GET['edit_course']) && (int)$_GET['edit_course'] > 0) {
    $edit_course_id = (int)$_GET['edit_course'];
    $edit_course_to_open = $db->fetchOne(
        "SELECT c.*, s.academic_year, s.semester_number, m.name_ar as major_name, m.college_id, col.name_ar as college_name
         FROM courses c 
         JOIN majors m ON c.major_id = m.id
         JOIN colleges col ON m.college_id = col.id
         LEFT JOIN semesters s ON c.semester_id = s.id
         WHERE c.id = ? AND m.id = ?",
        [$edit_course_id, $major_id]
    );
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة المواد</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدارة المواد</h2>
            <div>
                <a href="upload_grades.php" class="btn btn-info" style="margin-left: 10px;">📤 رفع الدرجات من Excel</a>
                <a href="semesters.php?major_id=<?php echo $major_id; ?>" class="btn btn-secondary">العودة للفصول</a>
                <a href="add_courses_bulk.php" class="btn btn-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; margin-left: 10px;">➕ إضافة مواد دفعة واحدة</a>
                <button onclick="document.getElementById('addForm').style.display='block'" class="btn btn-primary">إضافة مادة</button>
                <?php if ($auth->hasRole('admin')): ?>
                    <button onclick="document.getElementById('deleteAllForm').style.display='block'" class="btn btn-danger">🗑️ حذف جميع المواد</button>
                <?php endif; ?>
            </div>
        </div>
        
        <?php echo $message; ?>
        
        <!-- قسم الفلاتر للبحث عن المواد -->
        <div class="card" style="margin-bottom: 20px; background: #f9f9f9;">
            <div class="card-header">
                <h3>🔍 البحث والتصفية - تعديل/حذف المواد</h3>
            </div>
            <form method="GET" style="padding: 20px;">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px;">
                    <div class="form-group">
                        <label>الكلية:</label>
                        <select name="filter_college_id" id="filter_college_id" onchange="loadMajorsForFilter()" style="width: 100%; padding: 8px;">
                            <option value="">-- جميع الكليات --</option>
                            <?php foreach ($colleges as $college): ?>
                                <option value="<?php echo $college['id']; ?>" <?php echo $filter_college_id == $college['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($college['name_ar']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>التخصص:</label>
                        <select name="filter_major_id" id="filter_major_id" onchange="loadSemestersForFilter()" style="width: 100%; padding: 8px;">
                            <option value="">-- جميع التخصصات --</option>
                            <?php foreach ($majors_for_filter as $major): ?>
                                <option value="<?php echo $major['id']; ?>" <?php echo $filter_major_id == $major['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($major['name_ar']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>الفصل الدراسي:</label>
                        <select name="filter_semester_id" id="filter_semester_id" style="width: 100%; padding: 8px;">
                            <option value="">-- جميع الفصول --</option>
                            <?php foreach ($semesters_for_filter as $sem): ?>
                                <option value="<?php echo $sem['id']; ?>" <?php echo $filter_semester_id == $sem['id'] ? 'selected' : ''; ?>>
                                    فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>السنة الدراسية:</label>
                        <select name="filter_academic_year" style="width: 100%; padding: 8px;">
                            <option value="">-- جميع السنوات --</option>
                            <?php foreach ($academic_years as $year): ?>
                                <option value="<?php echo htmlspecialchars($year['academic_year']); ?>" <?php echo $filter_academic_year == $year['academic_year'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($year['academic_year']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div style="margin-top: 15px; text-align: center;">
                    <button type="submit" class="btn btn-primary">🔍 عرض المواد</button>
                    <a href="courses.php" class="btn btn-secondary">إعادة تعيين</a>
                </div>
            </form>
        </div>
        
        <div id="addForm" class="card" style="display:none; margin-bottom: 20px;">
            <div class="card-header">
                <h3>إضافة مادة جديدة</h3>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="add">
                <?php if ($major_info): ?>
                <div class="form-group" style="background: #f0f0f0; padding: 10px; border-radius: 5px; margin-bottom: 15px;">
                    <strong>الكلية:</strong> <?php echo htmlspecialchars($major_info['college_name']); ?><br>
                    <strong>التخصص:</strong> <?php echo htmlspecialchars($major_info['name_ar']); ?>
                </div>
                <?php endif; ?>
                <div class="form-group">
                    <label>رمز المادة</label>
                    <input type="text" name="code" required>
                </div>
                <div class="form-group">
                    <label>اسم المادة (عربي)</label>
                    <input type="text" name="name_ar" required>
                </div>
                <div class="form-group">
                    <label>اسم المادة (إنجليزي)</label>
                    <input type="text" name="name_en">
                </div>
                <div class="form-group">
                    <label>عدد الساعات المعتمدة</label>
                    <input type="number" name="credit_hours" min="1" max="6" required>
                </div>
                <div class="form-group">
                    <label>الفصل الدراسي (اختياري)</label>
                    <select name="semester_id">
                        <option value="">جميع الفصول</option>
                        <?php foreach ($semesters as $sem): ?>
                            <option value="<?php echo $sem['id']; ?>" <?php echo $sem['id'] == $semester_id ? 'selected' : ''; ?>>
                                فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="is_required" checked> مادة إجبارية
                    </label>
                </div>
                <button type="submit" class="btn btn-success">إضافة</button>
                <button type="button" onclick="document.getElementById('addForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
            </form>
        </div>
        
        <div id="editForm" class="card" style="display:none; margin-bottom: 20px;">
            <div class="card-header">
                <h3>تعديل مادة</h3>
            </div>
            <form method="POST" id="editCourseForm">
                <input type="hidden" name="action" value="edit">
                <input type="hidden" name="course_id" id="edit_course_id">
                <div class="form-group" id="edit_course_info" style="background: #f0f0f0; padding: 10px; border-radius: 5px; margin-bottom: 15px;">
                    <!-- سيتم ملؤها بواسطة JavaScript -->
                </div>
                <div class="form-group">
                    <label>رمز المادة</label>
                    <input type="text" name="code" id="edit_code" required>
                </div>
                <div class="form-group">
                    <label>اسم المادة (عربي)</label>
                    <input type="text" name="name_ar" id="edit_name_ar" required>
                </div>
                <div class="form-group">
                    <label>اسم المادة (إنجليزي)</label>
                    <input type="text" name="name_en" id="edit_name_en">
                </div>
                <div class="form-group">
                    <label>عدد الساعات المعتمدة</label>
                    <input type="number" name="credit_hours" id="edit_credit_hours" min="1" max="6" required>
                </div>
                <div class="form-group">
                    <label>الفصل الدراسي (اختياري)</label>
                    <select name="semester_id" id="edit_semester_id">
                        <option value="">جميع الفصول</option>
                        <?php foreach ($semesters as $sem): ?>
                            <option value="<?php echo $sem['id']; ?>">
                                فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="is_required" id="edit_is_required"> مادة إجبارية
                    </label>
                </div>
                <button type="submit" class="btn btn-success">حفظ التعديلات</button>
                <button type="button" onclick="document.getElementById('editForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
            </form>
        </div>
        
        <!-- قسم حذف جميع المواد -->
        <?php if ($auth->hasRole('admin')): ?>
        <div id="deleteAllForm" class="card" style="display:none; margin-bottom: 20px; border: 3px solid #f44336;">
            <div class="card-header" style="background-color: #f44336; color: white;">
                <h3>⚠️ حذف جميع المواد</h3>
            </div>
            <div style="padding: 20px;">
                <?php
                // جلب عدد المواد والتسجيلات والدرجات
                try {
                    $total_courses = $db->fetchOne("SELECT COUNT(*) as count FROM courses")['count'];
                    $total_registrations = $db->fetchOne("SELECT COUNT(*) as count FROM student_courses")['count'];
                    $total_grades = $db->fetchOne("SELECT COUNT(*) as count FROM grades")['count'];
                } catch (Exception $e) {
                    $total_courses = 0;
                    $total_registrations = 0;
                    $total_grades = 0;
                }
                ?>
                <div style="background: #fff3cd; padding: 15px; border-radius: 5px; margin-bottom: 20px; border: 2px solid #ffc107;">
                    <h4 style="color: #856404; margin-top: 0;">⚠️ تحذير خطير!</h4>
                    <p style="color: #856404; font-weight: bold; margin-bottom: 10px;">
                        هذه العملية ستحذف <strong>جميع المواد</strong> من النظام بشكل نهائي ولا يمكن التراجع عنها!
                    </p>
                    <p style="color: #856404; margin-bottom: 5px;"><strong>عدد المواد الحالي: <?php echo number_format($total_courses); ?> مادة</strong></p>
                    <p style="color: #856404; margin-bottom: 5px;"><strong>عدد التسجيلات المرتبطة: <?php echo number_format($total_registrations); ?> تسجيل</strong></p>
                    <p style="color: #856404; margin-bottom: 5px;"><strong>عدد الدرجات المرتبطة: <?php echo number_format($total_grades); ?> درجة</strong></p>
                    <p style="color: #856404; margin-bottom: 0;">
                        <strong>سيتم حذف:</strong>
                    </p>
                    <ul style="color: #856404; margin-right: 20px; margin-top: 10px;">
                        <li>جميع المواد الدراسية</li>
                        <li>جميع تسجيلات الطلاب على المواد</li>
                        <li>جميع الدرجات المرتبطة بالمواد</li>
                        <li>جميع البيانات المرتبطة الأخرى</li>
                    </ul>
                </div>
                
                <form method="POST" onsubmit="return confirmDeleteAllCourses()">
                    <input type="hidden" name="action" value="delete_all">
                    
                    <div class="form-group">
                        <label style="color: #f44336; font-weight: bold; font-size: 16px;">
                            للتأكيد، يرجى كتابة: <code style="background: #f4f4f4; padding: 5px 10px; border-radius: 3px;">حذف جميع المواد</code>
                        </label>
                        <input type="text" 
                               name="confirm_text" 
                               id="confirmDeleteCoursesText"
                               placeholder="اكتب: حذف جميع المواد"
                               required
                               style="width: 100%; max-width: 400px; padding: 12px; border: 2px solid #f44336; border-radius: 5px; font-size: 16px;"
                               oninput="checkConfirmCoursesText()">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            يجب كتابة النص بالضبط كما هو موضح أعلاه
                        </small>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <button type="submit" 
                                id="deleteAllCoursesBtn"
                                class="btn btn-danger" 
                                disabled
                                style="padding: 12px 30px; font-size: 16px; font-weight: bold;">
                            🗑️ حذف جميع المواد (<?php echo number_format($total_courses); ?>)
                        </button>
                        <button type="button" 
                                onclick="document.getElementById('deleteAllForm').style.display='none'; document.getElementById('confirmDeleteCoursesText').value=''; checkConfirmCoursesText();" 
                                class="btn btn-secondary">
                            إلغاء
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="table-container">
            <?php if (empty($courses)): ?>
                <div class="alert alert-info" style="text-align: center; padding: 30px;">
                    <h3>لا توجد مواد</h3>
                    <p>لم يتم العثور على أي مواد تطابق معايير البحث المحددة.</p>
                    <p>يرجى تعديل الفلاتر أو <a href="courses.php">إعادة تعيين الفلاتر</a> لعرض جميع المواد.</p>
                </div>
            <?php else: ?>
                <div style="margin-bottom: 15px; padding: 10px; background: #e3f2fd; border-radius: 5px;">
                    <strong>عدد المواد المعروضة: <?php echo count($courses); ?></strong>
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>الكلية</th>
                            <th>التخصص</th>
                            <th>الرمز</th>
                            <th>اسم المادة</th>
                            <th>الساعات</th>
                            <th>الفصل</th>
                            <th>السنة</th>
                            <th>النوع</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($courses as $course): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($course['college_name']); ?></td>
                                <td><?php echo htmlspecialchars($course['major_name']); ?></td>
                                <td><?php echo htmlspecialchars($course['code']); ?></td>
                                <td><?php echo htmlspecialchars($course['name_ar']); ?></td>
                                <td><?php echo $course['credit_hours']; ?></td>
                                <td><?php 
                                    if ($course['semester_number']) {
                                        echo 'فصل ' . $course['semester_number'];
                                    } else {
                                        echo 'جميع الفصول';
                                    }
                                ?></td>
                                <td><?php echo $course['academic_year'] ?? '-'; ?></td>
                                <td><?php echo $course['is_required'] ? 'إجبارية' : 'اختيارية'; ?></td>
                                <td>
                                    <a href="grades.php?course_id=<?php echo $course['id']; ?>" class="btn btn-success" style="margin: 2px;">الدرجات</a>
                                    <?php if ($semester_id || $filter_semester_id): ?>
                                        <a href="register_students_to_course.php?course_id=<?php echo $course['id']; ?>&semester_id=<?php echo $semester_id ?: $filter_semester_id; ?>" class="btn btn-primary" style="margin: 2px;">تسجيل طلاب</a>
                                    <?php endif; ?>
                                    <button onclick="editCourse(<?php echo $course['id']; ?>, '<?php echo htmlspecialchars($course['code'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['name_ar'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['name_en'] ?? '', ENT_QUOTES); ?>', <?php echo $course['credit_hours']; ?>, <?php echo $course['is_required']; ?>, <?php echo $course['semester_id'] ?? 'null'; ?>, '<?php echo htmlspecialchars($course['college_name'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['major_name'], ENT_QUOTES); ?>', '<?php echo $course['academic_year'] ?? ''; ?>', <?php echo $course['semester_number'] ?? 'null'; ?>)" class="btn btn-info" style="margin: 2px;">تعديل</button>
                                    <button onclick="deleteCourse(<?php echo $course['id']; ?>, '<?php echo htmlspecialchars($course['name_ar'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['code'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['college_name'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($course['major_name'], ENT_QUOTES); ?>', '<?php echo $course['academic_year'] ?? ''; ?>', <?php echo $course['semester_number'] ?? 'null'; ?>)" class="btn btn-danger" style="margin: 2px;">حذف</button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
    
    <script>
    // دالة تحميل التخصصات عند تغيير الكلية
    function loadMajorsForFilter() {
        const collegeId = document.getElementById('filter_college_id').value;
        const majorSelect = document.getElementById('filter_major_id');
        const semesterSelect = document.getElementById('filter_semester_id');
        
        majorSelect.innerHTML = '<option value="">-- جميع التخصصات --</option>';
        semesterSelect.innerHTML = '<option value="">-- جميع الفصول --</option>';
        majorSelect.disabled = !collegeId;
        semesterSelect.disabled = true;
        
        if (!collegeId) return;
        
        fetch(`courses.php?action=get_majors&college_id=${collegeId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(major => {
                    const option = document.createElement('option');
                    option.value = major.id;
                    option.textContent = major.name_ar;
                    majorSelect.appendChild(option);
                });
                majorSelect.disabled = false;
            })
            .catch(error => {
                console.error('Error loading majors:', error);
            });
    }
    
    // دالة تحميل الفصول عند تغيير التخصص
    function loadSemestersForFilter() {
        const majorId = document.getElementById('filter_major_id').value;
        const semesterSelect = document.getElementById('filter_semester_id');
        
        semesterSelect.innerHTML = '<option value="">-- جميع الفصول --</option>';
        semesterSelect.disabled = !majorId;
        
        if (!majorId) return;
        
        fetch(`courses.php?action=get_semesters&major_id=${majorId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(semester => {
                    const option = document.createElement('option');
                    option.value = semester.id;
                    option.textContent = 'فصل ' + semester.semester_number + ' - ' + semester.academic_year;
                    semesterSelect.appendChild(option);
                });
                semesterSelect.disabled = false;
            })
            .catch(error => {
                console.error('Error loading semesters:', error);
            });
    }
    
    // دالة تعديل المادة
    function editCourse(courseId, code, nameAr, nameEn, creditHours, isRequired, semesterId, collegeName, majorName, academicYear, semesterNumber) {
        // إخفاء نموذج الإضافة إذا كان مفتوحاً
        document.getElementById('addForm').style.display = 'none';
        
        // تعبئة معلومات المادة
        const courseInfo = document.getElementById('edit_course_info');
        let infoHtml = '<strong>الكلية:</strong> ' + collegeName + '<br>';
        infoHtml += '<strong>التخصص:</strong> ' + majorName + '<br>';
        if (academicYear && semesterNumber) {
            infoHtml += '<strong>الفصل الدراسي:</strong> فصل ' + semesterNumber + ' - ' + academicYear;
        } else {
            infoHtml += '<strong>الفصل الدراسي:</strong> جميع الفصول';
        }
        courseInfo.innerHTML = infoHtml;
        
        // تعبئة نموذج التعديل بالبيانات
        document.getElementById('edit_course_id').value = courseId;
        document.getElementById('edit_code').value = code;
        document.getElementById('edit_name_ar').value = nameAr;
        document.getElementById('edit_name_en').value = nameEn || '';
        document.getElementById('edit_credit_hours').value = creditHours;
        document.getElementById('edit_is_required').checked = isRequired == 1;
        document.getElementById('edit_semester_id').value = semesterId || '';
        
        // إظهار نموذج التعديل
        document.getElementById('editForm').style.display = 'block';
        
        // التمرير إلى نموذج التعديل
        document.getElementById('editForm').scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    // دالة حذف المادة
    function deleteCourse(courseId, courseName, courseCode, collegeName, majorName, academicYear, semesterNumber) {
        let confirmMsg = '⚠️ تحذير: حذف المادة\n\n';
        confirmMsg += 'الكلية: ' + collegeName + '\n';
        confirmMsg += 'التخصص: ' + majorName + '\n';
        if (academicYear && semesterNumber) {
            confirmMsg += 'الفصل الدراسي: فصل ' + semesterNumber + ' - ' + academicYear + '\n';
        } else {
            confirmMsg += 'الفصل الدراسي: جميع الفصول\n';
        }
        confirmMsg += 'الرمز: ' + courseCode + '\n';
        confirmMsg += 'الاسم: ' + courseName + '\n\n';
        confirmMsg += '⚠️ سيتم حذف المادة مع جميع:\n';
        confirmMsg += '  • تسجيلات الطلاب المرتبطة بها\n';
        confirmMsg += '  • الدرجات المرتبطة بها\n\n';
        confirmMsg += 'هذا الإجراء لا يمكن التراجع عنه!\n\n';
        confirmMsg += 'هل أنت متأكد من المتابعة؟';
        
        if (confirm(confirmMsg)) {
            // إنشاء نموذج مخفي لإرسال طلب الحذف
            const form = document.createElement('form');
            form.method = 'POST';
            
            // بناء URL مع الفلاتر من URL الحالي
            const currentUrl = new URL(window.location.href);
            const params = new URLSearchParams();
            
            // إضافة جميع المعاملات من URL الحالي
            if (currentUrl.searchParams.get('major_id')) {
                params.append('major_id', currentUrl.searchParams.get('major_id'));
            }
            if (currentUrl.searchParams.get('semester_id')) {
                params.append('semester_id', currentUrl.searchParams.get('semester_id'));
            }
            if (currentUrl.searchParams.get('filter_college_id')) {
                params.append('filter_college_id', currentUrl.searchParams.get('filter_college_id'));
            }
            if (currentUrl.searchParams.get('filter_major_id')) {
                params.append('filter_major_id', currentUrl.searchParams.get('filter_major_id'));
            }
            if (currentUrl.searchParams.get('filter_semester_id')) {
                params.append('filter_semester_id', currentUrl.searchParams.get('filter_semester_id'));
            }
            if (currentUrl.searchParams.get('filter_academic_year')) {
                params.append('filter_academic_year', currentUrl.searchParams.get('filter_academic_year'));
            }
            
            let url = 'courses.php';
            if (params.toString()) {
                url += '?' + params.toString();
            }
            form.action = url;
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'delete';
            form.appendChild(actionInput);
            
            const courseIdInput = document.createElement('input');
            courseIdInput.type = 'hidden';
            courseIdInput.name = 'course_id';
            courseIdInput.value = courseId;
            form.appendChild(courseIdInput);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // دالة للتحقق من نص التأكيد لحذف جميع المواد
    function checkConfirmCoursesText() {
        const confirmText = document.getElementById('confirmDeleteCoursesText');
        const deleteBtn = document.getElementById('deleteAllCoursesBtn');
        const expectedText = 'حذف جميع المواد';
        
        if (confirmText && deleteBtn) {
            if (confirmText.value.trim() === expectedText) {
                deleteBtn.disabled = false;
                deleteBtn.style.opacity = '1';
            } else {
                deleteBtn.disabled = true;
                deleteBtn.style.opacity = '0.5';
            }
        }
    }
    
    // دالة تأكيد نهائية قبل حذف جميع المواد
    function confirmDeleteAllCourses() {
        const confirmText = document.getElementById('confirmDeleteCoursesText');
        const expectedText = 'حذف جميع المواد';
        
        if (!confirmText || confirmText.value.trim() !== expectedText) {
            alert('⚠️ يجب كتابة "حذف جميع المواد" بالضبط للتأكيد');
            return false;
        }
        
        <?php
        // جلب عدد المواد للاستخدام في JavaScript
        try {
            $js_total_courses = $db->fetchOne("SELECT COUNT(*) as count FROM courses")['count'];
            $js_total_registrations = $db->fetchOne("SELECT COUNT(*) as count FROM student_courses")['count'];
            $js_total_grades = $db->fetchOne("SELECT COUNT(*) as count FROM grades")['count'];
        } catch (Exception $e) {
            $js_total_courses = 0;
            $js_total_registrations = 0;
            $js_total_grades = 0;
        }
        ?>
        const totalCourses = <?php echo $js_total_courses; ?>;
        const totalRegistrations = <?php echo $js_total_registrations; ?>;
        const totalGrades = <?php echo $js_total_grades; ?>;
        
        if (totalCourses === 0) {
            alert('لا توجد مواد لحذفها');
            return false;
        }
        
        const finalConfirm = confirm(
            '⚠️⚠️⚠️ تحذير نهائي ⚠️⚠️⚠️\n\n' +
            'أنت على وشك حذف جميع المواد (' + totalCourses + ' مادة) من النظام!\n\n' +
            'هذه العملية:\n' +
            '• لا يمكن التراجع عنها\n' +
            '• ستحذف جميع المواد الدراسية\n' +
            '• ستحذف ' + totalRegistrations + ' تسجيل طالب\n' +
            '• ستحذف ' + totalGrades + ' درجة\n' +
            '• ستحذف جميع البيانات المرتبطة\n\n' +
            'هل أنت متأكد تماماً من رغبتك في المتابعة؟'
        );
        
        if (!finalConfirm) {
            return false;
        }
        
        // تأكيد إضافي
        const secondConfirm = confirm(
            '⚠️ تأكيد نهائي أخير ⚠️\n\n' +
            'أنت على وشك حذف ' + totalCourses + ' مادة بشكل نهائي!\n\n' +
            'سيتم حذف:\n' +
            '• ' + totalCourses + ' مادة\n' +
            '• ' + totalRegistrations + ' تسجيل طالب\n' +
            '• ' + totalGrades + ' درجة\n\n' +
            'اضغط "موافق" فقط إذا كنت متأكداً 100%'
        );
        
        return secondConfirm;
    }
    <?php if ($edit_course_to_open): ?>
    document.addEventListener('DOMContentLoaded', function() {
        editCourse(
            <?php echo (int)$edit_course_to_open['id']; ?>,
            <?php echo json_encode($edit_course_to_open['code']); ?>,
            <?php echo json_encode($edit_course_to_open['name_ar']); ?>,
            <?php echo json_encode($edit_course_to_open['name_en'] ?? ''); ?>,
            <?php echo (int)$edit_course_to_open['credit_hours']; ?>,
            <?php echo (int)$edit_course_to_open['is_required']; ?>,
            <?php echo $edit_course_to_open['semester_id'] ? (int)$edit_course_to_open['semester_id'] : 'null'; ?>,
            <?php echo json_encode($edit_course_to_open['college_name']); ?>,
            <?php echo json_encode($edit_course_to_open['major_name']); ?>,
            <?php echo json_encode($edit_course_to_open['academic_year'] ?? ''); ?>,
            <?php echo $edit_course_to_open['semester_number'] !== null ? (int)$edit_course_to_open['semester_number'] : 'null'; ?>
        );
    });
    <?php endif; ?>
    </script>
</body>
</html>

