<?php
/**
 * حذف فصل دراسي معين وما يحتويه من تسجيلات ودرجات ومعدلات ورسوم
 * Delete a semester and all related data (registrations, grades, GPAs, fees, attendance)
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

// دالة لتحويل رقم الفصل إلى نص عربي (الفصل الأول، الثاني، ...)
function getSemesterName($number) {
    switch ((int)$number) {
        case 1: return 'الفصل الأول';
        case 2: return 'الفصل الثاني';
        case 3: return 'الفصل الثالث';
        case 4: return 'الفصل الرابع';
        case 5: return 'الفصل الخامس';
        case 6: return 'الفصل السادس';
        case 7: return 'الفصل السابع';
        case 8: return 'الفصل الثامن';
        default: return 'فصل ' . (int)$number;
    }
}
$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$semester_number = isset($_GET['semester_number']) ? (int)$_GET['semester_number'] : 0;
$academic_year = isset($_GET['academic_year']) ? trim($_GET['academic_year']) : '';
$semester_id = 0;
$semester = null;
$message = '';

// معالجة الحذف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_semester') {
    $confirm_word = trim($_POST['confirm_word'] ?? '');
    $post_semester_id = (int)($_POST['semester_id'] ?? 0);

    if ($confirm_word !== 'حذف') {
        $message = '<div class="alert alert-error">يجب كتابة كلمة "حذف" للتأكيد.</div>';
    } elseif ($post_semester_id <= 0) {
        $message = '<div class="alert alert-error">خطأ: لم يتم تحديد الفصل الدراسي.</div>';
    } else {
        $semester_check = $db->fetchOne("SELECT s.*, m.college_id FROM semesters s JOIN majors m ON s.major_id = m.id WHERE s.id = ?", [$post_semester_id]);
        if (!$semester_check) {
            $message = '<div class="alert alert-error">الفصل الدراسي غير موجود.</div>';
        } elseif (!$auth->belongsToCollege($semester_check['college_id']) && !$auth->hasRole('admin')) {
            $message = '<div class="alert alert-error">غير مصرح لك بحذف هذا الفصل.</div>';
        } else {
            try {
                $db->getConnection()->beginTransaction();

                // جمع الطلاب المتأثرين (لإعادة حساب المعدل التراكمي لاحقاً)
                $affected_students = $db->fetchAll("SELECT DISTINCT student_id FROM student_courses WHERE semester_id = ?", [$post_semester_id]);
                $affected_student_ids = array_column($affected_students, 'student_id');

                // 1) حذف الدرجات المرتبطة بهذا الفصل
                $db->query("DELETE FROM grades WHERE semester_id = ?", [$post_semester_id]);

                // 2) حذف تسجيلات الطلاب في المواد (student_courses)
                $db->query("DELETE FROM student_courses WHERE semester_id = ?", [$post_semester_id]);

                // 3) حذف المعدلات الفصلية
                $db->query("DELETE FROM semester_gpas WHERE semester_id = ?", [$post_semester_id]);

                // 4) حذف الرسوم
                $db->query("DELETE FROM fees WHERE semester_id = ?", [$post_semester_id]);

                // 5) حذف الحضور إن وُجد الجدول
                try {
                    $db->query("DELETE FROM attendance WHERE semester_id = ?", [$post_semester_id]);
                } catch (Exception $e) {
                    // جدول الحضور قد لا يكون موجوداً
                }

                // 6) جداول إضافية إن وُجدت (نشر النتائج، المعايرة، أرشيف النتائج)
                foreach (['published_results', 'semester_calibrations', 'results_archive'] as $tbl) {
                    try {
                        $db->query("DELETE FROM {$tbl} WHERE semester_id = ?", [$post_semester_id]);
                    } catch (Exception $e) {
                        // تجاهل إن لم يكن الجدول أو العمود موجوداً
                    }
                }

                // 7) إلغاء ربط المواد بهذا الفصل (courses.semester_id = NULL)
                $db->query("UPDATE courses SET semester_id = NULL WHERE semester_id = ?", [$post_semester_id]);

                // 8) حذف سجل الفصل الدراسي
                $db->query("DELETE FROM semesters WHERE id = ?", [$post_semester_id]);

                $db->getConnection()->commit();

                // إعادة حساب المعدل التراكمي للطلاب المتأثرين
                foreach ($affected_student_ids as $sid) {
                    try {
                        $gradeCalculator->updateAllGPAs($sid);
                    } catch (Exception $e) {
                        error_log("GPA update after semester delete (student {$sid}): " . $e->getMessage());
                    }
                }

                $message = '<div class="alert alert-success">تم حذف الفصل الدراسي وجميع بياناته (تسجيلات، درجات، معدلات، رسوم) بنجاح. تم تحديث المعدلات التراكمية للطلاب المتأثرين.</div>';
                $semester_id = 0;
                $semester = null;
            } catch (Exception $e) {
                $db->getConnection()->rollBack();
                $message = '<div class="alert alert-error">خطأ عند الحذف: ' . htmlspecialchars($e->getMessage()) . '</div>';
                error_log("delete_semester: " . $e->getMessage());
            }
        }
    }
}

// جلب البيانات للعرض والاختيار
$college = null;
$major = null;
$available_colleges = [];
if ($auth->hasRole('admin')) {
    $available_colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");
} else {
    $college_id = (int)($_SESSION['college_id'] ?? 0);
    $available_colleges = $db->fetchAll("SELECT * FROM colleges WHERE id = ?", [$college_id]);
}

$available_majors = [];
$available_semester_numbers = [];
$available_academic_years = [];
if ($college_id > 0) {
    $college = $db->fetchOne("SELECT * FROM colleges WHERE id = ?", [$college_id]);
    if ($college && ($auth->belongsToCollege($college_id) || $auth->hasRole('admin'))) {
        $available_majors = $db->fetchAll("SELECT * FROM majors WHERE college_id = ? ORDER BY name_ar", [$college_id]);
    }
}
if ($major_id > 0) {
    $major = $db->fetchOne("SELECT m.*, c.id as college_id FROM majors m JOIN colleges c ON m.college_id = c.id WHERE m.id = ?", [$major_id]);
    if ($major && ($auth->belongsToCollege($major['college_id']) || $auth->hasRole('admin'))) {
        $semesters_list = $db->fetchAll("SELECT DISTINCT semester_number FROM semesters WHERE major_id = ? ORDER BY semester_number", [$major_id]);
        $available_semester_numbers = array_column($semesters_list, 'semester_number');
        if ($semester_number > 0) {
            $years_list = $db->fetchAll(
                "SELECT DISTINCT academic_year FROM semesters WHERE major_id = ? AND semester_number = ? ORDER BY academic_year DESC",
                [$major_id, $semester_number]
            );
            $available_academic_years = array_column($years_list, 'academic_year');
            if ($academic_year !== '') {
                $semester = $db->fetchOne(
                    "SELECT s.*, m.name_ar as major_name FROM semesters s JOIN majors m ON s.major_id = m.id WHERE s.major_id = ? AND s.semester_number = ? AND s.academic_year = ?",
                    [$major_id, $semester_number, $academic_year]
                );
                if ($semester) {
                    $semester_id = (int)$semester['id'];
                }
            }
        }
    }
}

// إحصائيات الفصل المحدد (للعرض قبل الحذف)
$stats = ['student_courses' => 0, 'grades' => 0, 'semester_gpas' => 0, 'fees' => 0];
if ($semester_id > 0) {
    $stats['student_courses'] = $db->fetchOne("SELECT COUNT(*) as c FROM student_courses WHERE semester_id = ?", [$semester_id])['c'];
    $stats['grades'] = $db->fetchOne("SELECT COUNT(*) as c FROM grades WHERE semester_id = ?", [$semester_id])['c'];
    $stats['semester_gpas'] = $db->fetchOne("SELECT COUNT(*) as c FROM semester_gpas WHERE semester_id = ?", [$semester_id])['c'];
    $stats['fees'] = $db->fetchOne("SELECT COUNT(*) as c FROM fees WHERE semester_id = ?", [$semester_id])['c'];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>حذف فصل دراسي</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <div class="container">
        <div class="page-header">
            <h2>حذف فصل دراسي وما يحتويه</h2>
            <p style="color:#666;">اختر الكلية والتخصص ورقم الفصل والسنة الدراسية، ثم أكد الحذف. سيتم حذف التسجيلات والدرجات والمعدلات الفصلية والرسوم المرتبطة بهذا الفصل.</p>
            <div style="margin-top: 10px;">
                <a href="dashboard.php" class="btn btn-secondary">العودة للوحة التحكم</a>
            </div>
        </div>
        <?php echo $message; ?>

        <div class="card" style="margin-bottom: 20px;">
            <div class="card-header"><h3>اختر الفصل الدراسي المراد حذفه</h3></div>
            <form method="GET" style="padding: 20px;">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 20px;">
                    <div class="form-group">
                        <label style="display: block; margin-bottom: 5px; font-weight: bold;">1. الكلية:</label>
                        <select name="college_id" required style="width: 100%; padding: 10px;" onchange="this.form.submit()">
                            <option value="">-- اختر الكلية --</option>
                            <?php foreach ($available_colleges as $c): ?>
                                <option value="<?php echo $c['id']; ?>" <?php echo ($college_id == $c['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($c['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label style="display: block; margin-bottom: 5px; font-weight: bold;">2. التخصص:</label>
                        <select name="major_id" required style="width: 100%; padding: 10px;" <?php echo ($college_id <= 0) ? 'disabled' : ''; ?> onchange="this.form.submit()">
                            <option value="">-- اختر التخصص --</option>
                            <?php if ($college_id > 0): foreach ($available_majors as $m): ?>
                                <option value="<?php echo $m['id']; ?>" <?php echo ($major_id == $m['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($m['name_ar']); ?></option>
                            <?php endforeach; endif; ?>
                        </select>
                        <?php if ($college_id > 0): ?><input type="hidden" name="college_id" value="<?php echo $college_id; ?>"><?php endif; ?>
                    </div>
                    <div class="form-group">
                        <label style="display: block; margin-bottom: 5px; font-weight: bold;">3. رقم الفصل:</label>
                        <select name="semester_number" required style="width: 100%; padding: 10px;" <?php echo ($major_id <= 0) ? 'disabled' : ''; ?> onchange="this.form.submit()">
                            <option value="">-- اختر الفصل --</option>
                            <?php if ($major_id > 0): foreach ($available_semester_numbers as $n): ?>
                                <option value="<?php echo $n; ?>" <?php echo ($semester_number == $n) ? 'selected' : ''; ?>><?php echo getSemesterName($n); ?></option>
                            <?php endforeach; endif; ?>
                        </select>
                        <?php if ($college_id > 0): ?><input type="hidden" name="college_id" value="<?php echo $college_id; ?>"><?php endif; ?>
                        <?php if ($major_id > 0): ?><input type="hidden" name="major_id" value="<?php echo $major_id; ?>"><?php endif; ?>
                    </div>
                    <div class="form-group">
                        <label style="display: block; margin-bottom: 5px; font-weight: bold;">4. السنة الدراسية:</label>
                        <select name="academic_year" required style="width: 100%; padding: 10px;" <?php echo ($semester_number <= 0) ? 'disabled' : ''; ?> onchange="this.form.submit()">
                            <option value="">-- اختر السنة --</option>
                            <?php if ($semester_number > 0): foreach ($available_academic_years as $y): ?>
                                <option value="<?php echo htmlspecialchars($y); ?>" <?php echo ($academic_year === $y) ? 'selected' : ''; ?>><?php echo htmlspecialchars($y); ?></option>
                            <?php endforeach; endif; ?>
                        </select>
                        <?php if ($college_id > 0): ?><input type="hidden" name="college_id" value="<?php echo $college_id; ?>"><?php endif; ?>
                        <?php if ($major_id > 0): ?><input type="hidden" name="major_id" value="<?php echo $major_id; ?>"><?php endif; ?>
                        <?php if ($semester_number > 0): ?><input type="hidden" name="semester_number" value="<?php echo $semester_number; ?>"><?php endif; ?>
                    </div>
                </div>
            </form>
        </div>

        <?php if ($semester && $semester_id > 0): ?>
        <div class="card" style="border: 2px solid #c62828;">
            <div class="card-header" style="background-color: #c62828; color: white;">
                <h3>⚠ تأكيد حذف الفصل الدراسي</h3>
            </div>
            <div style="padding: 20px;">
                <p><strong>الفصل المحدد:</strong> <?php echo htmlspecialchars($semester['major_name'] ?? ''); ?> — فصل <?php echo (int)$semester['semester_number']; ?> — <?php echo htmlspecialchars($semester['academic_year'] ?? ''); ?></p>
                <p>سيتم حذف التالي:</p>
                <ul>
                    <li>تسجيلات الطلاب في المواد (student_courses): <strong><?php echo (int)$stats['student_courses']; ?></strong></li>
                    <li>الدرجات (grades): <strong><?php echo (int)$stats['grades']; ?></strong></li>
                    <li>المعدلات الفصلية (semester_gpas): <strong><?php echo (int)$stats['semester_gpas']; ?></strong></li>
                    <li>الرسوم (fees): <strong><?php echo (int)$stats['fees']; ?></strong></li>
                    <li>ربط المواد بهذا الفصل سيُلغى (courses.semester_id = NULL)</li>
                    <li>سجل الفصل الدراسي (semesters)</li>
                </ul>
                <p style="color: #c62828; font-weight: bold;">هذا الإجراء لا يمكن التراجع عنه.</p>
                <form method="POST" onsubmit="return confirm('هل أنت متأكد من حذف هذا الفصل وجميع بياناته؟');">
                    <input type="hidden" name="action" value="delete_semester">
                    <input type="hidden" name="semester_id" value="<?php echo $semester_id; ?>">
                    <div class="form-group" style="margin: 15px 0;">
                        <label for="confirm_word">للتأكيد، اكتب كلمة <strong>حذف</strong>:</label>
                        <input type="text" id="confirm_word" name="confirm_word" required placeholder="حذف" style="padding: 10px; width: 200px;" dir="rtl" autocomplete="off">
                    </div>
                    <button type="submit" class="btn" style="background-color: #c62828; color: white;">حذف الفصل الدراسي</button>
                </form>
            </div>
        </div>
        <?php endif; ?>
    </div>
    <?php include 'includes/footer.php'; ?>
</body>
</html>
