<?php
/**
 * شهادة التفاصيل
 * Detailed Certificate
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

$student_id = isset($_GET['student_id']) ? (int)$_GET['student_id'] : 0;
$selected_semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$lang = isset($_GET['lang']) ? $_GET['lang'] : 'ar'; // ar للعربية، en للإنجليزية
if (!in_array($lang, ['ar', 'en'])) {
    $lang = 'ar'; // افتراضي العربية
}

// إذا كان طالب، استخدم معرفه
if ($auth->hasRole('student')) {
    $user = $auth->getCurrentUser();
    $student = $db->fetchOne("SELECT id FROM students WHERE user_id = ?", [$user['id']]);
    if ($student) {
        $student_id = $student['id'];
    }
}

if (!$student_id) {
    header('Location: dashboard.php');
    exit;
}

// التحقق من الصلاحيات
if (!$auth->canAccessStudent($student_id) && !$auth->hasRole('admin')) {
    header('Location: dashboard.php');
    exit;
}

// معالجة رفع الشعار
$upload_message = '';
$upload_message_type = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['logo']) && isset($_POST['upload_type']) && $_POST['upload_type'] === 'logo') {
    // التحقق من الصلاحيات (فقط للمديرين)
    if ($auth->hasRole('admin') || $auth->hasRole('college_admin')) {
        // إنشاء جدول settings إذا لم يكن موجوداً
        try {
            $db->query("SELECT 1 FROM settings LIMIT 1");
        } catch (Exception $e) {
            $db->query("CREATE TABLE IF NOT EXISTS settings (
                setting_key VARCHAR(255) PRIMARY KEY,
                setting_value TEXT,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        }
        
        $upload_dir = __DIR__ . '/assets/images/logo/';
        
        // إنشاء المجلد إذا لم يكن موجوداً
        if (!file_exists($upload_dir)) {
            @mkdir($upload_dir, 0777, true);
        }
        
        $file = $_FILES['logo'];
        
        // التحقق من وجود ملف
        if ($file['error'] === UPLOAD_ERR_OK) {
            $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/svg+xml'];
            $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
            
            // محاولة الحصول على نوع MIME بطريقة آمنة
            $file_type = null;
            if (function_exists('finfo_open')) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                if ($finfo) {
                    $file_type = finfo_file($finfo, $file['tmp_name']);
                    finfo_close($finfo);
                }
            }
            if (!$file_type && function_exists('mime_content_type')) {
                $file_type = mime_content_type($file['tmp_name']);
            }
            
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $extension_valid = in_array($extension, $allowed_extensions);
            
            $file_valid = false;
            if ($file_type && in_array($file_type, $allowed_types)) {
                $file_valid = true;
            } elseif (!$file_type && $extension_valid) {
                $file_valid = true;
            }
            
            if (!$file_valid) {
                $upload_message = 'نوع الملف غير مدعوم. يرجى رفع صورة (JPG, PNG, GIF, SVG)';
                $upload_message_type = 'error';
            } elseif ($file['size'] > 5 * 1024 * 1024) {
                $upload_message = 'حجم الملف كبير جداً. الحد الأقصى 5MB';
                $upload_message_type = 'error';
            } else {
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = 'university_logo_' . time() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                // حذف الشعار القديم إذا كان موجوداً
                $current_logo = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'university_logo_path'");
                if ($current_logo && !empty($current_logo['setting_value']) && file_exists(__DIR__ . '/' . $current_logo['setting_value'])) {
                    @unlink(__DIR__ . '/' . $current_logo['setting_value']);
                }
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $relative_path = 'assets/images/logo/' . $filename;
                    $db->query(
                        "INSERT INTO settings (setting_key, setting_value) VALUES ('university_logo_path', ?)
                         ON DUPLICATE KEY UPDATE setting_value = ?",
                        [$relative_path, $relative_path]
                    );
                    $upload_message = 'تم رفع الشعار بنجاح';
                    $upload_message_type = 'success';
                } else {
                    $upload_message = 'فشل رفع الملف. يرجى المحاولة مرة أخرى';
                    $upload_message_type = 'error';
                }
            }
        } else {
            $upload_message = 'حدث خطأ أثناء رفع الملف';
            $upload_message_type = 'error';
        }
        
        // إعادة التوجيه لتجنب إعادة إرسال النموذج
        header('Location: detailed_certificate.php?student_id=' . $student_id . ($selected_semester_id > 0 ? '&semester_id=' . $selected_semester_id : '') . '&lang=' . $lang . '&msg=' . urlencode($upload_message) . '&msg_type=' . $upload_message_type);
        exit;
    }
}

// معالجة رفع صورة الطالب
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['student_photo']) && isset($_POST['upload_type']) && $_POST['upload_type'] === 'student_photo') {
    $upload_dir = __DIR__ . '/uploads/students/';
    
    // إنشاء المجلد إذا لم يكن موجوداً
    if (!file_exists($upload_dir)) {
        @mkdir($upload_dir, 0777, true);
    }
    
    $file = $_FILES['student_photo'];
    
    // التحقق من وجود ملف
    if ($file['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif'];
        
        // محاولة الحصول على نوع MIME بطريقة آمنة
        $file_type = null;
        if (function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo) {
                $file_type = finfo_file($finfo, $file['tmp_name']);
                finfo_close($finfo);
            }
        }
        if (!$file_type && function_exists('mime_content_type')) {
            $file_type = mime_content_type($file['tmp_name']);
        }
        
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $extension_valid = in_array($extension, $allowed_extensions);
        
        $file_valid = false;
        if ($file_type && in_array($file_type, $allowed_types)) {
            $file_valid = true;
        } elseif (!$file_type && $extension_valid) {
            $file_valid = true;
        }
        
        if (!$file_valid) {
            $upload_message = 'نوع الملف غير مدعوم. يرجى رفع صورة (JPG, PNG, GIF)';
            $upload_message_type = 'error';
        } elseif ($file['size'] > 5 * 1024 * 1024) {
            $upload_message = 'حجم الملف كبير جداً. الحد الأقصى 5MB';
            $upload_message_type = 'error';
        } else {
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'student_' . $student_id . '_' . time() . '.' . $extension;
            $file_path = $upload_dir . $filename;
            
            // جلب معلومات الطالب (مؤقتاً)
            $temp_student = $db->fetchOne("SELECT photo_path FROM students WHERE id = ?", [$student_id]);
            
            // حذف الصورة القديمة إذا كانت موجودة
            if ($temp_student && !empty($temp_student['photo_path']) && file_exists(__DIR__ . '/' . $temp_student['photo_path'])) {
                @unlink(__DIR__ . '/' . $temp_student['photo_path']);
            }
            
            if (move_uploaded_file($file['tmp_name'], $file_path)) {
                $relative_path = 'uploads/students/' . $filename;
                $db->query(
                    "UPDATE students SET photo_path = ? WHERE id = ?",
                    [$relative_path, $student_id]
                );
                $upload_message = 'تم رفع صورة الطالب بنجاح';
                $upload_message_type = 'success';
            } else {
                $upload_message = 'فشل رفع الملف. يرجى المحاولة مرة أخرى';
                $upload_message_type = 'error';
            }
        }
    } else {
        $upload_message = 'حدث خطأ أثناء رفع الملف';
        $upload_message_type = 'error';
    }
    
    // إعادة التوجيه لتجنب إعادة إرسال النموذج
    header('Location: detailed_certificate.php?student_id=' . $student_id . ($selected_semester_id > 0 ? '&semester_id=' . $selected_semester_id : '') . '&lang=' . $lang . '&msg=' . urlencode($upload_message) . '&msg_type=' . $upload_message_type);
    exit;
}

// جلب رسالة الرفع من URL (بعد إعادة التوجيه)
if (isset($_GET['msg']) && isset($_GET['msg_type'])) {
    $upload_message = urldecode($_GET['msg']);
    $upload_message_type = $_GET['msg_type'];
}

// جلب معلومات الطالب
$student = $db->fetchOne(
    "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name
     FROM students s
     JOIN majors m ON s.major_id = m.id
     JOIN colleges c ON s.college_id = c.id
     WHERE s.id = ?",
    [$student_id]
);

if (!$student) {
    header('Location: dashboard.php');
    exit;
}

// جلب شعار الجامعة
$university_logo_path = null;

try {
    $logo_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'university_logo_path'");
    
    if ($logo_setting && !empty($logo_setting['setting_value'])) {
        $logo_path_raw = trim($logo_setting['setting_value']);
        
        // تنظيف المسار من المسافات والأحرف غير المرغوب فيها
        $logo_path_raw = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $logo_path_raw);
        $logo_path_raw = ltrim($logo_path_raw, '/\\');
        
        // بناء المسار الكامل
        $full_path = __DIR__ . DIRECTORY_SEPARATOR . $logo_path_raw;
        
        // التحقق من وجود الملف
        if (file_exists($full_path) && is_file($full_path) && filesize($full_path) > 0) {
            // استخدام المسار النسبي للصورة (كما هو محفوظ في قاعدة البيانات)
            $university_logo_path = str_replace('\\', '/', $logo_path_raw); // تحويل لمسار URL
        } else {
            error_log("University logo file not found or empty: " . $full_path);
        }
    }
} catch (Exception $e) {
    $university_logo_path = null;
    error_log("Error loading university logo: " . $e->getMessage());
}

// التحقق من وجود حقل photo_path في جدول students وإضافته إذا لم يكن موجوداً
try {
    $db->query("SELECT photo_path FROM students LIMIT 1");
} catch (Exception $e) {
    // إضافة الحقل إذا لم يكن موجوداً
    try {
        $db->query("ALTER TABLE students ADD COLUMN photo_path VARCHAR(255) NULL AFTER graduation_date");
    } catch (Exception $e2) {
        // تجاهل الخطأ إذا كان الحقل موجوداً بالفعل
    }
}

// جلب صورة الطالب
$student_photo_path = isset($student['photo_path']) && !empty($student['photo_path']) ? $student['photo_path'] : null;

// جلب جميع الفصول الدراسية للطالب مرتبة
$semesters_query = "
    SELECT DISTINCT s.*, sg.gpa, sg.total_credit_hours, sg.total_points
     FROM semesters s
     JOIN student_courses sc ON s.id = sc.semester_id
     LEFT JOIN semester_gpas sg ON s.id = sg.semester_id AND sg.student_id = ?
     WHERE sc.student_id = ?";

$semesters_params = [$student_id, $student_id];

// إذا تم اختيار فصل محدد، عرضه فقط
if ($selected_semester_id > 0) {
    $semesters_query .= " AND s.id = ?";
    $semesters_params[] = $selected_semester_id;
}

$semesters_query .= " ORDER BY s.academic_year, s.semester_number";

$semesters = $db->fetchAll($semesters_query, $semesters_params);

// تأكد من تحديث وحساب المعدلات الفصلية قبل العرض
if (!empty($semesters)) {
    foreach ($semesters as &$semester) {
        $gpa_result = $gradeCalculator->calculateSemesterGPA($student_id, $semester['id']);
        if ($gpa_result) {
            $semester['gpa'] = $gpa_result['gpa'];
            $semester['total_credit_hours'] = $gpa_result['total_credit_hours'];
            $semester['total_points'] = $gpa_result['total_points'];
        } elseif (isset($semester['gpa']) && $semester['gpa'] !== null) {
            $semester['gpa'] = round((float)$semester['gpa'], 2);
        } else {
            $semester['gpa'] = null;
        }
    }
    unset($semester);
}

// جلب جميع الفصول المتاحة للاختيار
$all_semesters = $db->fetchAll(
    "SELECT DISTINCT s.*
     FROM semesters s
     JOIN student_courses sc ON s.id = sc.semester_id
     WHERE sc.student_id = ?
     ORDER BY s.academic_year, s.semester_number",
    [$student_id]
);

// جلب المعدل التراكمي النهائي للطالب
$final_cgpa = $db->fetchOne(
    "SELECT * FROM cumulative_gpas WHERE student_id = ?",
    [$student_id]
);

// حساب التقدير النهائي
$final_grade_text = '-';
if ($final_cgpa && $final_cgpa['cgpa'] > 0) {
    $final_grade_text = getGradeText($final_cgpa['cgpa']);
}

// دالة حساب حالة الطالب (من semester_results.php)
function calculateStudentStatus($student_grades, $courses_list, $total_semester_credits, $is_supplementary_student = false) {
    $failed_credits = 0;
    $failed_courses = [];
    $no_grade_courses = [];
    
    foreach ($courses_list as $course) {
        $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
        
        if ($grade) {
            $letter_grade = isset($grade['letter_grade']) ? trim($grade['letter_grade']) : '';
            $marks = isset($grade['marks']) ? $grade['marks'] : null;
            
            $is_failed = false;
            
            if ($marks !== null && !is_numeric($marks)) {
                $marks_text = trim($marks);
                if ($marks_text == 'ب') $marks_text = 'بد';
                
                if ($marks_text == 'بد') {
                    $is_failed = false;
                    $no_grade_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code']];
                } elseif ($marks_text == 'غ' || $marks_text == 'غياب' || $marks_text == 'غش' || $marks_text == 'غشش') {
                    $is_failed = true;
                } elseif ($letter_grade) {
                    $letter_grade_trimmed = trim($letter_grade);
                    if ($letter_grade_trimmed == 'ب') $letter_grade_trimmed = 'بد';
                    
                    if ($letter_grade_trimmed == 'بد') {
                        $is_failed = false;
                        $no_grade_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code']];
                    } elseif ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' || 
                              $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش' ||
                              $letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                        $is_failed = true;
                    }
                }
            } elseif ($letter_grade) {
                $letter_grade_trimmed = trim($letter_grade);
                if ($letter_grade_trimmed == 'ب') $letter_grade_trimmed = 'بد';
                
                if ($letter_grade_trimmed == 'بد') {
                    $is_failed = false;
                    $no_grade_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code']];
                } elseif ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب' || 
                          $letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش' ||
                          $letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                    $is_failed = true;
                }
            } elseif ($marks !== null && is_numeric($marks)) {
                $is_failed = ((float)$marks < 50);
            }
            
            if ($is_failed) {
                $failed_credits += (float)$course['credit_hours'];
                $failed_courses[] = ['name_ar' => $course['name_ar'], 'code' => $course['code'], 'credits' => (float)$course['credit_hours']];
            }
        }
    }
    
    $status = 'ناجح';
    
    if ($failed_credits > 0) {
        $one_third = $total_semester_credits / 3;
        $two_thirds = ($total_semester_credits * 2) / 3;
        $one_sixth = $total_semester_credits / 6;
        
        if ($is_supplementary_student) {
            if ($failed_credits > $one_sixth) {
                $status = 'إعادة الفصل';
            }
        } else {
            if ($failed_credits > $two_thirds) {
                $status = 'فصل';
            } elseif ($failed_credits > $one_third) {
                $status = 'إعادة الفصل';
            }
        }
    }
    
    return [
        'status' => $status,
        'failed_credits' => $failed_credits,
        'failed_courses' => $failed_courses,
        'no_grade_courses' => $no_grade_courses
    ];
}

// جلب جميع المواد والدرجات لكل فصل وحساب الحالة
$courses_by_semester = [];
$semester_statuses = []; // لتخزين حالة كل فصل
$semester_failed_credits = []; // لتخزين الساعات الراسبة لكل فصل

foreach ($semesters as $semester) {
    $courses = $db->fetchAll(
        "SELECT c.id, c.code, c.name_ar, c.name_en, c.credit_hours, g.letter_grade, g.marks, g.points
         FROM student_courses sc
         JOIN courses c ON sc.course_id = c.id
         LEFT JOIN grades g ON sc.id = g.student_course_id
         WHERE sc.student_id = ? AND sc.semester_id = ?
         ORDER BY c.code",
        [$student_id, $semester['id']]
    );
    
    $courses_by_semester[$semester['id']] = $courses;
    
    // حساب إجمالي الساعات المعتمدة للفصل
    $total_credits = 0;
    foreach ($courses as $course) {
        $total_credits += (float)$course['credit_hours'];
    }
    
    // تحضير بيانات الدرجات لحساب الحالة
    $student_grades = [];
    foreach ($courses as $course) {
        if ($course['marks'] !== null || $course['letter_grade'] !== null) {
            $student_grades[$course['id']] = [
                'marks' => $course['marks'],
                'letter_grade' => $course['letter_grade'],
                'points' => $course['points']
            ];
        }
    }
    
    // حساب حالة الطالب في هذا الفصل
    // ملاحظة: قرارات الإعادة والفصل تظهر فقط في الفصول الزوجية (الثاني، الرابع...) لأن النتيجة سنوية
    $status_data = calculateStudentStatus($student_grades, $courses, $total_credits, false);
    $semester_status = $status_data['status'];
    $semester_failed_credits[$semester['id']] = $status_data['failed_credits'];
    
    // إذا كان الفصل فردي (1، 3، 5...)، لا نعرض قرارات الإعادة أو الفصل
    if ($semester['semester_number'] % 2 == 1) {
        // الفصل الفردي: نحذف قرارات الإعادة والفصل
        if ($semester_status == 'إعادة الفصل' || $semester_status == 'فصل') {
            $semester_status = 'ناجح'; // أو يمكن تركه فارغاً
        }
    }
    
    $semester_statuses[$semester['id']] = $semester_status;
}

// حساب المعدل التراكمي
$cumulative_gpas = [];
$total_points = 0;
$total_credit_hours = 0;

// إذا كان هناك فصل محدد، نحسب المعدل التراكمي حتى ذلك الفصل
if ($selected_semester_id > 0 && !empty($semesters)) {
    // جلب معلومات الفصل المحدد
    $selected_semester = $semesters[0];
    
    // جلب جميع الفصول حتى الفصل المحدد (بما فيه)
    $all_semesters_until = $db->fetchAll(
        "SELECT DISTINCT s.*, sg.gpa, sg.total_credit_hours, sg.total_points
         FROM semesters s
         JOIN student_courses sc ON s.id = sc.semester_id
         LEFT JOIN semester_gpas sg ON s.id = sg.semester_id AND sg.student_id = ?
         WHERE sc.student_id = ?
         ORDER BY s.academic_year, s.semester_number",
        [$student_id, $student_id]
    );
    
    // حساب المعدل التراكمي حتى الفصل المحدد
    foreach ($all_semesters_until as $sem) {
        // التوقف عند الوصول للفصل المحدد
        if ($sem['id'] == $selected_semester_id) {
            if ($sem['gpa'] !== null && $sem['total_points'] !== null) {
                $total_points += $sem['total_points'];
                $total_credit_hours += $sem['total_credit_hours'];
            }
            break;
        }
        
        if ($sem['gpa'] !== null && $sem['total_points'] !== null) {
            $total_points += $sem['total_points'];
            $total_credit_hours += $sem['total_credit_hours'];
        }
    }
    
    // حساب المعدل التراكمي للفصل المحدد
    // الفصل الأول: المعدل التراكمي = المعدل الفصلي (نعرضه حتى لو كان 0 أو بدون ساعات)
    if ($selected_semester['semester_number'] == 1) {
        $cumulative_gpa = $selected_semester['gpa'] !== null ? (float)$selected_semester['gpa'] : 0;
        $cumulative_gpas[$selected_semester['id']] = [
            'cgpa' => round($cumulative_gpa, 2),
            'semester_number' => 1,
            'academic_year' => $selected_semester['academic_year']
        ];
    } elseif ($total_credit_hours > 0) {
        // للفصول الأخرى: المعدل التراكمي = مجموع النقاط / مجموع الساعات
        $cumulative_gpa = $total_points / $total_credit_hours;
        $cumulative_gpas[$selected_semester['id']] = [
            'cgpa' => round($cumulative_gpa, 2),
            'semester_number' => $selected_semester['semester_number'],
            'academic_year' => $selected_semester['academic_year']
        ];
    }
} else {
    // حساب المعدل التراكمي لكل فصل
    foreach ($semesters as $index => $semester) {
        if ($semester['gpa'] !== null && $semester['total_points'] !== null) {
            $total_points += $semester['total_points'];
            $total_credit_hours += $semester['total_credit_hours'];
        }
        
        // الفصل الأول: نعرض المعدل التراكمي دائماً (= المعدل الفصلي أو 0)
        if ($semester['semester_number'] == 1) {
            $cumulative_gpa = $semester['gpa'] !== null ? (float)$semester['gpa'] : 0;
            $cumulative_gpas[$semester['id']] = [
                'cgpa' => round($cumulative_gpa, 2),
                'semester_number' => 1,
                'academic_year' => $semester['academic_year']
            ];
        } elseif ($total_credit_hours > 0) {
            // للفصول الأخرى: المعدل التراكمي = مجموع النقاط / مجموع الساعات
            $cumulative_gpa = $total_points / $total_credit_hours;
            $cumulative_gpas[$semester['id']] = [
                'cgpa' => round($cumulative_gpa, 2),
                'semester_number' => $semester['semester_number'],
                'academic_year' => $semester['academic_year']
            ];
        }
    }
}

// دالة للحصول على التقدير من المعدل (المعدل من 5)
function getGradeText($gpa, $lang = 'ar', $failed_credits = 0) {
    if ($lang == 'en') {
        if ($gpa >= 4 && $gpa <= 5) return 'Excellent';
        if ($gpa >= 3.5 && $gpa < 4) return 'Very Good';
        if ($gpa >= 3 && $gpa < 3.5) return 'Good';
        if ($gpa >= 2.5 && $gpa < 3) return 'Acceptable';
        if ($gpa >= 1.5 && $gpa < 2.5) return 'Repeating Year';
        if ($gpa < 1.5) return 'Dismissed';
        return 'Acceptable';
    } else {
        // إذا كان المعدل أقل من 1.5: فصل
        if ($gpa < 1.5) {
            return 'فصل';
        }
        // إذا كان المعدل >= 1.5 و < 2.5: إعادة السنة
        elseif ($gpa >= 1.5 && $gpa < 2.5) {
            return 'إعادة السنة';
        }
        // إذا كان المعدل >= 2.5 و < 3: مقبول
        elseif ($gpa >= 2.5 && $gpa < 3) {
            // إذا رسب في مواد: ملاحق
            if ($failed_credits > 0) {
                return 'ملاحق';
            }
            return 'مقبول';
        }
        // إذا كان المعدل >= 3 و < 3.5: جيد
        elseif ($gpa >= 3 && $gpa < 3.5) {
            // إذا رسب في مواد: ملاحق
            if ($failed_credits > 0) {
                return 'ملاحق';
            }
            return 'جيد';
        }
        // إذا كان المعدل >= 3.5 و < 4: جيد جداً
        elseif ($gpa >= 3.5 && $gpa < 4) {
            // إذا رسب في مواد: ملاحق
            if ($failed_credits > 0) {
                return 'ملاحق';
            }
            return 'جيد جداً';
        }
        // إذا كان المعدل >= 4 و <= 5: ممتاز
        elseif ($gpa >= 4 && $gpa <= 5) {
            return 'ممتاز';
        }
        // حالات أخرى
        else {
            return 'مقبول';
        }
    }
}

// دالة لتحويل رقم الفصل إلى نص
function getSemesterNumberText($number, $lang = 'ar') {
    if ($lang == 'en') {
        return $number;
    } else {
        $numbers = [
            1 => 'الأول',
            2 => 'الثاني',
            3 => 'الثالث',
            4 => 'الرابع',
            5 => 'الخامس',
            6 => 'السادس',
            7 => 'السابع',
            8 => 'الثامن',
            9 => 'التاسع',
            10 => 'العاشر'
        ];
        return $numbers[$number] ?? $number;
    }
}

// جلب أسماء وأوصاف المسؤولين للتوقيعات من قاعدة البيانات
try {
    $registrar_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'registrar_name'");
    $registrar_title_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'registrar_title'");
    $academic_affairs_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'academic_affairs_name'");
    $academic_affairs_title_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'academic_affairs_title'");
    $dean_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'dean_name'");
    $dean_title_setting = $db->fetchOne("SELECT setting_value FROM settings WHERE setting_key = 'dean_title'");
    
    $registrar_name = $registrar_setting ? $registrar_setting['setting_value'] : 'مسجل الكلية';
    $registrar_title = $registrar_title_setting ? $registrar_title_setting['setting_value'] : 'مسجل الكلية';
    $academic_affairs_name = $academic_affairs_setting ? $academic_affairs_setting['setting_value'] : 'أمين الشؤون العلمية';
    $academic_affairs_title = $academic_affairs_title_setting ? $academic_affairs_title_setting['setting_value'] : 'أمين الشؤون العلمية';
    $dean_name = $dean_setting ? $dean_setting['setting_value'] : 'عميد الكلية';
    $dean_title = $dean_title_setting ? $dean_title_setting['setting_value'] : 'عميد الكلية';
} catch (Exception $e) {
    // في حالة عدم وجود جدول settings، استخدم القيم الافتراضية
    $registrar_name = 'مسجل الكلية';
    $registrar_title = 'مسجل الكلية';
    $academic_affairs_name = 'أمين الشؤون العلمية';
    $academic_affairs_title = 'أمين الشؤون العلمية';
    $dean_name = 'عميد الكلية';
    $dean_title = 'عميد الكلية';
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang == 'en' ? 'en' : 'ar'; ?>" dir="<?php echo $lang == 'en' ? 'ltr' : 'rtl'; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $lang == 'en' ? 'Detailed Certificate' : 'شهادة التفاصيل'; ?> - <?php echo htmlspecialchars($student['full_name_ar']); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        @media print {
            body {
                margin: 0;
                padding: 0;
            }
            .no-print {
                display: none;
            }
            .certificate-container {
                box-shadow: none;
                border: none;
                max-width: 100%;
                padding: 15mm;
            }
            
            /* تقسيم المحتوى إلى صفحتين */
            .page-break {
                page-break-after: always;
                break-after: page;
            }
            
            .page-1 {
                min-height: 250mm; /* ارتفاع الصفحة A4 */
                max-height: 270mm;
            }
            
            .page-2 {
                min-height: 250mm;
                max-height: 270mm;
                page-break-before: always;
                break-before: page;
            }
            
            .page-break {
                display: none;
            }
            
            /* منع تقسيم العناصر */
            .student-info,
            .semesters-pair-container {
                page-break-inside: avoid;
                break-inside: avoid;
            }
            
            .signatures {
                page-break-inside: avoid;
                break-inside: avoid;
                margin-top: 40px;
            }
            
            @page {
                size: A4;
                margin: 15mm;
            }
        }
        
        .certificate-container {
            max-width: 210mm;
            margin: 20px auto;
            padding: 30px;
            background: white;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            border: 3px solid #667eea;
        }
        
        .certificate-header {
            text-align: center;
            border-bottom: 3px solid #667eea;
            padding-bottom: 20px;
            margin-bottom: 30px;
            position: relative;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .certificate-header-content {
            text-align: center;
            flex: 1;
        }
        
        .certificate-header h1 {
            color: #667eea;
            font-size: 28px;
            margin: 10px 0;
        }
        
        .certificate-header h2 {
            color: #333;
            font-size: 20px;
            margin: 5px 0;
        }
        
        .logo-frame {
            width: 100px;
            height: 100px;
            border: 3px solid #667eea;
            border-radius: 8px;
            padding: 5px;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }
        
        .logo-frame img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            display: block;
        }
        
        .student-photo-frame {
            width: 100px;
            height: 120px;
            border: 3px solid #667eea;
            border-radius: 8px;
            padding: 5px;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }
        
        .student-photo-frame img {
            max-width: 100%;
            max-height: 100%;
            object-fit: cover;
        }
        
        .empty-frame {
            background: #f5f5f5;
            color: #999;
            font-size: 10px;
            text-align: center;
            display: flex;
            align-items: center;
            justify-content: center;
            width: 100%;
            height: 100%;
        }
        
        .upload-overlay {
            position: relative;
        }
        
        .upload-form-inline {
            position: absolute;
            bottom: -45px;
            left: 50%;
            transform: translateX(-50%);
            width: auto;
            text-align: center;
            z-index: 1000;
            white-space: nowrap;
            background: rgba(255, 255, 255, 0.95);
            padding: 5px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
        }
        
        .upload-form-inline input[type="file"] {
            display: none;
        }
        
        .upload-form-inline label {
            display: inline-block;
            padding: 8px 15px;
            background: #667eea;
            color: white !important;
            border-radius: 5px;
            cursor: pointer;
            font-size: 12px;
            font-weight: bold;
            margin: 0;
            transition: background 0.3s;
            white-space: nowrap;
            border: none;
        }
        
        .upload-form-inline label:hover {
            background: #5568d3;
        }
        
        .logo-frame,
        .student-photo-frame {
            position: relative;
            margin-bottom: 50px;
        }
        
        .logo-frame .upload-overlay,
        .student-photo-frame .upload-overlay {
            width: 100%;
            height: 100%;
        }
        
        @media print {
            .upload-form-inline {
                display: none !important;
            }
            .logo-frame,
            .student-photo-frame {
                margin-bottom: 0;
            }
        }
        
        .student-info {
            margin: 30px 0;
            padding: 20px;
            background: #f9f9f9;
            border-right: 4px solid #667eea;
        }
        
        .student-info p {
            margin: 10px 0;
            font-size: 16px;
        }
        
        .student-info strong {
            color: #667eea;
            min-width: 150px;
            display: inline-block;
        }
        
        .semesters-table {
            width: 100%;
            border-collapse: collapse;
            margin: 30px 0;
        }
        
        .semesters-table th,
        .semesters-table td {
            padding: 12px;
            text-align: center;
            border: 1px solid #ddd;
        }
        
        .semesters-table th {
            background: #667eea;
            color: white;
            font-weight: bold;
        }
        
        .semesters-table tr:nth-child(even) {
            background: #f9f9f9;
        }
        
        .cumulative-row {
            background: #e3f2fd !important;
            font-weight: bold;
        }
        
        .semesters-pair-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin: 30px 0;
            page-break-inside: avoid;
        }
        
        .semester-details {
            border: 2px solid #667eea;
            border-radius: 8px;
            padding: 15px;
            background: #f9f9f9;
        }
        
        .semester-header {
            text-align: center;
            background: #667eea;
            color: white;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 15px;
            font-weight: bold;
        }
        
        .semester-courses-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 10px;
        }
        
        .semester-courses-table th,
        .semester-courses-table td {
            padding: 8px;
            text-align: center;
            border: 1px solid #ddd;
            font-size: 12px;
        }
        
        .semester-courses-table th {
            background: #667eea;
            color: white;
            font-weight: bold;
        }
        
        .semester-courses-table tr:nth-child(even) {
            background: #f0f0f0;
        }
        
        .semester-gpa-row {
            background: #e3f2fd !important;
            font-weight: bold;
        }
        
        .cumulative-gpa-box {
            grid-column: 1 / -1;
            text-align: center;
            background: #e3f2fd;
            padding: 15px;
            border-radius: 8px;
            margin-top: 10px;
            border: 2px solid #2196F3;
        }
        
        .cumulative-gpa-box strong {
            font-size: 16px;
            color: #2196F3;
        }
        
        @media print {
            .semesters-pair-container {
                page-break-inside: avoid;
            }
        }
        
        .signatures {
            display: flex;
            justify-content: space-around;
            margin-top: 60px;
            padding-top: 30px;
            border-top: 2px solid #ddd;
        }
        
        .signature-box {
            text-align: center;
            width: 30%;
        }
        
        .signature-line {
            border-top: 2px solid #333;
            width: 200px;
            margin: 50px auto 10px;
        }
        
        .signature-name {
            font-weight: bold;
            margin-top: 10px;
        }
        
        .signature-title {
            margin-top: 5px;
            color: #666;
            font-size: 14px;
        }
        
        .print-button {
            text-align: center;
            margin: 20px 0;
        }
        
        .certificate-number {
            text-align: left;
            margin-top: 20px;
            color: #666;
            font-size: 14px;
        }
        
        .grade-key-table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 11px;
        }
        
        .grade-key-table th,
        .grade-key-table td {
            padding: 6px 4px;
            text-align: center;
            border: 1px solid #ddd;
        }
        
        .grade-key-table th {
            background: #667eea;
            color: white;
            font-weight: bold;
            font-size: 11px;
        }
        
        .grade-key-table tbody tr:nth-child(even) {
            background: #f9f9f9;
        }
        
        .grade-key-table tbody tr:nth-child(odd) {
            background: white;
        }
        
        .grade-key-title {
            text-align: center;
            font-weight: bold;
            margin: 15px 0 10px 0;
            color: #667eea;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="no-print print-button">
            <form method="GET" style="display: inline-block; margin-left: 10px;">
                <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                <input type="hidden" name="lang" value="<?php echo $lang; ?>">
                <select name="semester_id" onchange="this.form.submit()" style="padding: 8px; margin-left: 10px;">
                    <option value=""><?php echo $lang == 'en' ? 'All Semesters' : 'جميع الفصول'; ?></option>
                    <?php foreach ($all_semesters as $sem): ?>
                        <option value="<?php echo $sem['id']; ?>" <?php echo $selected_semester_id == $sem['id'] ? 'selected' : ''; ?>>
                            <?php echo $lang == 'en' ? 'Semester' : 'الفصل'; ?> <?php echo getSemesterNumberText($sem['semester_number'], $lang); ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </form>
            <button onclick="window.print()" class="btn btn-primary" style="margin-right: 10px;">🖨️ <?php echo $lang == 'en' ? 'Print Certificate' : 'طباعة الشهادة'; ?></button>
            <a href="student_details.php?id=<?php echo $student_id; ?>" class="btn btn-secondary"><?php echo $lang == 'en' ? 'Back' : 'العودة'; ?></a>
        </div>
        
        <?php if (!empty($upload_message)): ?>
            <div class="alert alert-<?php echo $upload_message_type === 'success' ? 'success' : 'error'; ?>" style="margin: 20px 0;">
                <?php echo htmlspecialchars($upload_message); ?>
            </div>
        <?php endif; ?>
        
        <!-- أزرار رفع بديلة واضحة -->
        <div class="no-print" style="background: #f9f9f9; padding: 20px; margin: 20px 0; border-radius: 10px; border: 2px solid #667eea;">
            <h3 style="margin-top: 0; color: #667eea;">رفع الصور</h3>
            <div style="display: flex; gap: 20px; flex-wrap: wrap;">
                <?php if ($auth->hasRole('admin') || $auth->hasRole('college_admin')): ?>
                    <div style="flex: 1; min-width: 250px;">
                        <form method="POST" enctype="multipart/form-data" style="margin: 0;">
                            <input type="hidden" name="upload_type" value="logo">
                            <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                            <label for="logo-file-main" style="display: block; margin-bottom: 5px; font-weight: bold;">رفع شعار الجامعة:</label>
                            <input type="file" name="logo" id="logo-file-main" accept="image/jpeg,image/jpg,image/png,image/gif,image/svg+xml" style="margin-bottom: 10px; width: 100%; padding: 8px; border: 2px dashed #667eea; border-radius: 5px;">
                            <button type="submit" class="btn btn-primary" style="width: 100%;">📤 رفع الشعار</button>
                        </form>
                    </div>
                <?php endif; ?>
                
                <?php if ($auth->canAccessStudent($student_id) || $auth->hasRole('admin')): ?>
                    <div style="flex: 1; min-width: 250px;">
                        <form method="POST" enctype="multipart/form-data" style="margin: 0;">
                            <input type="hidden" name="upload_type" value="student_photo">
                            <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                            <label for="student-photo-file-main" style="display: block; margin-bottom: 5px; font-weight: bold;">رفع صورة الطالب:</label>
                            <input type="file" name="student_photo" id="student-photo-file-main" accept="image/jpeg,image/jpg,image/png,image/gif" style="margin-bottom: 10px; width: 100%; padding: 8px; border: 2px dashed #667eea; border-radius: 5px;">
                            <button type="submit" class="btn btn-primary" style="width: 100%;">📤 رفع صورة الطالب</button>
                        </form>
                    </div>
                <?php endif; ?>
            </div>
            <small style="display: block; margin-top: 10px; color: #666;">
                ملاحظة: يمكنك أيضاً استخدام الأزرار الموجودة داخل الأطر في الشهادة أدناه
            </small>
        </div>
        
        <div class="certificate-container">
            <!-- الصفحة الأولى -->
            <div class="page-1">
                <!-- رأس الشهادة -->
                <div class="certificate-header">
                    <!-- إطار شعار الجامعة (يسار) -->
                    <div class="logo-frame upload-overlay">
                        <?php 
                        if ($university_logo_path && !empty($university_logo_path)): 
                            // تنظيف المسار للعرض في HTML
                            $logo_src = str_replace('\\', '/', $university_logo_path);
                            $logo_src = ltrim($logo_src, '/');
                            
                            // التحقق النهائي من وجود الملف
                            $logo_full_path = __DIR__ . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $logo_src);
                        ?>
                            <?php if (file_exists($logo_full_path) && is_file($logo_full_path) && filesize($logo_full_path) > 0): ?>
                                <img src="<?php echo htmlspecialchars($logo_src); ?>" alt="شعار الجامعة" style="width: 100%; height: 100%; object-fit: contain; display: block;">
                            <?php else: ?>
                                <div class="empty-frame">شعار الجامعة</div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="empty-frame">شعار الجامعة</div>
                        <?php endif; ?>
                        
                        <?php if ($auth->hasRole('admin') || $auth->hasRole('college_admin')): ?>
                            <div class="upload-form-inline no-print">
                                <form method="POST" enctype="multipart/form-data" id="logo-upload-form">
                                    <input type="hidden" name="upload_type" value="logo">
                                    <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                                    <label for="logo-file">📷 اضغط لرفع شعار</label>
                                    <input type="file" name="logo" id="logo-file" accept="image/jpeg,image/jpg,image/png,image/gif,image/svg+xml" onchange="document.getElementById('logo-upload-form').submit();">
                                </form>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- محتوى العنوان -->
                    <div class="certificate-header-content" style="flex: 1;">
                        <h1 style="font-size: 18px;"><?php echo $lang == 'en' ? 'Alsharg University of Science and Technology' : 'جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology'; ?></h1>
                        <h2><?php echo $lang == 'en' ? 'Detailed Certificate' : 'شهادة التفاصيل'; ?></h2>
                    </div>
                    
                    <!-- إطار صورة الطالب (يمين) -->
                    <div class="student-photo-frame upload-overlay">
                        <?php if ($student_photo_path && file_exists(__DIR__ . '/' . $student_photo_path)): ?>
                            <img src="<?php echo htmlspecialchars($student_photo_path); ?>" alt="صورة الطالب">
                        <?php else: ?>
                            <div class="empty-frame">صورة الطالب</div>
                        <?php endif; ?>
                        
                        <?php if ($auth->canAccessStudent($student_id) || $auth->hasRole('admin')): ?>
                            <div class="upload-form-inline no-print">
                                <form method="POST" enctype="multipart/form-data" id="student-photo-upload-form">
                                    <input type="hidden" name="upload_type" value="student_photo">
                                    <input type="hidden" name="student_id" value="<?php echo $student_id; ?>">
                                    <label for="student-photo-file">📷 اضغط لرفع صورة</label>
                                    <input type="file" name="student_photo" id="student-photo-file" accept="image/jpeg,image/jpg,image/png,image/gif" onchange="document.getElementById('student-photo-upload-form').submit();">
                                </form>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- معلومات الطالب -->
                <div class="student-info">
                    <p><strong><?php echo $lang == 'en' ? 'Student Name:' : 'اسم الطالب:'; ?></strong> <?php echo $lang == 'en' ? htmlspecialchars($student['full_name_en'] ?: $student['full_name_ar']) : htmlspecialchars($student['full_name_ar']); ?></p>
                    <p><strong><?php echo $lang == 'en' ? 'Student Number:' : 'الرقم الجامعي:'; ?></strong> <?php echo htmlspecialchars($student['student_number']); ?></p>
                    <p><strong><?php echo $lang == 'en' ? 'Major:' : 'التخصص:'; ?></strong> <?php echo htmlspecialchars($student['major_name']); ?></p>
                    <p><strong><?php echo $lang == 'en' ? 'College:' : 'الكلية:'; ?></strong> <?php echo htmlspecialchars($student['college_name']); ?></p>
                    <?php if ($final_cgpa && $final_cgpa['cgpa'] > 0): ?>
                        <p><strong><?php echo $lang == 'en' ? 'Final GPA:' : 'المعدل النهائي:'; ?></strong> <?php echo number_format($final_cgpa['cgpa'], 2); ?></p>
                        <p><strong><?php echo $lang == 'en' ? 'Final Grade:' : 'التقدير النهائي:'; ?></strong> <?php echo getGradeText($final_cgpa['cgpa'], $lang); ?></p>
                    <?php endif; ?>
                    <p><strong><?php echo $lang == 'en' ? 'Issue Date:' : 'تاريخ الإصدار:'; ?></strong> <?php echo date('Y-m-d'); ?></p>
                </div>
                
                <!-- عرض الفصول في عمودين (فصلين جنباً إلى جنب) -->
            <?php 
            // تجميع الفصول في أزواج
            $semester_pairs = [];
            for ($i = 0; $i < count($semesters); $i += 2) {
                $pair = [
                    'first' => $semesters[$i] ?? null,
                    'second' => isset($semesters[$i + 1]) ? $semesters[$i + 1] : null
                ];
                $semester_pairs[] = $pair;
            }
            
            // تقسيم الأزواج إلى صفحتين: الصفحة الأولى 4 فصول (زوجين)، الصفحة الثانية البقية
            $total_pairs = count($semester_pairs);
            $pairs_for_page1 = 2; // زوجين = 4 فصول في الصفحة الأولى
            
            foreach ($semester_pairs as $pair_index => $pair):
                // إضافة page-break بعد أول زوجين (4 فصول)
                if ($pair_index == $pairs_for_page1 && $pair_index > 0):
            ?>
                    </div> <!-- نهاية الصفحة الأولى -->
                    <div class="page-break"></div>
                    <div class="page-2">
            <?php 
                endif; 
                $first_semester = $pair['first'];
                $second_semester = $pair['second'];
                
                // حساب عدد المواد لكل فصل
                $first_courses_count = 0;
                $second_courses_count = 0;
                if ($first_semester) {
                    $first_courses_temp = $courses_by_semester[$first_semester['id']] ?? [];
                    $first_courses_count = count($first_courses_temp);
                }
                if ($second_semester) {
                    $second_courses_temp = $courses_by_semester[$second_semester['id']] ?? [];
                    $second_courses_count = count($second_courses_temp);
                }
                
                // حساب عدد الصفوف الناقصة لكل فصل
                $first_missing_rows = 0;
                $second_missing_rows = 0;
                if ($first_semester && $second_semester) {
                    if ($first_courses_count < $second_courses_count) {
                        $first_missing_rows = $second_courses_count - $first_courses_count;
                    } elseif ($second_courses_count < $first_courses_count) {
                        $second_missing_rows = $first_courses_count - $second_courses_count;
                    }
                }
                
                // حساب المعدل التراكمي لهذا الزوج
                $pair_cumulative = '-';
                if ($second_semester && isset($cumulative_gpas[$second_semester['id']])) {
                    $pair_cumulative = $cumulative_gpas[$second_semester['id']]['cgpa'];
                } elseif ($first_semester && isset($cumulative_gpas[$first_semester['id']])) {
                    $pair_cumulative = $cumulative_gpas[$first_semester['id']]['cgpa'];
                }
            ?>
                <div class="semesters-pair-container">
                    <!-- الفصل الأول -->
                    <?php if ($first_semester): 
                        $first_courses = $courses_by_semester[$first_semester['id']] ?? [];
                        $first_gpa = $first_semester['gpa'] ?? 0;
                        $first_failed_credits = $semester_failed_credits[$first_semester['id']] ?? 0;
                    ?>
                        <div class="semester-details">
                            <div class="semester-header">
                                <?php echo $lang == 'en' ? 'Semester' : 'الفصل'; ?> <?php echo $lang == 'en' ? $first_semester['semester_number'] : getSemesterNumberText($first_semester['semester_number'], $lang); ?> - <?php echo htmlspecialchars($first_semester['academic_year']); ?>
                                <?php 
                                // عرض قرارات الإعادة أو الفصل فقط في الفصول الزوجية (الثاني، الرابع، السادس...)
                                if ($first_semester['semester_number'] % 2 == 0 && 
                                    isset($semester_statuses[$first_semester['id']]) && 
                                    ($semester_statuses[$first_semester['id']] == 'إعادة الفصل' || 
                                     $semester_statuses[$first_semester['id']] == 'فصل')): ?>
                                    <span style="color: #f44336; font-weight: bold; margin-right: 10px;">(<?php echo htmlspecialchars($semester_statuses[$first_semester['id']]); ?>)</span>
                                <?php endif; ?>
                            </div>
                            <table class="semester-courses-table">
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th><?php echo $lang == 'en' ? 'Course Name' : 'اسم المادة'; ?></th>
                                        <th><?php echo $lang == 'en' ? 'Credit Hours' : 'الساعات'; ?></th>
                                        <th><?php echo $lang == 'en' ? 'Grade' : 'التقدير'; ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php $course_counter = 1; ?>
                                    <?php foreach ($first_courses as $course): ?>
                                        <tr>
                                            <td><?php echo $course_counter++; ?></td>
                                            <td><?php echo $lang == 'en' ? htmlspecialchars($course['name_en'] ?: $course['name_ar']) : htmlspecialchars($course['name_ar']); ?></td>
                                            <td><?php echo $course['credit_hours']; ?></td>
                                            <td><?php echo htmlspecialchars(cleanLetterGrade($course['letter_grade'] ?? '-')); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                    <?php 
                                    // إضافة الصفوف الناقصة للفصل الأول مع تظليلها
                                    if ($first_semester && $second_semester && $first_missing_rows > 0): 
                                        for ($i = 0; $i < $first_missing_rows; $i++): 
                                    ?>
                                        <tr style="background-color: #e8e8e8;">
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                        </tr>
                                    <?php 
                                        endfor;
                                    endif; 
                                    ?>
                                    <tr class="semester-gpa-row">
                                        <td colspan="2"><strong><?php echo $lang == 'en' ? 'Semester GPA' : 'المعدل الفصلي'; ?></strong></td>
                                        <td>
                                            <strong><?php echo $first_gpa !== null ? number_format((float)$first_gpa, 2) : '-'; ?></strong>
                                            <?php if ($first_gpa !== null): ?>
                                                (<?php echo getGradeText((float)$first_gpa, $lang, $first_failed_credits); ?>)
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (isset($cumulative_gpas[$first_semester['id']])): ?>
                                                <strong><?php echo $lang == 'en' ? 'Cumulative GPA:' : 'المعدل التراكمي:'; ?></strong> <?php echo number_format($cumulative_gpas[$first_semester['id']]['cgpa'], 2); ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="semester-details" style="visibility: hidden;"></div>
                    <?php endif; ?>
                    
                    <!-- الفصل الثاني -->
                    <?php if ($second_semester): 
                        $second_courses = $courses_by_semester[$second_semester['id']] ?? [];
                        $second_gpa = $second_semester['gpa'] ?? 0;
                        $second_failed_credits = $semester_failed_credits[$second_semester['id']] ?? 0;
                    ?>
                        <div class="semester-details">
                            <div class="semester-header">
                                <?php echo $lang == 'en' ? 'Semester' : 'الفصل'; ?> <?php echo $lang == 'en' ? $second_semester['semester_number'] : getSemesterNumberText($second_semester['semester_number'], $lang); ?> - <?php echo htmlspecialchars($second_semester['academic_year']); ?>
                                <?php 
                                // عرض قرارات الإعادة أو الفصل فقط في الفصول الزوجية (الثاني، الرابع، السادس...)
                                if ($second_semester['semester_number'] % 2 == 0 && 
                                    isset($semester_statuses[$second_semester['id']]) && 
                                    ($semester_statuses[$second_semester['id']] == 'إعادة الفصل' || 
                                     $semester_statuses[$second_semester['id']] == 'فصل')): ?>
                                    <span style="color: #f44336; font-weight: bold; margin-right: 10px;">(<?php echo htmlspecialchars($semester_statuses[$second_semester['id']]); ?>)</span>
                                <?php endif; ?>
                            </div>
                            <table class="semester-courses-table">
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th><?php echo $lang == 'en' ? 'Course Name' : 'اسم المادة'; ?></th>
                                        <th><?php echo $lang == 'en' ? 'Credit Hours' : 'الساعات'; ?></th>
                                        <th><?php echo $lang == 'en' ? 'Grade' : 'التقدير'; ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php $course_counter = 1; ?>
                                    <?php foreach ($second_courses as $course): ?>
                                        <tr>
                                            <td><?php echo $course_counter++; ?></td>
                                            <td><?php echo $lang == 'en' ? htmlspecialchars($course['name_en'] ?: $course['name_ar']) : htmlspecialchars($course['name_ar']); ?></td>
                                            <td><?php echo $course['credit_hours']; ?></td>
                                            <td><?php echo htmlspecialchars(cleanLetterGrade($course['letter_grade'] ?? '-')); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                    <?php 
                                    // إضافة الصفوف الناقصة للفصل الثاني مع تظليلها
                                    if ($first_semester && $second_semester && $second_missing_rows > 0): 
                                        for ($i = 0; $i < $second_missing_rows; $i++): 
                                    ?>
                                        <tr style="background-color: #e8e8e8;">
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                            <td>&nbsp;</td>
                                        </tr>
                                    <?php 
                                        endfor;
                                    endif; 
                                    ?>
                                    <tr class="semester-gpa-row">
                                        <td colspan="2"><strong><?php echo $lang == 'en' ? 'Semester GPA' : 'المعدل الفصلي'; ?></strong></td>
                                        <td>
                                            <strong><?php echo $second_gpa !== null ? number_format((float)$second_gpa, 2) : '-'; ?></strong>
                                            <?php if ($second_gpa !== null): ?>
                                                (<?php echo getGradeText((float)$second_gpa, $lang, $second_failed_credits); ?>)
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (isset($cumulative_gpas[$second_semester['id']])): ?>
                                                <strong><?php echo $lang == 'en' ? 'Cumulative GPA:' : 'المعدل التراكمي:'; ?></strong> <?php echo number_format($cumulative_gpas[$second_semester['id']]['cgpa'], 2); ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="semester-details" style="visibility: hidden;"></div>
                    <?php endif; ?>
                    
                </div>
            <?php endforeach; ?>
            
            <!-- مفتاح الدرجات في الصفحة الثانية -->
            <div class="grade-key-title"><?php echo $lang == 'en' ? 'Grade Key' : 'مفتاح الدرجات'; ?></div>
            <table class="grade-key-table">
                <thead>
                    <tr>
                        <th><?php echo $lang == 'en' ? 'Marks' : 'الدرجة'; ?></th>
                        <th><?php echo $lang == 'en' ? 'Grade' : 'التقدير'; ?></th>
                        <th><?php echo $lang == 'en' ? 'Marks' : 'الدرجة'; ?></th>
                        <th><?php echo $lang == 'en' ? 'Grade' : 'التقدير'; ?></th>
                        <th><?php echo $lang == 'en' ? 'Marks' : 'الدرجة'; ?></th>
                        <th><?php echo $lang == 'en' ? 'Grade' : 'التقدير'; ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>80 - 100</td>
                        <td><strong>A</strong></td>
                        <td>70 - 79</td>
                        <td><strong>B</strong></td>
                        <td>60 - 69</td>
                        <td><strong>C</strong></td>
                    </tr>
                    <tr>
                        <td>50 - 59</td>
                        <td><strong>D</strong></td>
                        <td><?php echo $lang == 'en' ? 'Less than 50' : 'أقل من 50'; ?></td>
                        <td><strong>F</strong></td>
                        <td></td>
                        <td></td>
                    </tr>
                </tbody>
            </table>
            
            <!-- التوقيعات في الصفحة الثانية -->
            <div class="signatures">
                <div class="signature-box">
                    <div class="signature-line"></div>
                    <div class="signature-name"><?php echo htmlspecialchars($registrar_name); ?></div>
                    <div class="signature-title"><?php echo htmlspecialchars($registrar_title); ?></div>
                </div>
                <div class="signature-box">
                    <div class="signature-line"></div>
                    <div class="signature-name"><?php echo htmlspecialchars($academic_affairs_name); ?></div>
                    <div class="signature-title"><?php echo htmlspecialchars($academic_affairs_title); ?></div>
                </div>
                <div class="signature-box">
                    <div class="signature-line"></div>
                    <div class="signature-name"><?php echo htmlspecialchars($dean_name); ?></div>
                    <div class="signature-title"><?php echo htmlspecialchars($dean_title); ?></div>
                </div>
            </div>
            
            <!-- رقم الشهادة -->
            <div class="certificate-number">
                <strong><?php echo $lang == 'en' ? 'Certificate Number:' : 'رقم الشهادة:'; ?></strong> DET-<?php echo date('Y'); ?>-<?php echo str_pad($student_id, 6, '0', STR_PAD_LEFT); ?>
            </div>
            </div> <!-- نهاية الصفحة الثانية -->
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
    
    <script>
        // تحسين رسائل الرفع
        document.addEventListener('DOMContentLoaded', function() {
            // رسالة تأكيد عند اختيار ملف
            const logoFileMain = document.getElementById('logo-file-main');
            const studentPhotoFileMain = document.getElementById('student-photo-file-main');
            const logoFile = document.getElementById('logo-file');
            const studentPhotoFile = document.getElementById('student-photo-file');
            
            if (logoFileMain) {
                logoFileMain.addEventListener('change', function(e) {
                    if (e.target.files.length > 0) {
                        console.log('تم اختيار ملف شعار:', e.target.files[0].name);
                    }
                });
            }
            
            if (studentPhotoFileMain) {
                studentPhotoFileMain.addEventListener('change', function(e) {
                    if (e.target.files.length > 0) {
                        console.log('تم اختيار ملف صورة طالب:', e.target.files[0].name);
                    }
                });
            }
            
            if (logoFile) {
                logoFile.addEventListener('change', function(e) {
                    if (e.target.files.length > 0) {
                        console.log('تم اختيار ملف شعار:', e.target.files[0].name);
                    }
                });
            }
            
            if (studentPhotoFile) {
                studentPhotoFile.addEventListener('change', function(e) {
                    if (e.target.files.length > 0) {
                        console.log('تم اختيار ملف صورة طالب:', e.target.files[0].name);
                    }
                });
            }
        });
    </script>
</body>
</html>

