<?php
/**
 * صفحة مبسطة لإدخال الدرجات
 * Simplified Grade Entry Page
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('enter_grades') && !$auth->hasPermission('manage_grades'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();
$message = '';

// جلب الفصول والمواد
$semesters = [];
$courses = [];
$students = [];

if ($auth->hasRole('admin')) {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name, m.college_id 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         ORDER BY s.academic_year DESC, s.semester_number DESC"
    );
} else {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name, m.college_id 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE m.college_id = ?
         ORDER BY s.academic_year DESC, s.semester_number DESC",
        [$_SESSION['college_id']]
    );
}

// معالجة اختيار الفصل
$selected_semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$selected_course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;

if ($selected_semester_id) {
    $semester = $db->fetchOne("SELECT * FROM semesters WHERE id = ?", [$selected_semester_id]);
    if ($semester && ($auth->belongsToCollege($db->fetchOne("SELECT college_id FROM majors WHERE id = ?", [$semester['major_id']])['college_id']) || $auth->hasRole('admin'))) {
        $courses = $db->fetchAll(
            "SELECT c.* FROM courses c 
             WHERE c.major_id = ? AND (c.semester_id = ? OR c.semester_id IS NULL)
             ORDER BY c.code",
            [$semester['major_id'], $selected_semester_id]
        );
    }
}

// معالجة إدخال الدرجات
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_grade') {
    $student_course_id = (int)($_POST['student_course_id'] ?? 0);
    $marks_input = trim($_POST['marks'] ?? '');
    
    if ($student_course_id && $marks_input !== '') {
        // معالجة الدرجة (رقمية أو نصية)
        $grade_data = $gradeCalculator->processGradeInput($marks_input);
        
        if (!$grade_data) {
            $message = '<div class="alert alert-error">✗ يرجى إدخال درجة صحيحة</div>';
        } else {
            // التحقق من نطاق الدرجة الرقمية
            if (!$grade_data['is_text'] && ($grade_data['marks'] < 0 || $grade_data['marks'] > 100)) {
                $message = '<div class="alert alert-error">✗ الدرجة يجب أن تكون بين 0 و 100</div>';
            } else {
                $student_course = $db->fetchOne(
                    "SELECT sc.*, s.college_id FROM student_courses sc 
                     JOIN students s ON sc.student_id = s.id 
                     WHERE sc.id = ?",
                    [$student_course_id]
                );
                
                if ($student_course && ($auth->belongsToCollege($student_course['college_id']) || $auth->hasRole('admin'))) {
                    try {
                        // إعداد القيم للحفظ
                        // إذا كانت الدرجة نصية، احفظ القيمة النصية في marks، وإلا احفظ الرقم
                        if ($grade_data['is_text']) {
                            $marks = $marks_input; // حفظ القيمة النصية الأصلية
                        } else {
                            $marks = $grade_data['marks'] !== null ? $grade_data['marks'] : 0;
                        }
                        $letter_grade = $grade_data['letter_grade'];
                        // حفظ points كما هي (null للدرجات النصية بدون نقاط مثل "بد")
                        $points = $grade_data['points'];
                        
                        $existing = $db->fetchOne("SELECT * FROM grades WHERE student_course_id = ?", [$student_course_id]);
                        
                        if ($existing) {
                            $db->query(
                                "UPDATE grades SET marks = ?, letter_grade = ?, points = ?, updated_at = CURRENT_TIMESTAMP, entered_by = ?
                                 WHERE student_course_id = ?",
                                [$marks, $letter_grade, $points, $_SESSION['user_id'], $student_course_id]
                            );
                        } else {
                            $db->query(
                                "INSERT INTO grades (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                                [
                                    $student_course_id,
                                    $student_course['student_id'],
                                    $student_course['course_id'],
                                    $student_course['semester_id'],
                                    $marks,
                                    $letter_grade,
                                    $points,
                                    $_SESSION['user_id']
                                ]
                            );
                        }
                        
                        $gradeCalculator->updateAllGPAs($student_course['student_id']);
                        $points_text = $points !== null ? ' (' . $points . ' نقاط)' : '';
                        $message = '<div class="alert alert-success">✓ تم حفظ الدرجة بنجاح! التقدير: ' . htmlspecialchars(cleanLetterGrade($letter_grade)) . $points_text . '</div>';
                    } catch (PDOException $e) {
                        $message = '<div class="alert alert-error">✗ خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
                    }
                } else {
                    $message = '<div class="alert alert-error">✗ ليس لديك صلاحية لإدخال الدرجة</div>';
                }
            }
        }
    } else {
        $message = '<div class="alert alert-error">✗ يرجى إدخال درجة صحيحة</div>';
    }
}

// جلب الطلاب المسجلين في المادة
$grades_data = [];
$selected_course = null;

if ($selected_course_id && $selected_semester_id) {
    $selected_course = $db->fetchOne("SELECT * FROM courses WHERE id = ?", [$selected_course_id]);
    
    if ($selected_course) {
        $grades_data = $db->fetchAll(
            "SELECT sc.*, s.student_number, s.full_name_ar, g.marks, g.letter_grade, g.points, g.id as grade_id
             FROM student_courses sc
             JOIN students s ON sc.student_id = s.id
             LEFT JOIN grades g ON sc.id = g.student_course_id
             WHERE sc.course_id = ? AND sc.semester_id = ?
             ORDER BY s.student_number",
            [$selected_course_id, $selected_semester_id]
        );
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدخال الدرجات - طريقة مبسطة</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .steps-box {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-right: 5px solid #2196F3;
        }
        .steps-box ol {
            margin-right: 20px;
        }
        .steps-box li {
            margin-bottom: 10px;
            font-size: 1.1em;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدخال الدرجات - طريقة مبسطة</h2>
            <div>
                <a href="upload_grades.php" class="btn btn-primary" style="margin-left: 10px;">📤 رفع الدرجات من Excel</a>
                <a href="grades.php" class="btn btn-secondary">الطريقة العادية</a>
            </div>
        </div>
        
        <div class="steps-box">
            <h3>📋 خطوات إدخال الدرجات:</h3>
            <ol>
                <li><strong>اختر الفصل الدراسي</strong> من القائمة أدناه</li>
                <li><strong>اختر المادة</strong> التي تريد إدخال درجاتها</li>
                <li><strong>ستظهر قائمة الطلاب</strong> المسجلين في هذه المادة</li>
                <li><strong>أدخل الدرجة</strong> لكل طالب (من 0 إلى 100)</li>
                <li><strong>اضغط "حفظ"</strong> - سيتم حساب التقدير تلقائياً</li>
            </ol>
        </div>
        
        <?php echo $message; ?>
        
        <div class="card">
            <div class="card-header">
                <h3>الخطوة 1: اختر الفصل الدراسي</h3>
            </div>
            <form method="GET" style="display:inline;">
                <div class="form-group">
                    <label>الفصل الدراسي</label>
                    <select name="semester_id" onchange="this.form.submit()" style="width:100%; padding:10px;">
                        <option value="">-- اختر الفصل الدراسي --</option>
                        <?php foreach ($semesters as $sem): ?>
                            <option value="<?php echo $sem['id']; ?>" <?php echo $sem['id'] == $selected_semester_id ? 'selected' : ''; ?>>
                                فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?> 
                                (<?php echo htmlspecialchars($sem['major_name']); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </form>
        </div>
        
        <?php if ($selected_semester_id): ?>
        <div class="card">
            <div class="card-header">
                <h3>الخطوة 2: اختر المادة</h3>
            </div>
            <?php if (empty($courses)): ?>
                <div class="alert alert-info">
                    <strong>لا توجد مواد في هذا الفصل الدراسي</strong><br>
                    يجب إضافة المواد أولاً من صفحة <a href="courses.php">إدارة المواد</a>
                </div>
            <?php else: ?>
            <form method="GET" style="display:inline;">
                <input type="hidden" name="semester_id" value="<?php echo $selected_semester_id; ?>">
                <div class="form-group">
                    <label>المادة</label>
                    <select name="course_id" onchange="this.form.submit()" style="width:100%; padding:10px;">
                        <option value="">-- اختر المادة --</option>
                        <?php foreach ($courses as $c): ?>
                            <option value="<?php echo $c['id']; ?>" <?php echo $c['id'] == $selected_course_id ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($c['code']); ?> - <?php echo htmlspecialchars($c['name_ar']); ?> 
                                (<?php echo $c['credit_hours']; ?> ساعات)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </form>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        
        <?php if ($selected_course_id && $selected_semester_id): ?>
            <?php if (!empty($grades_data)): ?>
            <div class="card">
                <div class="card-header">
                    <h3>الخطوة 3: أدخل درجات الطلاب<?php echo $selected_course ? ' - ' . htmlspecialchars($selected_course['name_ar']) : ''; ?></h3>
                </div>
                
                <div class="table-container">
                    <table>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>الرقم الجامعي</th>
                                <th>اسم الطالب</th>
                                <th>الدرجة (0-100)</th>
                                <th>التقدير</th>
                                <th>النقاط</th>
                                <th>الحالة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php $counter = 1; foreach ($grades_data as $grade): ?>
                                <tr>
                                    <td><?php echo $counter++; ?></td>
                                    <td><?php echo htmlspecialchars($grade['student_number']); ?></td>
                                    <td><?php echo htmlspecialchars($grade['full_name_ar']); ?></td>
                                    <td>
                                        <form method="POST" style="display:inline;" onsubmit="return confirm('هل تريد حفظ الدرجة <?php echo htmlspecialchars($grade['full_name_ar']); ?>؟');">
                                            <input type="hidden" name="action" value="save_grade">
                                            <input type="hidden" name="student_course_id" value="<?php echo $grade['id']; ?>">
                                            <input type="number" 
                                                   name="marks" 
                                                   value="<?php echo $grade['marks'] ?? ''; ?>" 
                                                   min="0" 
                                                   max="100" 
                                                   step="0.01" 
                                                   style="width:120px; padding:8px; font-size:1.1em;" 
                                                   placeholder="0-100"
                                                   required>
                                            <button type="submit" class="btn btn-success" style="padding:8px 20px; margin-right:5px;">💾 حفظ</button>
                                        </form>
                                    </td>
                                    <td><strong style="font-size:1.2em;"><?php echo cleanLetterGrade($grade['letter_grade'] ?? '-'); ?></strong></td>
                                    <td><strong><?php echo $grade['points'] ? number_format($grade['points'], 2) : '-'; ?></strong></td>
                                    <td>
                                        <?php if ($grade['grade_id']): ?>
                                            <span style="color: green; font-weight: bold; font-size:1.1em;">✓ تم</span>
                                        <?php else: ?>
                                            <span style="color: orange;">⚠ لم يتم</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php else: ?>
            <div class="alert alert-info">
                <strong>⚠ لا يوجد طلاب مسجلين في هذه المادة</strong><br>
                <p>يجب تسجيل الطلاب في المادة أولاً قبل إدخال الدرجات.</p>
                <?php if (!empty($selected_semester_id) && (int)$selected_semester_id > 0): ?>
                    <a href="register_students_to_course.php?course_id=<?php echo $selected_course_id; ?>&semester_id=<?php echo $selected_semester_id; ?>" class="btn btn-primary" style="margin-top:10px;">
                        📝 تسجيل طلاب في هذه المادة
                    </a>
                <?php else: ?>
                    <div style="margin-top:10px" class="alert alert-warning">
                        يجب اختيار فصل دراسي أولاً لتمكين تسجيل الطلاب.
                    </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <?php if ($selected_semester_id && empty($courses)): ?>
            <div class="alert alert-info">
                <strong>لا توجد مواد في هذا الفصل الدراسي</strong><br>
                يجب إضافة المواد أولاً من صفحة <a href="courses.php">إدارة المواد</a>
            </div>
        <?php endif; ?>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

