<?php
/**
 * استخراج قوائم الطلاب المسجلين للامتحانات
 * Exam Registration Lists Export
 * الترويسة: البسملة، اسم الجامعة، الكلية، التخصص، الفصل والسنة
 * الأعمدة: الرقم المتسلسل، الرقم الجامعي، اسم الطالب، اعمال السنة، درجة الامتحان، كود المادة، المجموع، ملحوظات
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();

// تحويل رقم الفصل إلى نص عربي (1 → الفصل الأول، 11 → الفصل الحادي عشر، ...)
$semester_number_ar = [
    1 => 'الأول', 2 => 'الثاني', 3 => 'الثالث', 4 => 'الرابع', 5 => 'الخامس',
    6 => 'السادس', 7 => 'السابع', 8 => 'الثامن', 9 => 'التاسع', 10 => 'العاشر',
    11 => 'الحادي عشر', 12 => 'الثاني عشر', 13 => 'الثالث عشر', 14 => 'الرابع عشر', 15 => 'الخامس عشر',
    16 => 'السادس عشر', 17 => 'السابع عشر', 18 => 'الثامن عشر', 19 => 'التاسع عشر', 20 => 'العشرون',
];
$semesterLabel = function($num) use ($semester_number_ar) {
    $n = (int)$num;
    return isset($semester_number_ar[$n]) ? 'الفصل ' . $semester_number_ar[$n] : 'الفصل ' . $num;
};

// اسم الجامعة (من الإعدادات أو ثابت)
$university_name = 'جامعة الشرق للعلوم والتكنولوجيا';
if (defined('SITE_NAME') && preg_match('/جامعة\s+[^\s:]+(?:\s+[^\s:]+)*/u', SITE_NAME, $m)) {
    $university_name = trim($m[0]);
}

// جلب الكليات
$colleges = [];
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $user = $auth->getCurrentUser();
    if ($user && !empty($user['college_id'])) {
        $c = $db->fetchOne("SELECT id, name_ar FROM colleges WHERE id = ?", [$user['college_id']]);
        if ($c) $colleges = [$c];
    }
}

$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$list_type = isset($_GET['list_type']) && $_GET['list_type'] === 'per_course' ? 'per_course' : 'all';

// التخصصات
$majors = [];
if ($college_id > 0 && ($auth->hasRole('admin') || $auth->belongsToCollege($college_id))) {
    $majors = $db->fetchAll("SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar", [$college_id]);
}

// الفصول الدراسية (حسب الكلية أو التخصص)
$semesters = [];
if ($major_id > 0) {
    $semesters = $db->fetchAll(
        "SELECT s.id, s.semester_number, s.academic_year, s.major_id 
         FROM semesters s 
         WHERE s.major_id = ? 
         ORDER BY s.academic_year DESC, s.semester_number ASC",
        [$major_id]
    );
} elseif ($college_id > 0) {
    $semesters = $db->fetchAll(
        "SELECT s.id, s.semester_number, s.academic_year, s.major_id 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE m.college_id = ? 
         ORDER BY s.academic_year DESC, s.semester_number ASC",
        [$college_id]
    );
}

$list_data = [];
$header_info = null; // college_name, major_name, semester_label

if ($semester_id > 0) {
    $sem = $db->fetchOne(
        "SELECT s.*, m.name_ar as major_name, m.college_id, c.name_ar as college_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         JOIN colleges c ON m.college_id = c.id 
         WHERE s.id = ?",
        [$semester_id]
    );
    if ($sem) {
        // إذا كانت الكلية ليس بها تخصصات (اسم التخصص "الكلية") لا نعرض التخصص في الترويسة
        $major_display = (trim($sem['major_name'] ?? '') === 'الكلية') ? '' : ($sem['major_name'] ?? '');
        $header_info = [
            'college_name' => $sem['college_name'],
            'major_name'   => $major_display,
            'semester_label' => $semesterLabel($sem['semester_number']) . ' — ' . $sem['academic_year'],
        ];
        // قائمة الطلاب المسجلين في مواد هذا الفصل (طالب × مادة) مع الدرجة إن وجدت
        $list_data = $db->fetchAll(
            "SELECT 
                sc.id as sc_id,
                s.id as student_id,
                s.student_number,
                s.full_name_ar,
                c.code as course_code,
                c.name_ar as course_name,
                g.marks,
                g.letter_grade
             FROM student_courses sc
             JOIN students s ON sc.student_id = s.id
             JOIN courses c ON sc.course_id = c.id
             LEFT JOIN grades g ON g.student_course_id = sc.id
             WHERE sc.semester_id = ?
             ORDER BY s.full_name_ar, c.code",
            [$semester_id]
        );
    }
}

// تجميع البيانات حسب المادة (للقائمة المنفصلة لكل مادة)
$list_by_course = [];
if (!empty($list_data)) {
    foreach ($list_data as $row) {
        $key = $row['course_code'];
        if (!isset($list_by_course[$key])) {
            $list_by_course[$key] = ['course_code' => $row['course_code'], 'course_name' => $row['course_name'], 'rows' => []];
        }
        $list_by_course[$key]['rows'][] = $row;
    }
    $list_by_course = array_values($list_by_course);
}

// تصدير CSV
$export_csv = isset($_GET['export']) && $_GET['export'] === 'csv' && $semester_id > 0 && $header_info;
$export_list_type = isset($_GET['list_type']) && $_GET['list_type'] === 'per_course' ? 'per_course' : 'all';
if ($export_csv && !empty($list_data)) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="قائمة_الامتحانات_' . $semester_id . '_' . $export_list_type . '.csv"');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    $out = fopen('php://output', 'w');
    fwrite($out, "\xEF\xBB\xBF"); // BOM UTF-8

    if ($export_list_type === 'per_course' && !empty($list_by_course)) {
        foreach ($list_by_course as $idx => $course_block) {
            if ($idx > 0) { fputcsv($out, []); fputcsv($out, []); }
            fputcsv($out, ['بسم الله الرحمن الرحيم'], ',');
            fputcsv($out, [$university_name], ',');
            fputcsv($out, [$header_info['college_name']], ',');
            if ((string)($header_info['major_name'] ?? '') !== '') { fputcsv($out, [$header_info['major_name']], ','); }
            fputcsv($out, [$header_info['semester_label']], ',');
            fputcsv($out, ['المادة: ' . $course_block['course_code'] . ' — ' . $course_block['course_name']], ',');
            fputcsv($out, []); // سطر فاصل
            fputcsv($out, ['م', 'الرقم الجامعي', 'اسم الطالب', 'اعمال السنة', 'درجة الامتحان', 'المجموع', 'ملحوظات'], ',');
            $row_num = 1;
            foreach ($course_block['rows'] as $row) {
                fputcsv($out, [$row_num++, $row['student_number'], $row['full_name_ar'], '', '', '', ''], ',');
            }
        }
    } else {
        fputcsv($out, ['بسم الله الرحمن الرحيم'], ',');
        fputcsv($out, [$university_name], ',');
        fputcsv($out, [$header_info['college_name']], ',');
        if ((string)($header_info['major_name'] ?? '') !== '') { fputcsv($out, [$header_info['major_name']], ','); }
        fputcsv($out, [$header_info['semester_label']], ',');
        fputcsv($out, []);
        fputcsv($out, ['م', 'الرقم الجامعي', 'اسم الطالب', 'اعمال السنة', 'درجة الامتحان', 'كود المادة', 'المجموع', 'ملحوظات'], ',');
        $row_num = 1;
        foreach ($list_data as $row) {
            fputcsv($out, [$row_num++, $row['student_number'], $row['full_name_ar'], '', '', $row['course_code'], '', ''], ',');
        }
    }
    fclose($out);
    exit;
}

$page_title = 'قوائم الطلاب المسجلين للامتحانات';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .exam-lists-page { max-width: 1000px; margin: 20px auto; }
        .exam-header-print { text-align: center; margin-bottom: 24px; padding: 16px; border-bottom: 2px solid #333; }
        .exam-header-print { font-size: 11px; }
        .exam-header-print .bismillah { font-size: 11px; margin-bottom: 12px; }
        .exam-header-print .university { font-size: 11px; font-weight: bold; margin-bottom: 8px; }
        .exam-header-print .college, .exam-header-print .major, .exam-header-print .semester { margin: 4px 0; font-size: 11px; }
        .exam-table { width: 100%; border-collapse: collapse; margin-top: 16px; font-size: 11px; }
        .exam-table th, .exam-table td { border: 1px solid #333; padding: 8px 10px; text-align: right; color: #000; font-weight: 700; font-size: 11px; }
        .exam-table th { background: #f0f0f0; }
        .exam-table .col-serial { width: 50px; }
        .exam-table .col-total { width: 80px; }
        .btn-print { background: #0d6efd; color: #fff !important; padding: 10px 20px; border: none; border-radius: 6px; font-size: 1rem; font-weight: 600; cursor: pointer; }
        .btn-print:hover { background: #0b5ed7; color: #fff; }
        .exam-list-block { margin-bottom: 32px; }
        .exam-list-block:last-child { page-break-after: auto; }
        @media print {
            .no-print { display: none !important; }
            .main-header { display: none !important; }
            .main-footer { display: none !important; }
            .exam-header-print { border-bottom-color: #000; }
            .exam-header-print, .exam-header-print * { font-size: 11px !important; }
            .exam-table th, .exam-table td { border-color: #000; color: #000 !important; font-weight: 700 !important; font-size: 11px !important; }
            .exam-list-block { page-break-after: always; }
            .exam-list-block:last-child { page-break-after: auto; }
        }
    </style>
</head>
<body>
<?php include 'includes/header.php'; ?>

<div class="container exam-lists-page">
    <div class="page-header no-print">
        <h2>قوائم الطلاب المسجلين للامتحانات</h2>
        <a href="grades.php" class="btn btn-secondary">العودة لإدارة الدرجات</a>
    </div>

    <form method="GET" class="no-print" style="margin-bottom: 24px;">
        <?php if ($auth->hasRole('admin')): ?>
        <div class="form-group" style="margin-bottom: 12px;">
            <label>الكلية:</label>
            <select name="college_id" onchange="this.form.submit()">
                <option value="0">-- اختر الكلية --</option>
                <?php foreach ($colleges as $c): ?>
                    <option value="<?php echo (int)$c['id']; ?>" <?php echo $college_id == $c['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($c['name_ar']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <?php else: ?>
            <input type="hidden" name="college_id" value="<?php echo $college_id; ?>">
        <?php endif; ?>

        <div class="form-group" style="margin-bottom: 12px;">
            <label>التخصص:</label>
            <select name="major_id" onchange="this.form.submit()" <?php echo $college_id <= 0 ? 'disabled' : ''; ?>>
                <option value="0">-- الكل / بدون تخصص --</option>
                <?php foreach ($majors as $m): ?>
                    <option value="<?php echo (int)$m['id']; ?>" <?php echo $major_id == $m['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($m['name_ar']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group" style="margin-bottom: 12px;">
            <label>الفصل الدراسي:</label>
            <select name="semester_id" onchange="this.form.submit()" <?php echo empty($semesters) ? 'disabled' : ''; ?>>
                <option value="0">-- اختر الفصل --</option>
                <?php foreach ($semesters as $s): ?>
                    <option value="<?php echo (int)$s['id']; ?>" <?php echo $semester_id == $s['id'] ? 'selected' : ''; ?>>
                        <?php echo $semesterLabel($s['semester_number']); ?> — <?php echo htmlspecialchars($s['academic_year']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <?php if ($semester_id > 0): ?>
        <div class="form-group" style="margin-bottom: 12px;">
            <label>نوع القائمة:</label>
            <select name="list_type" onchange="this.form.submit()">
                <option value="all" <?php echo $list_type === 'all' ? 'selected' : ''; ?>>قائمة واحدة (جميع المواد معاً)</option>
                <option value="per_course" <?php echo $list_type === 'per_course' ? 'selected' : ''; ?>>قائمة منفصلة لكل مادة</option>
            </select>
        </div>
        <?php endif; ?>
    </form>

    <?php if ($semester_id > 0 && $header_info): ?>
        <?php if (empty($list_data)): ?>
            <div class="alert alert-warning no-print">لا يوجد طلاب مسجلون في مواد هذا الفصل.</div>
        <?php else: ?>
            <div class="no-print" style="margin-bottom: 16px; display: flex; flex-wrap: wrap; gap: 10px; align-items: center;">
                <a href="exam_registration_lists.php?college_id=<?php echo $college_id; ?>&amp;major_id=<?php echo $major_id; ?>&amp;semester_id=<?php echo $semester_id; ?>&amp;list_type=<?php echo $list_type; ?>&amp;export=csv" class="btn btn-success">📥 تصدير CSV</a>
                <button type="button" onclick="window.print(); return false;" class="btn btn-print" title="طباعة القائمة">🖨️ للطابعة</button>
            </div>

            <?php if ($list_type === 'per_course' && !empty($list_by_course)): ?>
                <?php foreach ($list_by_course as $course_block): ?>
                    <div class="exam-list-block" style="page-break-after: always;">
                        <div class="exam-header-print">
                            <div class="bismillah">بسم الله الرحمن الرحيم</div>
                            <div class="university"><?php echo htmlspecialchars($university_name); ?></div>
                            <div class="college"><?php echo htmlspecialchars($header_info['college_name']); ?></div>
                            <?php if ((string)($header_info['major_name'] ?? '') !== ''): ?><div class="major"><?php echo htmlspecialchars($header_info['major_name']); ?></div><?php endif; ?>
                            <div class="semester"><?php echo htmlspecialchars($header_info['semester_label']); ?></div>
                            <div class="course-title" style="margin-top: 12px; font-weight: bold;">المادة: <?php echo htmlspecialchars($course_block['course_code']); ?> — <?php echo htmlspecialchars($course_block['course_name']); ?></div>
                        </div>
                        <table class="exam-table">
                            <thead>
                                <tr>
                                    <th class="col-serial">م</th>
                                    <th>الرقم الجامعي</th>
                                    <th>اسم الطالب</th>
                                    <th>اعمال السنة</th>
                                    <th>درجة الامتحان</th>
                                    <th>المجموع</th>
                                    <th class="col-total">ملحوظات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php $row_num = 1; foreach ($course_block['rows'] as $row): ?>
                                    <tr>
                                        <td><?php echo $row_num++; ?></td>
                                        <td><?php echo htmlspecialchars($row['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($row['full_name_ar']); ?></td>
                                        <td></td>
                                        <td></td>
                                        <td></td>
                                        <td></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="exam-header-print">
                    <div class="bismillah">بسم الله الرحمن الرحيم</div>
                    <div class="university"><?php echo htmlspecialchars($university_name); ?></div>
                    <div class="college"><?php echo htmlspecialchars($header_info['college_name']); ?></div>
                    <?php if ((string)($header_info['major_name'] ?? '') !== ''): ?><div class="major"><?php echo htmlspecialchars($header_info['major_name']); ?></div><?php endif; ?>
                    <div class="semester"><?php echo htmlspecialchars($header_info['semester_label']); ?></div>
                </div>
                <table class="exam-table">
                    <thead>
                        <tr>
                            <th class="col-serial">م</th>
                            <th>الرقم الجامعي</th>
                            <th>اسم الطالب</th>
                            <th>اعمال السنة</th>
                            <th>درجة الامتحان</th>
                            <th>كود المادة</th>
                            <th>المجموع</th>
                            <th class="col-total">ملحوظات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $row_num = 1; foreach ($list_data as $row): ?>
                            <tr>
                                <td><?php echo $row_num++; ?></td>
                                <td><?php echo htmlspecialchars($row['student_number']); ?></td>
                                <td><?php echo htmlspecialchars($row['full_name_ar']); ?></td>
                                <td></td>
                                <td></td>
                                <td><?php echo htmlspecialchars($row['course_code']); ?></td>
                                <td></td>
                                <td></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        <?php endif; ?>
    <?php elseif ($college_id > 0 && empty($semesters)): ?>
        <div class="alert alert-info no-print">اختر التخصص أولاً أو لا توجد فصول لهذه الكلية.</div>
    <?php elseif ($college_id <= 0): ?>
        <div class="alert alert-info no-print">اختر الكلية ثم التخصص ثم الفصل الدراسي.</div>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
</body>
</html>
