<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('enter_grades') && !$auth->hasPermission('manage_grades'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

function getSemesterName($number) {
    switch ((int)$number) {
        case 1: return 'الفصل الأول';
        case 2: return 'الفصل الثاني';
        case 3: return 'الفصل الثالث';
        case 4: return 'الفصل الرابع';
        case 5: return 'الفصل الخامس';
        case 6: return 'الفصل السادس';
        case 7: return 'الفصل السابع';
        case 8: return 'الفصل الثامن';
        default: return 'فصل ' . (int)$number;
    }
}
$student_id = isset($_GET['student_id']) ? (int)$_GET['student_id'] : 0;
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
// عند إرسال النموذج (POST) نعتمد قيم المادة والفصل من الطلب حتى لا تضيع
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['course_id']) && (int)$_POST['course_id'] > 0) {
        $course_id = (int)$_POST['course_id'];
    }
    if (isset($_POST['semester_id']) && (int)$_POST['semester_id'] > 0) {
        $semester_id = (int)$_POST['semester_id'];
    }
}
$message = '';

// معالجة إدخال/تعديل الدرجات لجميع الطلاب دفعة واحدة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'save_all_grades' && $course_id && $semester_id) {
        // حفظ الدرجات لجميع الطلاب دفعة واحدة
        $grades_data = $_POST['grades'] ?? [];
        $success_count = 0;
        $error_count = 0;
        $updated_students = [];
        
        if (empty($grades_data)) {
            $message = '<div class="alert alert-warning">⚠️ لم يتم إدخال أي درجات</div>';
        } else {
            try {
                // بدء معاملة قاعدة البيانات
                $db->getConnection()->beginTransaction();
                
                foreach ($grades_data as $student_course_id => $marks_input) {
                    $student_course_id = (int)$student_course_id;
                    $marks_input = trim($marks_input);
                    
                    // تخطي الحقول الفارغة
                    if ($marks_input === '') {
                        continue;
                    }
                    
                    // معالجة الدرجة (رقمية أو نصية)
                    $grade_data = $gradeCalculator->processGradeInput($marks_input);
                    
                    if (!$grade_data) {
                        $error_count++;
                        continue;
                    }
                    
                    // التحقق من نطاق الدرجة الرقمية
                    if (!$grade_data['is_text'] && ($grade_data['marks'] < 0 || $grade_data['marks'] > 100)) {
                        $error_count++;
                        continue;
                    }
                    
                    // جلب معلومات تسجيل الطالب في المادة
                    $student_course = $db->fetchOne(
                        "SELECT sc.*, s.college_id, s.id as student_id 
                         FROM student_courses sc 
                         JOIN students s ON sc.student_id = s.id 
                         WHERE sc.id = ? AND sc.course_id = ? AND sc.semester_id = ?",
                        [$student_course_id, $course_id, $semester_id]
                    );
                    
                    if (!$student_course) {
                        $error_count++;
                        continue;
                    }
                    
                    // التحقق من الصلاحيات
                    if (!$auth->belongsToCollege($student_course['college_id']) && !$auth->hasRole('admin')) {
                        $error_count++;
                        continue;
                    }
                    
                    // إعداد القيم للحفظ
                    // إذا كانت الدرجة نصية، احفظ القيمة النصية في marks، وإلا احفظ الرقم
                    if ($grade_data['is_text']) {
                        $marks = $marks_input; // حفظ القيمة النصية الأصلية
                    } else {
                        $marks = $grade_data['marks'] !== null ? $grade_data['marks'] : 0;
                    }
                    $letter_grade = $grade_data['letter_grade'];
                    // حفظ points كما هي (null للدرجات النصية بدون نقاط مثل "بد")
                    $points = $grade_data['points'];
                    
                    // التحقق من وجود درجة سابقة
                    $existing = $db->fetchOne(
                        "SELECT * FROM grades WHERE student_course_id = ?",
                        [$student_course_id]
                    );
                    
                    if ($existing) {
                        // تحديث الدرجة
                        $db->query(
                            "UPDATE grades SET marks = ?, letter_grade = ?, points = ?, updated_at = CURRENT_TIMESTAMP, entered_by = ?
                             WHERE student_course_id = ?",
                            [$marks, $letter_grade, $points, $_SESSION['user_id'], $student_course_id]
                        );
                    } else {
                        // إضافة درجة جديدة
                        $db->query(
                            "INSERT INTO grades (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                            [
                                $student_course_id,
                                $student_course['student_id'],
                                $student_course['course_id'],
                                $student_course['semester_id'],
                                $marks,
                                $letter_grade,
                                $points,
                                $_SESSION['user_id']
                            ]
                        );
                    }
                    
                    $success_count++;
                    $updated_students[] = $student_course['student_id'];
                }
                
                // تأكيد المعاملة
                $db->getConnection()->commit();
                
                // تحديث المعدلات لجميع الطلاب المحدثين
                $unique_students = array_unique($updated_students);
                foreach ($unique_students as $student_id) {
                    try {
                        $gradeCalculator->updateAllGPAs($student_id);
                    } catch (Exception $e) {
                        error_log("Error updating GPA for student $student_id: " . $e->getMessage());
                    }
                }
                
                // بناء رسالة النتيجة
                if ($success_count > 0) {
                    $msg_text = ($success_count === 1) ? 'تم حفظ 1 درجة بنجاح' : 'تم حفظ ' . $success_count . ' درجات بنجاح';
                    $message = '<div class="alert alert-success">✓ ' . $msg_text . '</div>';
                }
                if ($error_count > 0) {
                    $message .= '<div class="alert alert-warning">⚠ ' . $error_count . ' صف تم تخطيه (قيمة غير صالحة أو لا تطابق تسجيل الطالب)</div>';
                }
                if ($success_count === 0 && !empty($grades_data)) {
                    $message = '<div class="alert alert-warning">⚠ لم يتم حفظ أي درجة. تحقق من أن القيم أرقام بين 0 و 100 أو: بد، غ، غش. وإن استمرت المشكلة قد يكون تسجيل الطلاب في المادة أو الفصل غير صحيح.</div>';
                }
                
                // إعادة توجيه لتجنب إعادة إرسال النموذج
                if ($success_count > 0) {
                    header("Location: grades.php?course_id=" . $course_id . "&semester_id=" . $semester_id . "&msg=success&count=" . $success_count);
                    exit;
                }
                
            } catch (Exception $e) {
                // إلغاء المعاملة في حالة الخطأ
                $db->getConnection()->rollBack();
                $message = '<div class="alert alert-error">خطأ في حفظ الدرجات: ' . htmlspecialchars($e->getMessage()) . '</div>';
                error_log("Error saving grades: " . $e->getMessage());
            }
        }
    } elseif ($_POST['action'] === 'save_grade') {
        // حفظ درجة واحدة (للتوافق مع الكود القديم)
        $student_course_id = (int)($_POST['student_course_id'] ?? 0);
        $marks = trim($_POST['marks'] ?? '');
        
        if ($student_course_id && $marks !== '' && is_numeric($marks)) {
            $marks = (float)$marks;
            
            if ($marks >= 0 && $marks <= 100) {
                $student_course = $db->fetchOne(
                    "SELECT sc.*, s.college_id FROM student_courses sc 
                     JOIN students s ON sc.student_id = s.id 
                     WHERE sc.id = ?",
                    [$student_course_id]
                );
                
                if ($student_course && ($auth->belongsToCollege($student_course['college_id']) || $auth->hasRole('admin'))) {
                    try {
                        $grade_info = $gradeCalculator->calculatePoints($marks);
                        
                        // التحقق من وجود درجة سابقة
                        $existing = $db->fetchOne(
                            "SELECT * FROM grades WHERE student_course_id = ?",
                            [$student_course_id]
                        );
                        
                        if ($existing) {
                            // تحديث الدرجة
                            $db->query(
                                "UPDATE grades SET marks = ?, letter_grade = ?, points = ?, updated_at = CURRENT_TIMESTAMP, entered_by = ?
                                 WHERE student_course_id = ?",
                                [$marks, $grade_info['letter'], $grade_info['points'], $_SESSION['user_id'], $student_course_id]
                            );
                        } else {
                            // إضافة درجة جديدة
                            $db->query(
                                "INSERT INTO grades (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                                [
                                    $student_course_id,
                                    $student_course['student_id'],
                                    $student_course['course_id'],
                                    $student_course['semester_id'],
                                    $marks,
                                    $grade_info['letter'],
                                    $grade_info['points'],
                                    $_SESSION['user_id']
                                ]
                            );
                        }
                        
                        // تحديث المعدلات
                        $gradeCalculator->updateAllGPAs($student_course['student_id']);
                        
                        $message = '<div class="alert alert-success">تم حفظ الدرجة بنجاح</div>';
                        
                        // إعادة توجيه لتجنب إعادة إرسال النموذج
                        $redirect_url = "grades.php";
                        if ($course_id) {
                            $redirect_url .= "?course_id=" . $course_id;
                            if ($semester_id) {
                                $redirect_url .= "&semester_id=" . $semester_id;
                            }
                        } elseif ($student_id) {
                            $redirect_url .= "?student_id=" . $student_id;
                            if ($semester_id) {
                                $redirect_url .= "&semester_id=" . $semester_id;
                            }
                        }
                        header("Location: $redirect_url&msg=success");
                        exit;
                    } catch (PDOException $e) {
                        $message = '<div class="alert alert-error">خطأ في حفظ الدرجة: ' . htmlspecialchars($e->getMessage()) . '</div>';
                    }
                } else {
                    $message = '<div class="alert alert-error">ليس لديك صلاحية لإدخال الدرجة</div>';
                }
            } else {
                $message = '<div class="alert alert-error">الدرجة يجب أن تكون بين 0 و 100</div>';
            }
        } else {
            $message = '<div class="alert alert-error">يرجى إدخال درجة صحيحة</div>';
        }
    }
}

// عرض رسالة النجاح من URL
if (isset($_GET['msg']) && $_GET['msg'] === 'success') {
    if (isset($_GET['count']) && (int)$_GET['count'] > 0) {
        $c = (int)$_GET['count'];
        $msg_text = ($c === 1) ? 'تم حفظ 1 درجة بنجاح' : 'تم حفظ ' . $c . ' درجات بنجاح';
        $message = '<div class="alert alert-success">✓ ' . $msg_text . '</div>';
    } else {
        $message = '<div class="alert alert-success">✓ تم حفظ الدرجة بنجاح</div>';
    }
}

// جلب البيانات للعرض
$grades = [];
$course = null;
$semester = null;

if ($course_id) {
    $course = $db->fetchOne(
        "SELECT c.*, m.college_id FROM courses c JOIN majors m ON c.major_id = m.id WHERE c.id = ?",
        [$course_id]
    );
    
    if ($course && ($auth->belongsToCollege($course['college_id']) || $auth->hasRole('admin'))) {
        if ($semester_id) {
            $grades = $db->fetchAll(
                "SELECT sc.*, s.student_number, s.full_name_ar, g.marks, g.letter_grade, g.points, g.id as grade_id
                 FROM student_courses sc
                 JOIN students s ON sc.student_id = s.id
                 LEFT JOIN grades g ON sc.id = g.student_course_id
                 WHERE sc.course_id = ? AND sc.semester_id = ?
                 ORDER BY s.full_name_ar",
                [$course_id, $semester_id]
            );
        } else {
            $grades = $db->fetchAll(
                "SELECT sc.*, s.student_number, s.full_name_ar, g.marks, g.letter_grade, g.points, g.id as grade_id
                 FROM student_courses sc
                 JOIN students s ON sc.student_id = s.id
                 LEFT JOIN grades g ON sc.id = g.student_course_id
                 WHERE sc.course_id = ?
                 ORDER BY s.full_name_ar",
                [$course_id]
            );
            // عند وجود مادة ودرجات بدون فصل: استخدام فصل أول طالب لعرض نموذج "حفظ الكل"
            if (!empty($grades) && !$semester_id) {
                $semester_id = (int)($grades[0]['semester_id'] ?? 0);
                $grades = array_values(array_filter($grades, function ($r) use ($semester_id) {
                    return (int)($r['semester_id'] ?? 0) === $semester_id;
                }));
            }
        }
    }
} elseif ($student_id && $auth->canAccessStudent($student_id)) {
    if ($semester_id) {
        $grades = $db->fetchAll(
            "SELECT sc.*, c.code as course_code, c.name_ar as course_name, c.credit_hours,
             g.marks, g.letter_grade, g.points, g.id as grade_id, s.academic_year, s.semester_number
             FROM student_courses sc
             JOIN courses c ON sc.course_id = c.id
             JOIN semesters s ON sc.semester_id = s.id
             LEFT JOIN grades g ON sc.id = g.student_course_id
             WHERE sc.student_id = ? AND sc.semester_id = ?
             ORDER BY s.academic_year, s.semester_number, c.code",
            [$student_id, $semester_id]
        );
    } else {
        $grades = $db->fetchAll(
            "SELECT sc.*, c.code as course_code, c.name_ar as course_name, c.credit_hours,
             g.marks, g.letter_grade, g.points, g.id as grade_id, s.academic_year, s.semester_number
             FROM student_courses sc
             JOIN courses c ON sc.course_id = c.id
             JOIN semesters s ON sc.semester_id = s.id
             LEFT JOIN grades g ON sc.id = g.student_course_id
             WHERE sc.student_id = ?
             ORDER BY s.academic_year, s.semester_number, c.code",
            [$student_id]
        );
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الدرجات</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدارة الدرجات</h2>
            <div>
                <?php if ($course): ?>
                    <h3>المادة: <?php echo htmlspecialchars($course['name_ar']); ?> (<?php echo htmlspecialchars($course['code']); ?>)</h3>
                <?php endif; ?>
                <div style="margin-top: 10px;">
                    <a href="upload_grades.php" class="btn btn-primary" style="margin-left: 10px;">📤 رفع الدرجات من Excel</a>
                    <?php if ($course_id): ?>
                        <a href="courses.php?semester_id=<?php echo $semester_id; ?>" class="btn btn-secondary">العودة للمواد</a>
                    <?php elseif ($student_id): ?>
                        <a href="student_details.php?id=<?php echo $student_id; ?>" class="btn btn-secondary">العودة لتفاصيل الطالب</a>
                    <?php else: ?>
                        <a href="dashboard.php" class="btn btn-secondary">العودة للوحة التحكم</a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <?php echo $message; ?>
        
        <?php if ($course_id && !$semester_id): ?>
            <div class="card" style="margin-bottom: 20px;">
                <div class="card-header">
                    <h3>اختر الفصل الدراسي</h3>
                </div>
                <p>يجب اختيار فصل دراسي لعرض الطلاب المسجلين وتسجيل طلاب جدد.</p>
                <?php
                // جلب الفصول الدراسية المتاحة لهذه المادة
                try {
                    $available_semesters = $db->fetchAll(
                        "SELECT DISTINCT s.*, m.name_ar as major_name
                         FROM semesters s
                         JOIN majors m ON s.major_id = m.id
                         JOIN courses c ON c.major_id = m.id
                         WHERE c.id = ?
                         ORDER BY s.academic_year DESC, s.semester_number DESC",
                        [$course_id]
                    );
                    
                    if (!empty($available_semesters)):
                ?>
                    <form method="GET" style="margin-top: 15px;">
                        <input type="hidden" name="course_id" value="<?php echo $course_id; ?>">
                        <div class="form-group">
                            <label>الفصل الدراسي:</label>
                            <select name="semester_id" required style="width: 100%; padding: 10px; margin-top: 5px;">
                                <option value="">-- اختر الفصل الدراسي --</option>
                                <?php foreach ($available_semesters as $sem): ?>
                                    <option value="<?php echo $sem['id']; ?>">
                                        فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                                        (<?php echo htmlspecialchars($sem['major_name']); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <button type="submit" class="btn btn-primary">اختر الفصل الدراسي</button>
                    </form>
                <?php
                    else:
                ?>
                    <div class="alert alert-warning">
                        <strong>لا توجد فصول دراسية متاحة لهذه المادة.</strong><br>
                        يجب إنشاء فصل دراسي أولاً من صفحة <a href="semesters.php">إدارة الفصول الدراسية</a>
                    </div>
                <?php
                    endif;
                } catch (Exception $e) {
                    echo '<div class="alert alert-error">خطأ في جلب الفصول الدراسية: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
                ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($grades)): ?>
        <?php if ($course_id): ?>
        <!-- نموذج إدخال الدرجات لجميع الطلاب دفعة واحدة (حفظ الكل) -->
        <div class="card" style="margin-bottom: 20px; border: 2px solid #4CAF50;">
            <div class="card-header" style="background-color: #4CAF50; color: white;">
                <h3>📝 إدخال الدرجات لجميع الطلاب دفعة واحدة</h3>
            </div>
            <div style="padding: 20px;">
                <p><strong>كيفية الاستخدام:</strong></p>
                <ol style="margin-right: 20px;">
                    <li>أدخل الدرجات لجميع الطلاب في الجدول أدناه</li>
                    <li>يمكنك ترك الحقول الفارغة للطلاب الذين لا تريد إدخال درجاتهم</li>
                    <li>اضغط زر "حفظ جميع الدرجات" في الأسفل</li>
                    <li>سيتم حفظ جميع الدرجات وتحديث المعدلات تلقائياً</li>
                </ol>
                
                <form method="POST" id="gradesForm" action="grades.php?course_id=<?php echo (int)$course_id; ?>&amp;semester_id=<?php echo (int)$semester_id; ?>" onsubmit="return confirm('هل تريد حفظ جميع الدرجات المدخلة؟');">
                    <input type="hidden" name="action" value="save_all_grades">
                    <input type="hidden" name="course_id" value="<?php echo (int)$course_id; ?>">
                    <input type="hidden" name="semester_id" value="<?php echo (int)$semester_id; ?>">
                    
                    <div class="table-container" style="margin-top: 20px;">
                        <table>
                            <thead>
                                <tr>
                                    <th>الرقم الجامعي</th>
                                    <th>اسم الطالب</th>
                                    <th>الدرجة الحالية</th>
                                    <th>الدرجة الجديدة (0-100)</th>
                                    <th>التقدير</th>
                                    <th>النقاط</th>
                                    <th>الحالة</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($grades as $grade): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($grade['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($grade['full_name_ar']); ?></td>
                                        <td>
                                            <?php if ($grade['marks'] !== null && $grade['marks'] !== ''): ?>
                                                <?php if (is_numeric($grade['marks'])): ?>
                                                    <strong style="color: #4caf50;"><?php echo number_format((float)$grade['marks'], 2); ?></strong>
                                                <?php else: ?>
                                                    <strong style="color: #4caf50;"><?php echo htmlspecialchars($grade['marks']); ?></strong>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span style="color: #999;">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <input type="text" 
                                                   name="grades[<?php echo $grade['id']; ?>]" 
                                                   value="<?php echo htmlspecialchars($grade['marks'] ?? ''); ?>" 
                                                   style="width:120px; padding:8px; border: 2px solid #ddd; border-radius: 4px;" 
                                                   placeholder="درجة أو: بد، غ، غش"
                                                   class="grade-input"
                                                   onchange="updateGradePreview(this, <?php echo $grade['id']; ?>)">
                                        </td>
                                        <td id="preview_letter_<?php echo $grade['id']; ?>" class="preview-cell">
                                            <strong><?php 
                                            $letter_grade = $grade['letter_grade'] ?? '-';
                                            if ($letter_grade !== '-') {
                                                $letter_grade = str_replace(['+', 'B+', 'C+', 'A+'], ['', 'B', 'C', 'A'], $letter_grade);
                                            }
                                            echo htmlspecialchars($letter_grade); 
                                            ?></strong>
                                        </td>
                                        <td id="preview_points_<?php echo $grade['id']; ?>" class="preview-cell">
                                            <strong><?php echo $grade['points'] ? number_format($grade['points'], 2) : '-'; ?></strong>
                                        </td>
                                        <td>
                                            <?php if ($grade['grade_id'] !== null): ?>
                                                <span style="color: green; font-weight: bold;">✓ تم الإدخال</span>
                                            <?php else: ?>
                                                <span style="color: orange;">⚠ لم يتم الإدخال</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div style="margin-top: 20px; padding: 15px; background: #e3f2fd; border-radius: 5px;">
                        <p><strong>عدد الطلاب:</strong> <?php echo count($grades); ?> طالب</p>
                        <p><strong>عدد الطلاب المدخلة درجاتهم:</strong> <span id="filledCount">0</span> طالب</p>
                    </div>
                    
                    <button type="submit" class="btn btn-success" style="margin-top: 15px; padding: 12px 30px; font-size: 16px;">
                        ✅ حفظ جميع الدرجات
                    </button>
                    <button type="button" onclick="clearAllGrades()" class="btn btn-secondary" style="margin-top: 15px; padding: 12px 30px; font-size: 16px;">
                        🗑️ مسح جميع الدرجات
                    </button>
                </form>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- جدول عرض الدرجات مع حفظ درجة درجة (للعرض حسب الطالب فقط) -->
        <?php if (!$course_id && !empty($grades)): ?>
        <div class="card" style="margin-bottom: 20px;">
            <div class="card-header">
                <h3>تعليمات إدخال الدرجات</h3>
            </div>
            <p><strong>كيفية الاستخدام:</strong></p>
            <ol style="margin-right: 20px;">
                <li>اكتب الدرجة في المربع (من 0 إلى 100)</li>
                <li>اضغط زر "حفظ"</li>
                <li>سيتم حساب التقدير والنقاط تلقائياً</li>
                <li>ستظهر رسالة نجاح عند الحفظ</li>
            </ol>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <?php if ($course_id): ?>
                            <th>الرقم الجامعي</th>
                            <th>اسم الطالب</th>
                        <?php else: ?>
                            <th>رمز المادة</th>
                            <th>اسم المادة</th>
                            <th>الفصل</th>
                        <?php endif; ?>
                        <th>الدرجة (0-100)</th>
                        <th>التقدير</th>
                        <th>النقاط</th>
                        <th>الحالة</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($grades as $grade): ?>
                        <tr>
                            <?php if ($course_id): ?>
                                <td><?php echo htmlspecialchars($grade['student_number']); ?></td>
                                <td><?php echo htmlspecialchars($grade['full_name_ar']); ?></td>
                            <?php else: ?>
                                <td><?php echo htmlspecialchars($grade['course_code']); ?></td>
                                <td><?php echo htmlspecialchars($grade['course_name']); ?></td>
                                <td><?php echo getSemesterName($grade['semester_number']); ?> - <?php echo htmlspecialchars($grade['academic_year']); ?></td>
                            <?php endif; ?>
                            <td>
                                <form method="POST" style="display:inline;" onsubmit="return confirm('هل تريد حفظ هذه الدرجة؟');">
                                    <input type="hidden" name="action" value="save_grade">
                                    <input type="hidden" name="student_course_id" value="<?php echo $grade['id']; ?>">
                                    <input type="text" 
                                           name="marks" 
                                           value="<?php echo htmlspecialchars($grade['marks'] ?? ''); ?>" 
                                           style="width:100px; padding:5px;" 
                                           placeholder="درجة أو نص"
                                           required>
                                    <button type="submit" class="btn btn-success" style="padding:5px 15px; margin-right:5px;">حفظ</button>
                                </form>
                            </td>
                            <td><strong><?php 
                            $letter_grade = $grade['letter_grade'] ?? '-';
                            if ($letter_grade !== '-') {
                                $letter_grade = str_replace(['+', 'B+', 'C+', 'A+'], ['', 'B', 'C', 'A'], $letter_grade);
                            }
                            echo htmlspecialchars($letter_grade); 
                            ?></strong></td>
                            <td><strong><?php echo $grade['points'] ? number_format($grade['points'], 2) : '-'; ?></strong></td>
                            <td>
                                <?php if ($grade['grade_id']): ?>
                                    <span style="color: green; font-weight: bold;">✓ تم الإدخال</span>
                                <?php else: ?>
                                    <span style="color: orange;">⚠ لم يتم الإدخال</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
        <?php else: ?>
            <div class="alert alert-info">
                <strong>لا توجد بيانات لعرضها</strong><br>
                <?php if ($course_id): ?>
                    <p><strong>⚠ المشكلة:</strong> لا يوجد طلاب مسجلين في هذه المادة.</p>
                    <p><strong>الحل:</strong> يجب تسجيل الطلاب في المادة أولاً قبل إدخال الدرجات.</p>
                    <?php if (!empty($semester_id) && (int)$semester_id > 0): ?>
                        <div style="margin-top:15px;">
                            <a href="register_students_to_course.php?course_id=<?php echo $course_id; ?>&semester_id=<?php echo $semester_id; ?>" class="btn btn-primary">
                                📝 تسجيل طلاب في هذه المادة
                            </a>
                            <a href="courses.php?semester_id=<?php echo $semester_id; ?>" class="btn btn-secondary">
                                العودة للمواد
                            </a>
                        </div>
                    <?php else: ?>
                        <div style="margin-top:10px" class="alert alert-warning">
                            <strong>يجب اختيار فصل دراسي أولاً لتمكين تسجيل الطلاب.</strong><br>
                            استخدم القائمة أعلاه لاختيار الفصل الدراسي.
                        </div>
                    <?php endif; ?>
                <?php elseif ($student_id): ?>
                    <p><strong>⚠ المشكلة:</strong> لا يوجد مواد مسجلة لهذا الطالب.</p>
                    <p><strong>الحل:</strong> يجب تسجيل الطالب في المواد أولاً.</p>
                    <a href="register_courses.php?student_id=<?php echo $student_id; ?>&semester_id=<?php echo $semester_id; ?>" class="btn btn-primary" style="margin-top:10px;">
                        📝 تسجيل مواد للطالب
                    </a>
                <?php else: ?>
                    <p><strong>يرجى اختيار مادة أو طالب لعرض الدرجات.</strong></p>
                    <p>يمكنك:</p>
                    <ul style="margin-right: 20px;">
                        <li>الذهاب إلى <a href="courses.php">إدارة المواد</a> واختيار مادة ثم الضغط على "الدرجات"</li>
                        <li>الذهاب إلى <a href="students.php">إدارة الطلاب</a> واختيار طالب ثم الضغط على "الدرجات"</li>
                        <li>استخدام <a href="enter_grades_simple.php">صفحة إدخال الدرجات المبسطة</a></li>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        // جدول التقديرات
        // التقدير الجديد: A (80-100), B (70-79), C (60-69), D (50-59), F (<50)
        const gradeScale = {
            'A': { min: 80, max: 100, points: 4.0 },
            'B': { min: 70, max: 79, points: 3.0 },
            'C': { min: 60, max: 69, points: 2.0 },
            'D': { min: 50, max: 59, points: 1.0 },
            'F': { min: 0, max: 49, points: 0.0 }
        };
        
        function calculateGrade(marks) {
            if (marks === '' || isNaN(marks)) {
                return { letter: '-', points: '-' };
            }
            
            marks = parseFloat(marks);
            
            if (marks >= 80 && marks <= 100) return { letter: 'A', points: 4.0 };
            if (marks >= 70 && marks <= 79) return { letter: 'B', points: 3.0 };
            if (marks >= 60 && marks <= 69) return { letter: 'C', points: 2.0 };
            if (marks >= 50 && marks <= 59) return { letter: 'D', points: 1.0 };
            return { letter: 'F', points: 0.0 };
        }
        
        function updateGradePreview(input, studentCourseId) {
            const marks = input.value;
            const grade = calculateGrade(marks);
            
            const letterCell = document.getElementById('preview_letter_' + studentCourseId);
            const pointsCell = document.getElementById('preview_points_' + studentCourseId);
            
            if (letterCell) {
                letterCell.innerHTML = '<strong>' + grade.letter + '</strong>';
                if (grade.letter !== '-') {
                    letterCell.style.color = grade.letter === 'F' ? '#f44336' : '#4caf50';
                }
            }
            
            if (pointsCell) {
                pointsCell.innerHTML = '<strong>' + (grade.points !== '-' ? grade.points.toFixed(2) : '-') + '</strong>';
                if (grade.points !== '-') {
                    pointsCell.style.color = grade.points === 0 ? '#f44336' : '#4caf50';
                }
            }
            
            updateFilledCount();
        }
        
        function updateFilledCount() {
            const inputs = document.querySelectorAll('.grade-input');
            let filled = 0;
            inputs.forEach(input => {
                if (input.value !== '' && input.value !== null) {
                    filled++;
                }
            });
            const countEl = document.getElementById('filledCount');
            if (countEl) {
                countEl.textContent = filled;
            }
        }
        
        function clearAllGrades() {
            if (confirm('هل تريد مسح جميع الدرجات المدخلة؟')) {
                document.querySelectorAll('.grade-input').forEach(input => {
                    input.value = '';
                    const studentCourseId = input.name.match(/\[(\d+)\]/)[1];
                    updateGradePreview(input, studentCourseId);
                });
            }
        }
        
        // تحديث العدد عند تحميل الصفحة
        document.addEventListener('DOMContentLoaded', function() {
            updateFilledCount();
            // تحديث معاينة التقديرات للقيم الموجودة
            document.querySelectorAll('.grade-input').forEach(input => {
                const match = input.name.match(/\[(\d+)\]/);
                if (match) {
                    updateGradePreview(input, match[1]);
                }
            });
        });
    </script>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

