<?php
/**
 * صفحة لصق قائمة الدرجات وتوليد القالب مباشرة (بدون Excel)
 * تجنّب مشاكل الترميز — مثل paste_students_list.php
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar') && !$auth->hasPermission('enter_grades') && !$auth->hasPermission('manage_grades'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$message = '';
$saved_count = 0;
$download_url = '';

// جلب الكليات
$colleges = [];
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $user = $auth->getCurrentUser();
    if ($user && !empty($user['college_id'])) {
        $college = $db->fetchOne("SELECT id, name_ar FROM colleges WHERE id = ?", [$user['college_id']]);
        if ($college) $colleges = [$college];
    }
}

// معالجة التوليد
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'generate') {
    $raw = trim($_POST['grades_list'] ?? '');
    $course_codes_raw = trim($_POST['course_codes'] ?? '');
    
    // استخراج أكواد المواد من السطر الأول أو من الحقل المخصص
    $course_codes = [];
    if (!empty($course_codes_raw)) {
        $course_codes = preg_split('/[\t,;\s]+/', $course_codes_raw);
        $course_codes = array_filter(array_map('trim', $course_codes));
    }
    
    $lines = preg_split('/\r\n|\r|\n/', $raw);
    $rows = [];
    $first_line = true;
    
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '') continue;
        
        // فصل الأعمدة: Tab أو فاصلة أو فاصلة منقوطة
        $parts = preg_split('/\t|[,;]/u', $line);
        $parts = array_map('trim', $parts);
        
        // إذا كان السطر الأول ولم نحدد أكواد المواد، نحاول استخراجها
        if ($first_line && empty($course_codes)) {
            // تحقق إذا كان سطر عناوين
            if (preg_match('/^(student|رقم|number|الرقم)/iu', $parts[0] ?? '')) {
                $course_codes = array_slice($parts, 1);
                $first_line = false;
                continue;
            }
        }
        $first_line = false;
        
        // السطر العادي: رقم الطالب ثم الدرجات
        $student_number = $parts[0] ?? '';
        if ($student_number === '') continue;
        
        // تخطي إذا يبدو كعنوان
        if (preg_match('/^(student|رقم|number|الرقم)/iu', $student_number)) continue;
        
        $grades = array_slice($parts, 1);
        $rows[] = ['student_number' => $student_number, 'grades' => $grades];
    }
    
    if (empty($rows)) {
        $message = '<div class="alert alert-error">لم يتم العثور على أي سطر صالح. تأكد من أن كل سطر يحتوي على: رقم جامعي ثم الدرجات (مفصولة بـ Tab أو فاصلة).</div>';
    } elseif (empty($course_codes)) {
        $message = '<div class="alert alert-error">لم يتم تحديد أكواد المواد. أدخل أكواد المواد في الحقل المخصص أو ضعها كسطر أول في البيانات.</div>';
    } else {
        // بناء CSV
        $csv_header = 'student_number,' . implode(',', $course_codes) . "\n";
        $csv_content = $csv_header;
        
        foreach ($rows as $r) {
            $num = str_replace(["\r", "\n", '"', ','], ['', '', '""', ''], $r['student_number']);
            $grade_values = [];
            foreach ($course_codes as $i => $code) {
                $val = isset($r['grades'][$i]) ? trim($r['grades'][$i]) : '';
                $val = str_replace(["\r", "\n", '"'], ['', '', '""'], $val);
                $grade_values[] = $val;
            }
            $csv_content .= $num . ',' . implode(',', $grade_values) . "\n";
        }
        
        if (!mb_check_encoding($csv_content, 'UTF-8')) {
            $csv_content = mb_convert_encoding($csv_content, 'UTF-8', 'auto');
        }
        
        $bom = "\xEF\xBB\xBF";
        $template_path = __DIR__ . '/templates/template_grades_paste.csv';
        @file_put_contents($template_path, $bom . $csv_content, LOCK_EX);
        
        $saved_count = count($rows);
        $download_url = 'templates/template_grades_paste.csv';
        $message = '<div class="alert alert-success">✓ تم إنشاء القالب بنجاح: <strong>' . $saved_count . '</strong> طالب × <strong>' . count($course_codes) . '</strong> مادة.<br>
        <a href="' . htmlspecialchars($download_url) . '" download class="btn btn-success" style="margin: 10px 5px 0 0;">📥 تحميل القالب</a>
        ثم اذهب إلى <a href="upload_grades_excel.php" class="btn btn-primary" style="margin: 10px 5px 0 0;">📤 رفع الدرجات</a></div>';
    }
}

include __DIR__ . '/includes/header.php';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لصق قائمة الدرجات وتوليد القالب</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .paste-container { max-width: 950px; margin: 20px auto; }
        .paste-hero { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 24px; border-radius: 12px; margin-bottom: 20px; }
        .paste-hero h1 { margin: 0 0 8px 0; font-size: 1.5rem; }
        .paste-hero p { margin: 0; opacity: 0.9; }
        .paste-card { background: white; border-radius: 12px; box-shadow: 0 2px 12px rgba(0,0,0,0.08); margin-bottom: 20px; overflow: hidden; }
        .paste-card-header { background: linear-gradient(135deg, #48bb78 0%, #38a169 100%); color: white; padding: 16px 20px; }
        .paste-card-header h2 { margin: 0; font-size: 1.1rem; }
        .paste-card-body { padding: 20px; }
        .paste-box { width: 100%; min-height: 220px; font-family: 'Courier New', monospace; font-size: 14px; padding: 12px; border: 2px solid #48bb78; border-radius: 8px; direction: ltr; text-align: left; }
        .paste-box:focus { border-color: #38a169; outline: none; box-shadow: 0 0 0 3px rgba(72,187,120,0.2); }
        .codes-input { width: 100%; padding: 10px 12px; border: 1px solid #e2e8f0; border-radius: 8px; font-size: 14px; direction: ltr; }
        .steps-box { background: #f0fff4; padding: 16px; border-radius: 8px; border-right: 4px solid #48bb78; margin-bottom: 16px; }
        .steps-box ol { margin: 10px 0 0 20px; padding: 0; }
        .steps-box li { margin: 6px 0; }
        .example-table { border-collapse: collapse; font-size: 12px; margin-top: 10px; direction: ltr; }
        .example-table th, .example-table td { border: 1px solid #e2e8f0; padding: 6px 10px; text-align: center; }
        .example-table th { background: #edf2f7; }
        .btn-generate { background: linear-gradient(135deg, #48bb78 0%, #38a169 100%); color: white !important; padding: 12px 24px; border: none; border-radius: 8px; font-size: 1rem; font-weight: 600; cursor: pointer; }
        .btn-generate:hover { opacity: 0.95; }
    </style>
</head>
<body>
<div class="container paste-container">
    <div class="paste-hero">
        <h1>📋 لصق قائمة الدرجات وتوليد القالب</h1>
        <p>الصق الدرجات من Excel أو أي جدول — سيتم توليد ملف CSV جاهز بترميز UTF-8 لرفعه مباشرة</p>
    </div>

    <?php echo $message; ?>

    <div class="paste-card">
        <div class="paste-card-header">
            <h2>📝 خطوات الاستخدام</h2>
        </div>
        <div class="paste-card-body">
            <div class="steps-box">
                <ol>
                    <li><strong>أدخل أكواد المواد</strong> في الحقل الأول (مثل: CS101, MATH101, ENG101) — أو ضعها كسطر أول في البيانات.</li>
                    <li><strong>انسخ الدرجات</strong> من Excel أو أي جدول — كل سطر = طالب واحد (رقم جامعي ثم الدرجات).</li>
                    <li><strong>الصق في الصندوق</strong> أدناه.</li>
                    <li>اضغط <strong>«توليد القالب»</strong> ثم حمّل الملف.</li>
                    <li>اذهب إلى <strong>رفع الدرجات من Excel</strong> واختر الفصل وارفع الملف.</li>
                </ol>
            </div>
            
            <p><strong>مثال على شكل البيانات:</strong></p>
            <table class="example-table">
                <tr><th>student_number</th><th>CS101</th><th>MATH101</th><th>ENG101</th></tr>
                <tr><td>11079388-23</td><td>85</td><td>90</td><td>75</td></tr>
                <tr><td>11079100-23</td><td>80</td><td>بد</td><td>غياب</td></tr>
            </table>
        </div>
    </div>

    <div class="paste-card">
        <div class="paste-card-header">
            <h2>📊 أدخل البيانات</h2>
        </div>
        <div class="paste-card-body">
            <form method="POST">
                <input type="hidden" name="action" value="generate">
                
                <div class="form-group" style="margin-bottom: 16px;">
                    <label for="course_codes"><strong>أكواد المواد (مفصولة بفاصلة أو مسافة):</strong></label>
                    <input type="text" name="course_codes" id="course_codes" class="codes-input" 
                           placeholder="مثال: CS101, MATH101, ENG101, PHYS101"
                           value="<?php echo isset($_POST['course_codes']) ? htmlspecialchars($_POST['course_codes']) : ''; ?>">
                    <small style="color: #718096;">اتركه فارغاً إذا كان السطر الأول في البيانات يحتوي على أكواد المواد</small>
                </div>
                
                <div class="form-group" style="margin-bottom: 16px;">
                    <label for="grades_list"><strong>الصق بيانات الدرجات هنا:</strong></label>
                    <textarea name="grades_list" id="grades_list" class="paste-box" placeholder="student_number	CS101	MATH101	ENG101
11079388-23	85	90	75
11079100-23	80	بد	غياب
11079544-23	92	88	95

أو بدون سطر العناوين (إذا أدخلت أكواد المواد أعلاه):
11079388-23	85	90	75
11079100-23	80	بد	غياب"><?php echo isset($_POST['grades_list']) ? htmlspecialchars($_POST['grades_list']) : ''; ?></textarea>
                </div>
                
                <button type="submit" class="btn btn-generate">📄 توليد القالب</button>
                <a href="upload_grades_excel.php" class="btn btn-secondary" style="margin-right: 10px;">العودة لرفع الدرجات</a>
                <a href="dashboard.php" class="btn btn-secondary">لوحة التحكم</a>
            </form>
        </div>
    </div>

    <p style="margin-top: 20px; color: #718096; font-size: 14px;">
        <strong>ملاحظة:</strong> الملف المُولَّد يُحفظ في <code>templates/template_grades_paste.csv</code> بترميز UTF-8 مع BOM، 
        لذا الدرجات النصية (بد، غياب) ستظهر بشكل صحيح.
    </p>
</div>

<?php include 'includes/footer.php'; ?>
</body>
</html>
