<?php
/**
 * صفحة عامة للطلاب للبحث عن نتائجهم
 * Public page for students to search their results
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';

$db = Database::getInstance();
$error = '';
$results = null;
$student_info = null;

// دالة لتحويل رقم الفصل إلى نص عربي
function getSemesterNameInArabic($number) {
    $names = [
        1 => 'الأول',
        2 => 'الثاني',
        3 => 'الثالث',
        4 => 'الرابع',
        5 => 'الخامس',
        6 => 'السادس',
        7 => 'السابع',
        8 => 'الثامن',
        9 => 'التاسع',
        10 => 'العاشر',
    ];
    
    $num = (int)$number;
    if (isset($names[$num])) {
        return $names[$num];
    }
    return $number; // إذا كان الرقم أكبر من 10، نعيده كما هو
}

// التحقق من وجود جدول published_results وإنشائه إذا لم يكن موجوداً
try {
    $table_exists = false;
    $tables = $db->fetchAll("SHOW TABLES LIKE 'published_results'");
    if (!empty($tables)) {
        $table_exists = true;
    }
    
    if (!$table_exists) {
        $db->query("
            CREATE TABLE IF NOT EXISTS published_results (
                id INT PRIMARY KEY AUTO_INCREMENT,
                semester_id INT NOT NULL,
                published_by INT NOT NULL,
                published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMP NOT NULL,
                is_active BOOLEAN DEFAULT TRUE,
                FOREIGN KEY (semester_id) REFERENCES semesters(id) ON DELETE CASCADE,
                FOREIGN KEY (published_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_semester_published (semester_id),
                INDEX idx_expires_at (expires_at),
                INDEX idx_is_active (is_active)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (Exception $e) {
    error_log("Published results table check: " . $e->getMessage());
}

// التحقق من وجود جدول published_annual_results وإنشائه إذا لم يكن موجوداً
try {
    $table_exists = false;
    $tables = $db->fetchAll("SHOW TABLES LIKE 'published_annual_results'");
    if (!empty($tables)) {
        $table_exists = true;
    }
    
    if (!$table_exists) {
        $db->query("
            CREATE TABLE IF NOT EXISTS published_annual_results (
                id INT PRIMARY KEY AUTO_INCREMENT,
                academic_year VARCHAR(50) NOT NULL,
                major_id INT NOT NULL,
                published_by INT NOT NULL,
                published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at DATETIME NOT NULL,
                is_active BOOLEAN DEFAULT TRUE,
                FOREIGN KEY (major_id) REFERENCES majors(id) ON DELETE CASCADE,
                FOREIGN KEY (published_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_annual_published (academic_year, major_id),
                INDEX idx_expires_at (expires_at),
                INDEX idx_is_active (is_active)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (Exception $e) {
    error_log("Published annual results table check: " . $e->getMessage());
}

// معالجة البحث
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['student_number'])) {
    $student_number = trim($_POST['student_number']);
    
    if (empty($student_number)) {
        $error = 'الرجاء إدخال الرقم الجامعي';
    } else {
        // البحث عن الطالب (الطلاب النشطين فقط وغير المحجوبة نتائجهم)
        $student = $db->fetchOne(
            "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name 
             FROM students s 
             LEFT JOIN majors m ON s.major_id = m.id 
             LEFT JOIN colleges c ON s.college_id = c.id 
             WHERE s.student_number = ? 
             AND s.status = 'active' 
             AND (s.results_blocked IS NULL OR s.results_blocked = 0)",
            [$student_number]
        );
        
        if (!$student) {
            // التحقق من سبب عدم العثور على الطالب
            $student_check = $db->fetchOne(
                "SELECT s.*, s.results_blocked 
                 FROM students s 
                 WHERE s.student_number = ?",
                [$student_number]
            );
            
            if ($student_check) {
                if ($student_check['status'] !== 'active') {
                    $error = 'الرقم الجامعي غير موجود في النظام أو الطالب غير نشط';
                } elseif (isset($student_check['results_blocked']) && $student_check['results_blocked']) {
                    $error = 'نتائج هذا الطالب محجوبة حالياً';
                } else {
                    $error = 'الرقم الجامعي غير موجود في النظام أو الطالب غير نشط';
                }
            } else {
                $error = 'الرقم الجامعي غير موجود في النظام';
            }
        } else {
            $student_info = $student;
            
            // البحث عن النتائج المنشورة للطالب (الفصلية والسنوية)
            $published_semesters = $db->fetchAll(
                "SELECT pr.*, s.id as semester_id, s.semester_number, s.academic_year, s.major_id
                 FROM published_results pr
                 INNER JOIN semesters s ON pr.semester_id = s.id
                 WHERE pr.is_active = 1 
                 AND pr.expires_at > NOW()
                 AND s.major_id = ?
                 ORDER BY s.academic_year DESC, s.semester_number DESC",
                [$student['major_id']]
            );
            
            // البحث عن النتائج السنوية المنشورة
            $published_annual = [];
            try {
                $published_annual = $db->fetchAll(
                    "SELECT * FROM published_annual_results
                     WHERE is_active = 1 
                     AND expires_at > NOW()
                     AND major_id = ?
                     ORDER BY academic_year DESC",
                    [$student['major_id']]
                );
            } catch (Exception $e) {
                // إذا لم يكن الجدول موجوداً، نستمر بدون النتائج السنوية
                error_log("Error fetching published annual results: " . $e->getMessage());
                $published_annual = [];
            }
            
            if (empty($published_semesters) && empty($published_annual)) {
                $error = 'لا توجد نتائج منشورة حالياً';
            } else {
                // جلب النتائج لكل فصل منشور
                $results = [];
                foreach ($published_semesters as $semester) {
                    // جلب درجات الطالب في هذا الفصل
                    $grades = $db->fetchAll(
                        "SELECT g.*, c.code as course_code, c.name_ar as course_name, c.credit_hours, c.is_alternative,
                         COALESCE(g.calibrated_marks, g.marks) as display_marks,
                         COALESCE(g.calibrated_letter_grade, g.letter_grade) as display_letter_grade,
                         COALESCE(g.calibrated_points, g.points) as display_points
                         FROM grades g
                         INNER JOIN courses c ON g.course_id = c.id
                         WHERE g.student_id = ? AND g.semester_id = ?
                         ORDER BY c.code",
                        [$student['id'], $semester['semester_id']]
                    );
                    
                    // حساب المعدل الفصلي - نفس الصيغة المستخدمة في GradeCalculator
                    // الصيغة: مجموع (الدرجة/20*الساعات المعتمدة) / مجموع الساعات المعتمدة
                    // المعدل من 5
                    $total_points = 0;
                    $total_credit_hours = 0;
                    
                    foreach ($grades as $grade) {
                        // تجاهل مواد البديل في حساب المعدل
                        if (isset($grade['is_alternative']) && $grade['is_alternative']) {
                            continue;
                        }
                        
                        // تجاهل الدرجات التي ليس لها marks رقمية (مثل "بد", "غياب", "غش")
                        $marks = $grade['display_marks'];
                        $letter_grade = isset($grade['display_letter_grade']) ? trim($grade['display_letter_grade']) : '';
                        
                        // التحقق من "بد" (بديل)
                        if ($letter_grade === 'بد' || $letter_grade === 'ب' || strpos($letter_grade, 'بد') !== false) {
                            continue;
                        }
                        
                        // إذا كانت marks رقمية، استخدمها مباشرة
                        if ($marks !== null && is_numeric($marks)) {
                            $marks_value = (float)$marks;
                            $credit_hours = (float)$grade['credit_hours'];
                            
                            // الصيغة: (الدرجة/20) * الساعات المعتمدة
                            $total_points += ($marks_value / 20.0) * $credit_hours;
                            $total_credit_hours += $credit_hours;
                        }
                        // تجاهل الدرجات النصية (غياب، غش، إلخ)
                    }
                    
                    $gpa = $total_credit_hours > 0 ? $total_points / $total_credit_hours : 0;
                    $gpa = round($gpa, 2);
                    
                    // جلب المعدل الفصلي المحفوظ إن وجد (يستخدم للتحقق فقط)
                    $semester_gpa = $db->fetchOne(
                        "SELECT * FROM semester_gpas WHERE student_id = ? AND semester_id = ?",
                        [$student['id'], $semester['semester_id']]
                    );
                    
                    // استخدام المعدل المحفوظ إذا كان متوفراً (لضمان الدقة)
                    if ($semester_gpa && $semester_gpa['gpa'] > 0) {
                        $gpa = round((float)$semester_gpa['gpa'], 2);
                    }
                    
                    // تحديد التقدير النصي بناءً على المعدل من 5
                    // نفس المقياس المستخدم في نتيجة الطلاب
                    $text_grade = '';
                    if ($gpa >= 4.0 && $gpa <= 5.0) {
                        $text_grade = 'ممتاز';
                    } elseif ($gpa >= 3.5 && $gpa < 4.0) {
                        $text_grade = 'جيد جداً';
                    } elseif ($gpa >= 3.0 && $gpa < 3.5) {
                        $text_grade = 'جيد';
                    } elseif ($gpa >= 2.5 && $gpa < 3.0) {
                        $text_grade = 'مقبول';
                    } elseif ($gpa >= 2.0 && $gpa < 2.5) {
                        $text_grade = 'مقبول';
                    } elseif ($gpa >= 1.5 && $gpa < 2.0) {
                        $text_grade = 'إعادة السنة';
                    } elseif ($gpa > 0 && $gpa < 1.5) {
                        $text_grade = 'فصل';
                    } else {
                        $text_grade = '-';
                    }
                    
                    $results[] = [
                        'type' => 'semester',
                        'semester' => $semester,
                        'grades' => $grades,
                        'gpa' => $gpa,
                        'text_grade' => $text_grade,
                        'total_credits' => $total_credit_hours,
                        'expires_at' => $semester['expires_at']
                    ];
                }
                
                // جلب وعرض النتائج السنوية المنشورة
                foreach ($published_annual as $annual_pub) {
                    $academic_year = $annual_pub['academic_year'];
                    
                    // جلب الفصول الدراسية في هذا العام الدراسي لهذا التخصص
                    $semesters_in_year = $db->fetchAll(
                        "SELECT * FROM semesters 
                         WHERE academic_year = ? AND major_id = ? 
                         ORDER BY semester_number",
                        [$academic_year, $student['major_id']]
                    );
                    
                    if (count($semesters_in_year) < 2) {
                        continue; // النتيجة السنوية تحتاج فصلين دراسيين على الأقل
                    }
                    
                    $semester1_id = null;
                    $semester2_id = null;
                    foreach ($semesters_in_year as $sem) {
                        if ($sem['semester_number'] == 1) {
                            $semester1_id = $sem['id'];
                        } elseif ($sem['semester_number'] == 2) {
                            $semester2_id = $sem['id'];
                        }
                    }
                    
                    // جلب جميع درجات الطالب في العام الدراسي
                    $all_grades = [];
                    $total_points = 0;
                    $total_credit_hours = 0;
                    
                    foreach ($semesters_in_year as $sem) {
                        $sem_grades = $db->fetchAll(
                            "SELECT g.*, c.code as course_code, c.name_ar as course_name, c.credit_hours, c.is_alternative,
                             COALESCE(g.calibrated_marks, g.marks) as display_marks,
                             COALESCE(g.calibrated_letter_grade, g.letter_grade) as display_letter_grade
                             FROM grades g
                             INNER JOIN courses c ON g.course_id = c.id
                             WHERE g.student_id = ? AND g.semester_id = ?
                             ORDER BY c.code",
                            [$student['id'], $sem['id']]
                        );
                        
                        foreach ($sem_grades as $grade) {
                            $all_grades[] = $grade;
                            
                            // حساب المعدل السنوي - نفس منطق النتائج الفصلية
                            if (isset($grade['is_alternative']) && $grade['is_alternative']) {
                                continue;
                            }
                            
                            $marks = $grade['display_marks'];
                            $letter_grade = isset($grade['display_letter_grade']) ? trim($grade['display_letter_grade']) : '';
                            
                            if ($letter_grade === 'بد' || $letter_grade === 'ب' || strpos($letter_grade, 'بد') !== false) {
                                continue;
                            }
                            
                            if ($marks !== null && is_numeric($marks)) {
                                $marks_value = (float)$marks;
                                $credit_hours = (float)$grade['credit_hours'];
                                $total_points += ($marks_value / 20.0) * $credit_hours;
                                $total_credit_hours += $credit_hours;
                            }
                        }
                    }
                    
                    // حساب المعدل السنوي
                    $annual_gpa = $total_credit_hours > 0 ? $total_points / $total_credit_hours : 0;
                    $annual_gpa = round($annual_gpa, 2);
                    
                    // جلب المعدلات الفصلية
                    $gpa_sem1 = 0;
                    $gpa_sem2 = 0;
                    if ($semester1_id) {
                        $sem1_gpa = $db->fetchOne(
                            "SELECT gpa FROM semester_gpas WHERE student_id = ? AND semester_id = ?",
                            [$student['id'], $semester1_id]
                        );
                        $gpa_sem1 = $sem1_gpa ? (float)$sem1_gpa['gpa'] : 0;
                    }
                    if ($semester2_id) {
                        $sem2_gpa = $db->fetchOne(
                            "SELECT gpa FROM semester_gpas WHERE student_id = ? AND semester_id = ?",
                            [$student['id'], $semester2_id]
                        );
                        $gpa_sem2 = $sem2_gpa ? (float)$sem2_gpa['gpa'] : 0;
                    }
                    
                    // حساب التقدير النصي
                    $text_grade = '';
                    if ($annual_gpa >= 4.0 && $annual_gpa <= 5.0) {
                        $text_grade = 'ممتاز';
                    } elseif ($annual_gpa >= 3.5 && $annual_gpa < 4.0) {
                        $text_grade = 'جيد جداً';
                    } elseif ($annual_gpa >= 3.0 && $annual_gpa < 3.5) {
                        $text_grade = 'جيد';
                    } elseif ($annual_gpa >= 2.5 && $annual_gpa < 3.0) {
                        $text_grade = 'مقبول';
                    } elseif ($annual_gpa >= 2.0 && $annual_gpa < 2.5) {
                        $text_grade = 'مقبول';
                    } elseif ($annual_gpa >= 1.5 && $annual_gpa < 2.0) {
                        $text_grade = 'إعادة السنة';
                    } elseif ($annual_gpa > 0 && $annual_gpa < 1.5) {
                        $text_grade = 'فصل';
                    } else {
                        $text_grade = '-';
                    }
                    
                    $results[] = [
                        'type' => 'annual',
                        'academic_year' => $academic_year,
                        'grades' => $all_grades,
                        'gpa_sem1' => $gpa_sem1,
                        'gpa_sem2' => $gpa_sem2,
                        'annual_gpa' => $annual_gpa,
                        'text_grade' => $text_grade,
                        'total_credits' => $total_credit_hours,
                        'expires_at' => $annual_pub['expires_at']
                    ];
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>البحث عن النتائج - <?php echo SITE_NAME; ?></title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            margin: 0;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .search-card {
            background: white;
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            margin-bottom: 30px;
        }
        .search-card h1 {
            color: #333;
            text-align: center;
            margin-bottom: 30px;
            font-size: 2em;
        }
        .search-form {
            display: flex;
            gap: 15px;
            max-width: 600px;
            margin: 0 auto;
        }
        .search-form input {
            flex: 1;
            padding: 15px 20px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        .search-form input:focus {
            outline: none;
            border-color: #667eea;
        }
        .search-form button {
            padding: 15px 40px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            cursor: pointer;
            transition: background 0.3s;
        }
        .search-form button:hover {
            background: #5568d3;
        }
        .error {
            background: #f44336;
            color: white;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            text-align: center;
        }
        .results-card {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            margin-bottom: 20px;
        }
        .student-info {
            background: #f5f5f5;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
        }
        .student-info h2 {
            color: #333;
            margin-bottom: 15px;
        }
        .student-info p {
            margin: 8px 0;
            font-size: 16px;
        }
        .semester-results {
            margin-bottom: 40px;
        }
        .semester-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .semester-header h3 {
            margin: 0 0 10px 0;
        }
        .semester-header p {
            margin: 5px 0;
            opacity: 0.9;
        }
        .grades-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        .grades-table th,
        .grades-table td {
            padding: 12px;
            text-align: right;
            border-bottom: 1px solid #ddd;
        }
        .grades-table th {
            background: #f5f5f5;
            font-weight: bold;
            color: #333;
        }
        .grades-table tr:hover {
            background: #f9f9f9;
        }
        .gpa-summary {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 10px;
            margin-top: 20px;
            display: flex;
            justify-content: space-around;
            flex-wrap: wrap;
        }
        .gpa-item {
            text-align: center;
        }
        .gpa-item strong {
            display: block;
            font-size: 24px;
            color: #667eea;
            margin-bottom: 5px;
        }
        .gpa-item span {
            color: #666;
            font-size: 14px;
        }
        .expiry-notice {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            text-align: center;
            color: #856404;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        @media print {
            body {
                background: white;
            }
            .search-card {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="search-card">
            <h1>🔍 البحث عن النتائج</h1>
            <form method="POST" class="search-form">
                <input 
                    type="text" 
                    name="student_number" 
                    placeholder="أدخل الرقم الجامعي" 
                    required
                    value="<?php echo isset($_POST['student_number']) ? htmlspecialchars($_POST['student_number']) : ''; ?>"
                >
                <button type="submit">بحث</button>
            </form>
        </div>
        
        <?php if ($error): ?>
            <div class="error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if ($student_info && $results): ?>
            <div class="results-card">
                <div class="student-info">
                    <h2>معلومات الطالب</h2>
                    <p><strong>الاسم:</strong> <?php echo htmlspecialchars($student_info['full_name_ar']); ?></p>
                    <p><strong>التخصص:</strong> <?php echo htmlspecialchars($student_info['major_name']); ?></p>
                    <p><strong>الكلية:</strong> <?php echo htmlspecialchars($student_info['college_name']); ?></p>
                </div>
                
                <?php foreach ($results as $result): ?>
                    <?php if ($result['type'] === 'semester'): ?>
                        <!-- عرض النتائج الفصلية -->
                        <div class="semester-results">
                            <div class="semester-header">
                                <h3>الفصل الدراسي <?php echo htmlspecialchars(getSemesterNameInArabic($result['semester']['semester_number'])); ?></h3>
                                <p>العام الدراسي: <?php echo htmlspecialchars($result['semester']['academic_year']); ?></p>
                            </div>
                            
                            <?php if (!empty($result['grades'])): ?>
                                <table class="grades-table">
                                    <thead>
                                        <tr>
                                            <th>م</th>
                                            <th>اسم المادة</th>
                                            <th>الساعات المعتمدة</th>
                                            <th>التقدير</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $counter = 1; foreach ($result['grades'] as $grade): ?>
                                            <tr>
                                                <td><?php echo $counter++; ?></td>
                                                <td><?php echo htmlspecialchars($grade['course_name']); ?></td>
                                                <td><?php echo htmlspecialchars($grade['credit_hours']); ?></td>
                                                <td><?php echo htmlspecialchars(cleanLetterGrade($grade['display_letter_grade'] ?? '-')); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                                
                                <div class="gpa-summary">
                                    <div class="gpa-item">
                                        <strong><?php echo number_format($result['gpa'], 2); ?></strong>
                                        <span>المعدل الفصلي (GPA)</span>
                                    </div>
                                    <div class="gpa-item">
                                        <strong><?php echo htmlspecialchars($result['text_grade']); ?></strong>
                                        <span>التقدير</span>
                                    </div>
                                    <div class="gpa-item">
                                        <strong><?php echo number_format($result['total_credits'], 1); ?></strong>
                                        <span>إجمالي الساعات المعتمدة</span>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="no-results">لا توجد درجات مسجلة في هذا الفصل</div>
                            <?php endif; ?>
                            
                            <div class="expiry-notice">
                                ⏰ هذه النتائج متاحة حتى: <?php echo date('Y-m-d H:i', strtotime($result['expires_at'])); ?>
                            </div>
                        </div>
                    <?php elseif ($result['type'] === 'annual'): ?>
                        <!-- عرض النتائج السنوية -->
                        <div class="semester-results">
                            <div class="semester-header">
                                <h3>النتائج السنوية</h3>
                                <p>العام الدراسي: <?php echo htmlspecialchars($result['academic_year']); ?></p>
                            </div>
                            
                            <?php if (!empty($result['grades'])): ?>
                                <table class="grades-table">
                                    <thead>
                                        <tr>
                                            <th>م</th>
                                            <th>اسم المادة</th>
                                            <th>الساعات المعتمدة</th>
                                            <th>التقدير</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $counter = 1; foreach ($result['grades'] as $grade): ?>
                                            <tr>
                                                <td><?php echo $counter++; ?></td>
                                                <td><?php echo htmlspecialchars($grade['course_name']); ?></td>
                                                <td><?php echo htmlspecialchars($grade['credit_hours']); ?></td>
                                                <td><?php echo htmlspecialchars(cleanLetterGrade($grade['display_letter_grade'] ?? '-')); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                                
                                <div class="gpa-summary">
                                    <?php if ($result['gpa_sem1'] > 0 || $result['gpa_sem2'] > 0): ?>
                                        <div class="gpa-item">
                                            <strong><?php echo $result['gpa_sem1'] > 0 ? number_format($result['gpa_sem1'], 2) : '-'; ?></strong>
                                            <span>المعدل الفصلي الأول</span>
                                        </div>
                                        <div class="gpa-item">
                                            <strong><?php echo $result['gpa_sem2'] > 0 ? number_format($result['gpa_sem2'], 2) : '-'; ?></strong>
                                            <span>المعدل الفصلي الثاني</span>
                                        </div>
                                    <?php endif; ?>
                                    <div class="gpa-item">
                                        <strong><?php echo number_format($result['annual_gpa'], 2); ?></strong>
                                        <span>المعدل السنوي (GPA)</span>
                                    </div>
                                    <div class="gpa-item">
                                        <strong><?php echo htmlspecialchars($result['text_grade']); ?></strong>
                                        <span>التقدير</span>
                                    </div>
                                    <div class="gpa-item">
                                        <strong><?php echo number_format($result['total_credits'], 1); ?></strong>
                                        <span>إجمالي الساعات المعتمدة</span>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="no-results">لا توجد درجات مسجلة في هذا العام الدراسي</div>
                            <?php endif; ?>
                            
                            <div class="expiry-notice">
                                ⏰ هذه النتائج متاحة حتى: <?php echo date('Y-m-d H:i', strtotime($result['expires_at'])); ?>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
                
                <div style="text-align: center; margin-top: 30px;">
                    <button onclick="window.print()" style="padding: 15px 40px; background: #667eea; color: white; border: none; border-radius: 8px; font-size: 16px; cursor: pointer;">
                        طباعة النتائج
                    </button>
                </div>
            </div>
        <?php elseif ($student_info && empty($results)): ?>
            <div class="results-card">
                <div class="no-results">
                    <h3>لا توجد نتائج منشورة حالياً</h3>
                    <p>الرجاء المحاولة لاحقاً</p>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>

