<?php
/**
 * صفحة إدارية لنشر/إلغاء نشر نتائج الفصول الدراسية
 * Admin page to publish/unpublish semester results
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$message = '';
$message_type = '';

// دالة للتحقق من وجود جدول published_annual_results وإنشائه
function ensurePublishedAnnualResultsTable($db) {
    // محاولة إنشاء الجدول مباشرة (IF NOT EXISTS يضمن عدم إنشائه مرة أخرى إذا كان موجوداً)
    try {
        $db->query("
            CREATE TABLE IF NOT EXISTS published_annual_results (
                id INT PRIMARY KEY AUTO_INCREMENT,
                academic_year VARCHAR(50) NOT NULL,
                major_id INT NOT NULL,
                published_by INT NOT NULL,
                published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at DATETIME NOT NULL,
                is_active BOOLEAN DEFAULT TRUE,
                FOREIGN KEY (major_id) REFERENCES majors(id) ON DELETE CASCADE,
                FOREIGN KEY (published_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_annual_published (academic_year, major_id),
                INDEX idx_expires_at (expires_at),
                INDEX idx_is_active (is_active)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        return true;
    } catch (Exception $e) {
        error_log("Failed to ensure published_annual_results table exists: " . $e->getMessage());
        // محاولة إنشاء الجدول بدون Foreign Keys في حالة وجود مشكلة
        try {
            $db->query("
                CREATE TABLE IF NOT EXISTS published_annual_results (
                    id INT PRIMARY KEY AUTO_INCREMENT,
                    academic_year VARCHAR(50) NOT NULL,
                    major_id INT NOT NULL,
                    published_by INT NOT NULL,
                    published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    expires_at DATETIME NOT NULL,
                    is_active BOOLEAN DEFAULT TRUE,
                    UNIQUE KEY unique_annual_published (academic_year, major_id),
                    INDEX idx_expires_at (expires_at),
                    INDEX idx_is_active (is_active),
                    INDEX idx_major_id (major_id),
                    INDEX idx_published_by (published_by)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
            error_log("Created published_annual_results table without foreign keys");
            return true;
        } catch (Exception $e2) {
            error_log("Failed to create published_annual_results table even without foreign keys: " . $e2->getMessage());
            return false;
        }
    }
}

// التحقق من وجود جدول published_results وإنشائه إذا لم يكن موجوداً
try {
    $table_exists = $db->fetchOne("SHOW TABLES LIKE 'published_results'");
    if (!$table_exists) {
        $db->query("
            CREATE TABLE IF NOT EXISTS published_results (
                id INT PRIMARY KEY AUTO_INCREMENT,
                semester_id INT NOT NULL,
                published_by INT NOT NULL,
                published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                expires_at TIMESTAMP NOT NULL,
                is_active BOOLEAN DEFAULT TRUE,
                FOREIGN KEY (semester_id) REFERENCES semesters(id) ON DELETE CASCADE,
                FOREIGN KEY (published_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_semester_published (semester_id),
                INDEX idx_expires_at (expires_at),
                INDEX idx_is_active (is_active)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (Exception $e) {
    error_log("Published results table check: " . $e->getMessage());
}

// التحقق من وجود عمود results_blocked في جدول students وإضافته إذا لم يكن موجوداً
try {
    $columns = $db->fetchAll("SHOW COLUMNS FROM students");
    $existing_columns = array_column($columns, 'Field');
    
    if (!in_array('results_blocked', $existing_columns)) {
        $db->query("ALTER TABLE students ADD COLUMN results_blocked BOOLEAN DEFAULT FALSE AFTER status");
    }
} catch (Exception $e) {
    error_log("Students columns check error: " . $e->getMessage());
}

// معالجة النشر/إلغاء النشر
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // التأكد من وجود جميع الجداول المطلوبة قبل المعالجة
    ensurePublishedAnnualResultsTable($db);
    if (isset($_POST['publish'])) {
        $semester_id = (int)$_POST['semester_id'];
        $days = (int)($_POST['days'] ?? 7); // افتراضياً أسبوع واحد
        
        if ($semester_id > 0 && $days > 0) {
            try {
                $user = $auth->getCurrentUser();
                $expires_at = date('Y-m-d H:i:s', strtotime("+{$days} days"));
                
                // التحقق من وجود نشر سابق
                $existing = $db->fetchOne(
                    "SELECT * FROM published_results WHERE semester_id = ?",
                    [$semester_id]
                );
                
                if ($existing) {
                    // تحديث النشر الموجود
                    $db->query(
                        "UPDATE published_results 
                         SET published_by = ?, published_at = NOW(), expires_at = ?, is_active = 1 
                         WHERE semester_id = ?",
                        [$user['id'], $expires_at, $semester_id]
                    );
                    $message = 'تم تحديث نشر النتائج بنجاح';
                } else {
                    // نشر جديد
                    $db->query(
                        "INSERT INTO published_results (semester_id, published_by, expires_at, is_active) 
                         VALUES (?, ?, ?, 1)",
                        [$semester_id, $user['id'], $expires_at]
                    );
                    $message = 'تم نشر النتائج بنجاح';
                }
                $message_type = 'success';
            } catch (Exception $e) {
                $message = 'حدث خطأ أثناء النشر: ' . $e->getMessage();
                $message_type = 'error';
            }
        }
    } elseif (isset($_POST['unpublish'])) {
        $semester_id = (int)$_POST['semester_id'];
        
        try {
            $db->query(
                "UPDATE published_results SET is_active = 0 WHERE semester_id = ?",
                [$semester_id]
            );
            $message = 'تم إلغاء نشر النتائج بنجاح';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'حدث خطأ أثناء إلغاء النشر: ' . $e->getMessage();
            $message_type = 'error';
        }
    } elseif (isset($_POST['toggle_student_status'])) {
        // معالجة تغيير حالة الطالب (إيقاف/تفعيل النشاط)
        $student_id = (int)$_POST['student_id'];
        $new_status = $_POST['new_status'] ?? 'active';
        
        try {
            $student = $db->fetchOne("SELECT * FROM students WHERE id = ?", [$student_id]);
            if (!$student) {
                $message = 'الطالب غير موجود';
                $message_type = 'error';
            } else {
                $db->query(
                    "UPDATE students SET status = ? WHERE id = ?",
                    [$new_status, $student_id]
                );
                $status_text = $new_status === 'active' ? 'نشط' : 'موقوف';
                $message = "تم تغيير حالة الطالب ({$student['full_name_ar']}) إلى: {$status_text}";
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'حدث خطأ أثناء تغيير حالة الطالب: ' . $e->getMessage();
            $message_type = 'error';
        }
    } elseif (isset($_POST['toggle_results_block'])) {
        // معالجة حجب/فك حجب النتائج
        $student_id = (int)$_POST['student_id'];
        $block_status = isset($_POST['block']) ? (int)$_POST['block'] : 0;
        
        try {
            $student = $db->fetchOne("SELECT * FROM students WHERE id = ?", [$student_id]);
            if (!$student) {
                $message = 'الطالب غير موجود';
                $message_type = 'error';
            } else {
                $db->query(
                    "UPDATE students SET results_blocked = ? WHERE id = ?",
                    [$block_status, $student_id]
                );
                $block_text = $block_status ? 'محجوبة' : 'غير محجوبة';
                $message = "تم " . ($block_status ? 'حجب' : 'فك حجب') . " نتائج الطالب ({$student['full_name_ar']}) - النتائج الآن: {$block_text}";
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'حدث خطأ أثناء حجب/فك حجب النتائج: ' . $e->getMessage();
            $message_type = 'error';
        }
    } elseif (isset($_POST['publish_annual'])) {
        // التأكد من وجود الجدول قبل النشر (دون إيقاف العملية إذا فشل)
        ensurePublishedAnnualResultsTable($db);
        
        // معالجة نشر النتائج السنوية
        $academic_year = trim($_POST['academic_year'] ?? '');
        $major_id = (int)($_POST['major_id'] ?? 0);
        $days = (int)($_POST['days'] ?? 7);
        
        if (!empty($academic_year) && $major_id > 0 && $days > 0) {
            try {
                $user = $auth->getCurrentUser();
                $expires_at = date('Y-m-d H:i:s', strtotime("+{$days} days"));
                
                // التحقق من وجود نشر سابق
                $existing = $db->fetchOne(
                    "SELECT * FROM published_annual_results WHERE academic_year = ? AND major_id = ?",
                    [$academic_year, $major_id]
                );
                
                if ($existing) {
                    // تحديث النشر الموجود
                    $db->query(
                        "UPDATE published_annual_results 
                         SET published_by = ?, published_at = NOW(), expires_at = ?, is_active = 1 
                         WHERE academic_year = ? AND major_id = ?",
                        [$user['id'], $expires_at, $academic_year, $major_id]
                    );
                    $message = 'تم تحديث نشر النتائج السنوية بنجاح';
                } else {
                    // نشر جديد
                    $db->query(
                        "INSERT INTO published_annual_results (academic_year, major_id, published_by, expires_at, is_active) 
                         VALUES (?, ?, ?, ?, 1)",
                        [$academic_year, $major_id, $user['id'], $expires_at]
                    );
                    $message = 'تم نشر النتائج السنوية بنجاح';
                }
                $message_type = 'success';
            } catch (Exception $e) {
                $message = 'حدث خطأ أثناء نشر النتائج السنوية: ' . $e->getMessage();
                $message_type = 'error';
            }
        } else {
            $message = 'الرجاء ملء جميع الحقول المطلوبة';
            $message_type = 'error';
        }
    } elseif (isset($_POST['unpublish_annual'])) {
        // معالجة إلغاء نشر النتائج السنوية
        $annual_id = (int)$_POST['annual_id'];
        
        try {
            $db->query(
                "UPDATE published_annual_results SET is_active = 0 WHERE id = ?",
                [$annual_id]
            );
            $message = 'تم إلغاء نشر النتائج السنوية بنجاح';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'حدث خطأ أثناء إلغاء نشر النتائج السنوية: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// التأكد من وجود جدول published_annual_results قبل جلب البيانات
ensurePublishedAnnualResultsTable($db);

// جلب الفصول الدراسية
$user = $auth->getCurrentUser();
$college_id = $user['college_id'] ?? null;

if ($auth->hasRole('admin')) {
    // المدير العام يمكنه رؤية جميع الفصول
    $semesters_query = "
        SELECT s.*, m.name_ar as major_name, m.code as major_code, c.name_ar as college_name,
               (SELECT COUNT(*) FROM grades WHERE semester_id = s.id) as grades_count,
               pr.id as published_id, pr.is_active as is_published, pr.expires_at
        FROM semesters s
        INNER JOIN majors m ON s.major_id = m.id
        INNER JOIN colleges c ON m.college_id = c.id
        LEFT JOIN published_results pr ON s.id = pr.semester_id
        ORDER BY s.academic_year DESC, s.semester_number DESC
    ";
    $semesters = $db->fetchAll($semesters_query);
} else {
    // مدير الكلية يمكنه رؤية فصول كليته فقط
    $semesters_query = "
        SELECT s.*, m.name_ar as major_name, m.code as major_code, c.name_ar as college_name,
               (SELECT COUNT(*) FROM grades WHERE semester_id = s.id) as grades_count,
               pr.id as published_id, pr.is_active as is_published, pr.expires_at
        FROM semesters s
        INNER JOIN majors m ON s.major_id = m.id
        INNER JOIN colleges c ON m.college_id = c.id
        LEFT JOIN published_results pr ON s.id = pr.semester_id
        WHERE c.id = ?
        ORDER BY s.academic_year DESC, s.semester_number DESC
    ";
    $semesters = $db->fetchAll($semesters_query, [$college_id]);
}

// جلب النتائج السنوية المتاحة
$annual_results = [];
if ($auth->hasRole('admin')) {
    // المدير العام يمكنه رؤية جميع السنوات الدراسية والتخصصات
    $annual_query = "
        SELECT DISTINCT s.academic_year, s.major_id, m.name_ar as major_name, 
               m.code as major_code, c.name_ar as college_name, c.id as college_id,
               (SELECT COUNT(DISTINCT st.id) 
                FROM students st 
                INNER JOIN grades g1 ON g1.student_id = st.id
                INNER JOIN semesters s1 ON g1.semester_id = s1.id
                WHERE s1.academic_year = s.academic_year AND s1.major_id = s.major_id) as students_count
        FROM semesters s
        INNER JOIN majors m ON s.major_id = m.id
        INNER JOIN colleges c ON m.college_id = c.id
        WHERE s.academic_year IS NOT NULL AND s.academic_year != ''
        ORDER BY s.academic_year DESC, m.name_ar
    ";
    $annual_data = $db->fetchAll($annual_query);
} else {
    // مدير الكلية يمكنه رؤية سنوات كليته فقط
    $annual_query = "
        SELECT DISTINCT s.academic_year, s.major_id, m.name_ar as major_name, 
               m.code as major_code, c.name_ar as college_name, c.id as college_id,
               (SELECT COUNT(DISTINCT st.id) 
                FROM students st 
                INNER JOIN grades g1 ON g1.student_id = st.id
                INNER JOIN semesters s1 ON g1.semester_id = s1.id
                WHERE s1.academic_year = s.academic_year AND s1.major_id = s.major_id) as students_count
        FROM semesters s
        INNER JOIN majors m ON s.major_id = m.id
        INNER JOIN colleges c ON m.college_id = c.id
        WHERE c.id = ? AND s.academic_year IS NOT NULL AND s.academic_year != ''
        ORDER BY s.academic_year DESC, m.name_ar
    ";
    $annual_data = $db->fetchAll($annual_query, [$college_id]);
}

// جلب حالة النشر للنتائج السنوية
$annual_results = [];
foreach ($annual_data as $annual) {
    // التحقق من وجود الجدول قبل الاستعلام
    try {
        $published_annual = $db->fetchOne(
            "SELECT * FROM published_annual_results 
             WHERE academic_year = ? AND major_id = ?",
            [$annual['academic_year'], $annual['major_id']]
        );
    } catch (Exception $e) {
        // إذا لم يكن الجدول موجوداً، نستمر بدون بيانات النشر
        $published_annual = null;
        error_log("Error fetching published annual results: " . $e->getMessage());
    }
    
    $annual_results[] = [
        'academic_year' => $annual['academic_year'],
        'major_id' => $annual['major_id'],
        'major_name' => $annual['major_name'],
        'major_code' => $annual['major_code'],
        'college_name' => $annual['college_name'],
        'college_id' => $annual['college_id'],
        'students_count' => $annual['students_count'] ?? 0,
        'published_id' => $published_annual['id'] ?? null,
        'is_published' => $published_annual['is_active'] ?? false,
        'expires_at' => $published_annual['expires_at'] ?? null
    ];
}

// تنظيف النتائج المنتهية الصلاحية تلقائياً
try {
    $db->query("UPDATE published_results SET is_active = 0 WHERE expires_at < NOW() AND is_active = 1");
} catch (Exception $e) {
    error_log("Auto-expire published results: " . $e->getMessage());
}
try {
    $db->query("UPDATE published_annual_results SET is_active = 0 WHERE expires_at < NOW() AND is_active = 1");
} catch (Exception $e) {
    error_log("Auto-expire published annual results: " . $e->getMessage());
}

include 'includes/header.php';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نشر النتائج - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .publish-form {
            background: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            align-items: flex-end;
        }
        .publish-form select,
        .publish-form input {
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        .publish-form select {
            flex: 1;
        }
        .publish-form input[type="number"] {
            width: 100px;
        }
        .status-badge {
            display: inline-block;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        .status-published {
            background: #4caf50;
            color: white;
        }
        .status-expired {
            background: #f44336;
            color: white;
        }
        .status-not-published {
            background: #999;
            color: white;
        }
        .expires-info {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>نشر النتائج على الموقع</h1>
        
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?>" style="margin-bottom: 20px;">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-header">
                <h2>الفصول الدراسية</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>الفصل</th>
                            <th>العام الدراسي</th>
                            <th>التخصص</th>
                            <th>الكلية</th>
                            <th>عدد الدرجات</th>
                            <th>حالة النشر</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($semesters)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center; padding: 20px;">
                                    لا توجد فصول دراسية
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($semesters as $semester): ?>
                                <tr>
                                    <td>الفصل <?php echo htmlspecialchars($semester['semester_number']); ?></td>
                                    <td><?php echo htmlspecialchars($semester['academic_year']); ?></td>
                                    <td><?php echo htmlspecialchars($semester['major_name']); ?></td>
                                    <td><?php echo htmlspecialchars($semester['college_name']); ?></td>
                                    <td><?php echo htmlspecialchars($semester['grades_count']); ?></td>
                                    <td>
                                        <?php if ($semester['is_published'] && strtotime($semester['expires_at']) > time()): ?>
                                            <span class="status-badge status-published">منشور</span>
                                            <div class="expires-info">
                                                ينتهي في: <?php echo date('Y-m-d H:i', strtotime($semester['expires_at'])); ?>
                                            </div>
                                        <?php elseif ($semester['is_published'] && strtotime($semester['expires_at']) <= time()): ?>
                                            <span class="status-badge status-expired">منتهي الصلاحية</span>
                                        <?php else: ?>
                                            <span class="status-badge status-not-published">غير منشور</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($semester['is_published'] && strtotime($semester['expires_at']) > time()): ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="semester_id" value="<?php echo $semester['id']; ?>">
                                                <button type="submit" name="unpublish" class="btn btn-danger" 
                                                        onclick="return confirm('هل أنت متأكد من إلغاء نشر هذه النتائج؟')">
                                                    إلغاء النشر
                                                </button>
                                            </form>
                                        <?php else: ?>
                                            <form method="POST" class="publish-form" style="display: inline-flex; gap: 10px;">
                                                <input type="hidden" name="semester_id" value="<?php echo $semester['id']; ?>">
                                                <input type="number" name="days" value="7" min="1" max="30" 
                                                       title="عدد الأيام التي ستكون النتائج متاحة خلالها">
                                                <span>يوم</span>
                                                <button type="submit" name="publish" class="btn btn-primary">
                                                    نشر
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- قسم النتائج السنوية -->
        <div class="card" style="margin-top: 20px;">
            <div class="card-header">
                <h2>النتائج السنوية (فصلين دراسيين)</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>العام الدراسي</th>
                            <th>التخصص</th>
                            <th>الكلية</th>
                            <th>عدد الطلاب</th>
                            <th>حالة النشر</th>
                            <th>الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($annual_results)): ?>
                            <tr>
                                <td colspan="6" style="text-align: center; padding: 20px;">
                                    لا توجد نتائج سنوية متاحة
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($annual_results as $annual): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($annual['academic_year']); ?></td>
                                    <td><?php echo htmlspecialchars($annual['major_name']); ?></td>
                                    <td><?php echo htmlspecialchars($annual['college_name']); ?></td>
                                    <td><?php echo htmlspecialchars($annual['students_count']); ?></td>
                                    <td>
                                        <?php if ($annual['is_published'] && strtotime($annual['expires_at']) > time()): ?>
                                            <span class="status-badge status-published">منشور</span>
                                            <div class="expires-info">
                                                ينتهي في: <?php echo date('Y-m-d H:i', strtotime($annual['expires_at'])); ?>
                                            </div>
                                        <?php elseif ($annual['is_published'] && strtotime($annual['expires_at']) <= time()): ?>
                                            <span class="status-badge status-expired">منتهي الصلاحية</span>
                                        <?php else: ?>
                                            <span class="status-badge status-not-published">غير منشور</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($annual['is_published'] && strtotime($annual['expires_at']) > time()): ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="annual_id" value="<?php echo $annual['published_id']; ?>">
                                                <button type="submit" name="unpublish_annual" class="btn btn-danger" 
                                                        onclick="return confirm('هل أنت متأكد من إلغاء نشر هذه النتائج السنوية؟')">
                                                    إلغاء النشر
                                                </button>
                                            </form>
                                        <?php else: ?>
                                            <form method="POST" class="publish-form" style="display: inline-flex; gap: 10px;">
                                                <input type="hidden" name="academic_year" value="<?php echo htmlspecialchars($annual['academic_year']); ?>">
                                                <input type="hidden" name="major_id" value="<?php echo $annual['major_id']; ?>">
                                                <input type="number" name="days" value="7" min="1" max="30" 
                                                       title="عدد الأيام التي ستكون النتائج متاحة خلالها">
                                                <span>يوم</span>
                                                <button type="submit" name="publish_annual" class="btn btn-primary">
                                                    نشر النتائج السنوية
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- قسم إدارة نشاط الطلاب -->
        <div class="card" style="margin-top: 20px;">
            <div class="card-header">
                <h2>إدارة نشاط الطلاب</h2>
            </div>
            <div style="padding: 20px;">
                <p style="margin-bottom: 15px;">
                    <strong>ملاحظة:</strong> يمكنك إيقاف نشاط طالب معين حتى لا تظهر نتائجه على الموقع، أو إعادة تفعيل نشاطه مرة أخرى.
                </p>
                
                <form method="GET" style="margin-bottom: 20px;">
                    <div style="display: flex; gap: 15px; align-items: flex-end; flex-wrap: wrap;">
                        <div style="flex: 1; min-width: 300px;">
                            <label for="search_student" style="display: block; margin-bottom: 8px; font-weight: bold;">البحث عن طالب:</label>
                            <input type="text" 
                                   id="search_student" 
                                   name="search_student" 
                                   placeholder="أدخل الرقم الجامعي أو الاسم الكامل للطالب"
                                   value="<?php echo isset($_GET['search_student']) ? htmlspecialchars($_GET['search_student']) : ''; ?>"
                                   style="width: 100%; padding: 12px; font-size: 16px; border: 2px solid #ddd; border-radius: 5px; box-sizing: border-box;">
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button type="submit" class="btn btn-primary" style="padding: 12px 30px; font-size: 16px; white-space: nowrap;">بحث</button>
                            <?php if (!empty($search_term)): ?>
                                <a href="publish_results.php" class="btn" style="padding: 12px 20px; font-size: 16px; white-space: nowrap; background: #999; color: white; text-decoration: none; border-radius: 5px;">إلغاء</a>
                            <?php endif; ?>
                        </div>
                    </div>
                </form>
                
                <?php
                // البحث عن الطلاب
                $search_term = isset($_GET['search_student']) ? trim($_GET['search_student']) : '';
                $students = [];
                
                if (!empty($search_term)) {
                    $user = $auth->getCurrentUser();
                    $college_id = $user['college_id'] ?? null;
                    
                    if ($auth->hasRole('admin')) {
                        $students = $db->fetchAll(
                            "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name
                             FROM students s
                             LEFT JOIN majors m ON s.major_id = m.id
                             LEFT JOIN colleges c ON s.college_id = c.id
                             WHERE s.student_number LIKE ? OR s.full_name_ar LIKE ?
                             ORDER BY s.student_number
                             LIMIT 20",
                            ["%{$search_term}%", "%{$search_term}%"]
                        );
                    } else {
                        $students = $db->fetchAll(
                            "SELECT s.*, m.name_ar as major_name, c.name_ar as college_name
                             FROM students s
                             LEFT JOIN majors m ON s.major_id = m.id
                             LEFT JOIN colleges c ON s.college_id = c.id
                             WHERE c.id = ? AND (s.student_number LIKE ? OR s.full_name_ar LIKE ?)
                             ORDER BY s.student_number
                             LIMIT 20",
                            [$college_id, "%{$search_term}%", "%{$search_term}%"]
                        );
                    }
                }
                ?>
                
                <?php if (!empty($students)): ?>
                    <div class="table-container" style="margin-top: 20px;">
                        <table>
                            <thead>
                                <tr>
                                    <th>الرقم الجامعي</th>
                                    <th>الاسم</th>
                                    <th>التخصص</th>
                                    <th>الكلية</th>
                                    <th>الحالة الحالية</th>
                                    <th>حجب النتائج</th>
                                    <th>الإجراءات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($students as $student): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                        <td><?php echo htmlspecialchars($student['major_name'] ?? '-'); ?></td>
                                        <td><?php echo htmlspecialchars($student['college_name'] ?? '-'); ?></td>
                                        <td>
                                            <?php
                                            $status_text = [
                                                'active' => 'نشط',
                                                'suspended' => 'موقوف',
                                                'graduated' => 'متخرج',
                                                'withdrawn' => 'منسحب'
                                            ];
                                            $status_class = [
                                                'active' => 'status-published',
                                                'suspended' => 'status-expired',
                                                'graduated' => 'status-not-published',
                                                'withdrawn' => 'status-expired'
                                            ];
                                            $current_status = $student['status'] ?? 'active';
                                            ?>
                                            <span class="status-badge <?php echo $status_class[$current_status] ?? 'status-not-published'; ?>">
                                                <?php echo $status_text[$current_status] ?? $current_status; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php
                                            $results_blocked = isset($student['results_blocked']) && $student['results_blocked'];
                                            ?>
                                            <?php if ($results_blocked): ?>
                                                <span class="status-badge status-expired">محجوبة</span>
                                            <?php else: ?>
                                                <span class="status-badge status-published">غير محجوبة</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div style="display: flex; gap: 5px; flex-wrap: wrap;">
                                                <?php if ($results_blocked): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                        <input type="hidden" name="block" value="0">
                                                        <button type="submit" 
                                                                name="toggle_results_block" 
                                                                class="btn btn-success"
                                                                style="font-size: 12px; padding: 6px 12px;"
                                                                onclick="return confirm('هل أنت متأكد من فك حجب نتائج هذا الطالب؟')">
                                                            فك الحجب
                                                        </button>
                                                    </form>
                                                <?php else: ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                        <input type="hidden" name="block" value="1">
                                                        <button type="submit" 
                                                                name="toggle_results_block" 
                                                                class="btn btn-danger"
                                                                style="font-size: 12px; padding: 6px 12px;"
                                                                onclick="return confirm('هل أنت متأكد من حجب نتائج هذا الطالب؟ لن تظهر نتائجه على الموقع.')">
                                                            حجب النتائج
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <?php if ($current_status === 'active'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                        <input type="hidden" name="new_status" value="suspended">
                                                        <button type="submit" 
                                                                name="toggle_student_status" 
                                                                class="btn btn-warning"
                                                                style="font-size: 12px; padding: 6px 12px;"
                                                                onclick="return confirm('هل أنت متأكد من إيقاف نشاط هذا الطالب؟')">
                                                            إيقاف النشاط
                                                        </button>
                                                    </form>
                                                <?php else: ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="student_id" value="<?php echo $student['id']; ?>">
                                                        <input type="hidden" name="new_status" value="active">
                                                        <button type="submit" 
                                                                name="toggle_student_status" 
                                                                class="btn btn-info"
                                                                style="font-size: 12px; padding: 6px 12px;"
                                                                onclick="return confirm('هل أنت متأكد من إعادة تفعيل نشاط هذا الطالب؟')">
                                                            تفعيل النشاط
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php elseif (!empty($search_term)): ?>
                    <div class="alert alert-info" style="margin-top: 20px;">
                        لم يتم العثور على أي طلاب مطابقين للبحث.
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="card" style="margin-top: 20px;">
            <div class="card-header">
                <h3>معلومات</h3>
            </div>
            <div style="padding: 20px;">
                <p><strong>كيفية الاستخدام:</strong></p>
                <ul style="margin-right: 20px;">
                    <li>اختر الفصل الدراسي الذي تريد نشر نتائجه</li>
                    <li>حدد عدد الأيام التي ستكون النتائج متاحة خلالها (افتراضياً 7 أيام)</li>
                    <li>اضغط على "نشر" لنشر النتائج</li>
                    <li>يمكن للطلاب البحث عن نتائجهم من خلال صفحة <a href="public_results.php" target="_blank">البحث عن النتائج</a></li>
                    <li>النتائج ستنتهي تلقائياً بعد انتهاء المدة المحددة</li>
                    <li><strong>لإيقاف نشاط طالب:</strong> ابحث عنه ثم اضغط على "إيقاف النشاط" - لن تظهر نتائجه على الموقع</li>
                    <li><strong>لإعادة تفعيل طالب:</strong> ابحث عنه ثم اضغط على "تفعيل النشاط" - ستظهر نتائجه على الموقع</li>
                    <li><strong>لحجب نتائج طالب:</strong> ابحث عنه ثم اضغط على "حجب النتائج" - لن تظهر نتائجه على الموقع حتى لو كان نشطاً</li>
                    <li><strong>لفك حجب نتائج طالب:</strong> ابحث عنه ثم اضغط على "فك الحجب" - ستظهر نتائجه على الموقع مرة أخرى</li>
                    <li><strong>الفرق بين إيقاف النشاط وحجب النتائج:</strong> إيقاف النشاط يوقف الطالب بالكامل، بينما حجب النتائج يمنع فقط ظهور النتائج على الموقع مع بقاء الطالب نشطاً في النظام</li>
                </ul>
                <p style="margin-top: 15px;">
                    <strong>رابط صفحة البحث:</strong> 
                    <a href="public_results.php" target="_blank">public_results.php</a>
                </p>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

