<?php
/**
 * تسجيل طلاب في مادة معينة
 * Register Students to a Course
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$__old_exception_handler = set_exception_handler(function($e) {
    $trace = $e->getTraceAsString();
    $logMsg = 'Uncaught Exception: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine() . PHP_EOL . $trace;
    error_log($logMsg);
    if (defined('LOG_DIR')) {
        @file_put_contents(LOG_DIR . '/app-error.log', '[' . date('Y-m-d H:i:s') . '] ' . $logMsg . PHP_EOL, FILE_APPEND);
    }
    http_response_code(500);
    if (defined('APP_DEBUG') && APP_DEBUG) {
        echo '<div style="direction:rtl;padding:12px;margin:12px;border:1px solid #f00;background:#fee;color:#900">';
        echo '<strong>حدث خطأ غير متوقع:</strong><br>';
        echo htmlspecialchars($e->getMessage());
        echo ' (' . basename($e->getFile()) . ':' . $e->getLine() . ')';
        echo '<pre style="white-space:pre-wrap;max-height:300px;overflow:auto;background:#fff;border:1px solid #fcc;padding:8px;margin-top:8px;">' . htmlspecialchars($trace) . '</pre>';
        echo '</div>';
    } else {
        echo '<div class="alert alert-error">حدث خطأ غير متوقع. الرجاء المحاولة لاحقاً.</div>';
    }
    exit;
});
set_error_handler(function($severity, $message, $file, $line) {
    throw new ErrorException($message, 0, $severity, $file, $line);
});

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('manage_students') && !$auth->hasRole('admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 0;
$semester_id = isset($_GET['semester_id']) ? (int)$_GET['semester_id'] : 0;
$message = '';
$currentCollegeId = isset($_SESSION['college_id']) ? (int)$_SESSION['college_id'] : null;

// عرض رسالة النجاح من URL
if (isset($_GET['msg']) && $_GET['msg'] === 'success') {
    $message = '<div class="alert alert-success">✓ تم تسجيل الطلاب بنجاح</div>';
}

// تحقق مبكر من المُدخلات لتفادي أخطاء 500 بسبب قيم غير صحيحة
if ($course_id <= 0 || $semester_id <= 0) {
    http_response_code(400);
    echo '<!DOCTYPE html><html lang="ar" dir="rtl"><head><meta charset="UTF-8"><title>خطأ في المُدخلات</title></head><body>';
    echo '<div class="alert alert-error" style="direction:rtl;padding:12px;margin:12px;border:1px solid #f00;background:#fee;color:#900">';
    echo '<strong>خطأ في المُدخلات:</strong> يجب تحديد مادة وفصل دراسي صالحين.';
    echo '<div style="margin-top:8px">';
    echo 'course_id=' . htmlspecialchars((string)$course_id) . ' | semester_id=' . htmlspecialchars((string)$semester_id);
    echo '</div>';
    echo '<div style="margin-top:8px"><a href="courses.php" class="btn btn-secondary">الرجوع إلى صفحة المواد</a></div>';
    echo '</div></body></html>';
    exit;
}

// جلب معلومات المادة
try {
    $course = $db->fetchOne(
        "SELECT c.*, m.college_id, m.id as major_id
         FROM courses c 
         JOIN majors m ON c.major_id = m.id 
         WHERE c.id = ?",
        [$course_id]
    );
} catch (Exception $e) {
    http_response_code(500);
    die('<div class="alert alert-error">حدث خطأ أثناء جلب بيانات المادة. الرجاء المحاولة لاحقاً.</div>');
}

if (!$course || (!$auth->belongsToCollege($course['college_id']) && !$auth->hasRole('admin'))) {
    header('Location: courses.php');
    exit;
}

// جلب معلومات الفصل
try {
    $semester = $db->fetchOne("SELECT * FROM semesters WHERE id = ?", [$semester_id]);
} catch (Exception $e) {
    http_response_code(500);
    die('<div class="alert alert-error">حدث خطأ أثناء جلب بيانات الفصل الدراسي.</div>');
}

if (!$semester) {
    header('Location: courses.php');
    exit;
}

// جلب معلومات التخصص (يجب أن يكون قبل معالجة Excel/CSV لاستخدامه في رسائل الخطأ)
$major = $db->fetchOne("SELECT * FROM majors WHERE id = ?", [$course['major_id']]);

// معالجة التسجيل من Excel/CSV
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_excel') {
    // تسجيل للمساعدة في التشخيص
    error_log("Excel upload attempt - POST data: " . print_r($_POST, true));
    error_log("Excel upload attempt - FILES data: " . print_r($_FILES, true));
    
    // التحقق من وجود الملف أولاً
    if (!isset($_FILES['excel_file'])) {
        $message = '<div class="alert alert-error">خطأ: لم يتم رفع الملف. تأكد من اختيار ملف وتأكد من أن النموذج يحتوي على enctype="multipart/form-data"</div>';
        error_log("Excel upload error: No file uploaded");
    } elseif ($_FILES['excel_file']['error'] !== UPLOAD_ERR_OK) {
        // معالجة أخطاء الرفع
        $upload_error = $_FILES['excel_file']['error'];
        $error_msg = 'خطأ غير معروف في رفع الملف';
        switch ($upload_error) {
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $error_msg = 'حجم الملف كبير جداً. الحد الأقصى: ' . ini_get('upload_max_filesize') . ' (PHP) أو 10MB (النظام)';
                break;
            case UPLOAD_ERR_PARTIAL:
                $error_msg = 'تم رفع جزء من الملف فقط. يرجى المحاولة مرة أخرى';
                break;
            case UPLOAD_ERR_NO_FILE:
                $error_msg = 'لم يتم اختيار ملف';
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $error_msg = 'مجلد الملفات المؤقتة غير موجود على الخادم';
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $error_msg = 'فشل كتابة الملف على القرص. تحقق من صلاحيات المجلد';
                break;
            case UPLOAD_ERR_EXTENSION:
                $error_msg = 'تم إيقاف رفع الملف بواسطة أحد امتدادات PHP';
                break;
        }
        $message = '<div class="alert alert-error">' . $error_msg . ' (رمز الخطأ: ' . $upload_error . ')</div>';
    } else {
        $file = $_FILES['excel_file'];
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        // التحقق من وجود الملف المؤقت
        if (!file_exists($file['tmp_name'])) {
            $message = '<div class="alert alert-error">خطأ: الملف المؤقت غير موجود. قد يكون حجم الملف كبير جداً أو حدث خطأ في الرفع</div>';
        }
        // التحقق من حجم الملف (10MB)
        elseif ($file['size'] > (10 * 1024 * 1024)) {
            $message = '<div class="alert alert-error">خطأ: حجم الملف كبير جداً. الحد الأقصى 10MB</div>';
        }
        // التحقق من نوع الملف
        elseif (!in_array($file_ext, ['csv', 'xlsx', 'xls'])) {
            $message = '<div class="alert alert-error">خطأ: نوع الملف غير مدعوم. يرجى رفع ملف CSV أو Excel (.xlsx, .xls)</div>';
        } else {
            try {
                // التحقق من وجود ملف ExcelReader
                $excelReaderPath = __DIR__ . '/classes/ExcelReader.php';
                if (!file_exists($excelReaderPath)) {
                    throw new Exception('ملف ExcelReader.php غير موجود في: ' . $excelReaderPath);
                }
                
                require_once $excelReaderPath;
                
                if (!class_exists('ExcelReader')) {
                    throw new Exception('فئة ExcelReader غير موجودة');
                }
                
                $excelReader = new ExcelReader();
                
                if (!method_exists($excelReader, 'readFile')) {
                    throw new Exception('دالة readFile غير موجودة في ExcelReader');
                }
                
                $students_data = $excelReader->readFile($file['tmp_name'], $file_ext);
                
                if ($students_data === false || empty($students_data)) {
                    $message = '<div class="alert alert-error">خطأ: فشل قراءة الملف أو الملف فارغ. يرجى التحقق من صحة الملف والتنسيق.<br>';
                    if ($file_ext === 'xlsx' || $file_ext === 'xls') {
                        $message .= '<strong>ملاحظة:</strong> إذا كان الملف من نوع Excel، قد تحتاج إلى تثبيت مكتبة PhpSpreadsheet. جرب تحويل الملف إلى CSV أولاً.';
                    }
                    $message .= '</div>';
                } else {
                    $success_count = 0;
                    $error_count = 0;
                    $not_found = [];
                    $already_registered = 0;
                    
                    foreach ($students_data as $row) {
                    // قراءة الحقول المحتملة
                    $input_student_id = trim($row['student_id'] ?? '');
                    $student_number   = trim($row['student_number'] ?? '');
                    $national_id      = trim($row['national_id'] ?? '');
                    $student_name     = trim($row['student_name'] ?? '');
                    
                    if ($input_student_id === '' && $student_number === '' && $national_id === '' && $student_name === '') {
                        continue;
                    }
                    
                    // البحث عن الطالب
                    $student = null;
                    
                    // 1) البحث بالمعرّف الداخلي مباشرة
                    if (!empty($input_student_id) && ctype_digit((string)$input_student_id)) {
                        if ($auth->hasRole('admin')) {
                            $student = $db->fetchOne(
                                "SELECT * FROM students WHERE id = ? AND major_id = ?",
                                [$input_student_id, $course['major_id']]
                            );
                        } else {
                            if ($currentCollegeId !== null) {
                                $student = $db->fetchOne(
                                    "SELECT * FROM students WHERE id = ? AND major_id = ? AND college_id = ?",
                                    [$input_student_id, $course['major_id'], $currentCollegeId]
                                );
                            }
                        }
                    }
                    
                    // 2) البحث بالرقم الجامعي (مع تجربة الصيغة المقلوبة XX-YYYY أو YYYY-XX)
                    if (!empty($student_number)) {
                        if (!$student) {
                            $numbers_to_try = [$student_number];
                            if (strpos($student_number, '-') !== false) {
                                $parts = explode('-', $student_number, 2);
                                if (count($parts) === 2) {
                                    $alt = trim($parts[1]) . '-' . trim($parts[0]);
                                    if ($alt !== $student_number) {
                                        $numbers_to_try[] = $alt;
                                    }
                                }
                            }
                            foreach ($numbers_to_try as $try_num) {
                                if ($student) break;
                                if ($auth->hasRole('admin')) {
                                    $student = $db->fetchOne(
                                        "SELECT * FROM students WHERE student_number = ? AND major_id = ?",
                                        [$try_num, $course['major_id']]
                                    );
                                } else {
                                    if ($currentCollegeId !== null) {
                                        $student = $db->fetchOne(
                                            "SELECT * FROM students WHERE student_number = ? AND major_id = ? AND college_id = ?",
                                            [$try_num, $course['major_id'], $currentCollegeId]
                                        );
                                    }
                                }
                            }
                        }
                    }
                    
                    // 3) البحث بالرقم الوطني
                    if (!$student && !empty($national_id)) {
                        if ($auth->hasRole('admin')) {
                            $student = $db->fetchOne(
                                "SELECT * FROM students WHERE national_id = ? AND major_id = ?",
                                [$national_id, $course['major_id']]
                            );
                        } else {
                            if ($currentCollegeId !== null) {
                                $student = $db->fetchOne(
                                    "SELECT * FROM students WHERE national_id = ? AND major_id = ? AND college_id = ?",
                                    [$national_id, $course['major_id'], $currentCollegeId]
                                );
                            }
                        }
                    }
                    
                    // 4) إذا لم يُعثر عليه بأي معرّف، البحث بالاسم
                    if (!$student && !empty($student_name)) {
                        if ($auth->hasRole('admin')) {
                            $student = $db->fetchOne(
                                "SELECT * FROM students WHERE full_name_ar = ? AND major_id = ?",
                                [$student_name, $course['major_id']]
                            );
                        } else {
                            if ($currentCollegeId !== null) {
                                $student = $db->fetchOne(
                                    "SELECT * FROM students WHERE full_name_ar = ? AND major_id = ? AND college_id = ?",
                                    [$student_name, $course['major_id'], $currentCollegeId]
                                );
                            }
                        }
                    }
                    
                    if ($student && $student['status'] === 'active') {
                        // التحقق من التسجيل المسبق
                        $existing = $db->fetchOne(
                            "SELECT * FROM student_courses 
                             WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                            [$student['id'], $course_id, $semester_id]
                        );
                        
                        if ($existing) {
                            $already_registered++;
                        } else {
                            try {
                                // استخدام التاريخ الحالي من PHP
                                $registration_date = date('Y-m-d');
                                
                                $db->query(
                                    "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date) 
                                     VALUES (?, ?, ?, ?)",
                                    [$student['id'], $course_id, $semester_id, $registration_date]
                                );
                                $success_count++;
                            } catch (PDOException $e) {
                                $error_count++;
                                error_log("Error registering student from Excel: " . $e->getMessage());
                            }
                        }
                    } else {
                        // حفظ ما أدخله المستخدم للتشخيص مع معلومات إضافية
                        $search_value = $input_student_id ?: ($student_number ?: ($national_id ?: $student_name));
                        
                        // محاولة البحث في قاعدة البيانات لمعرفة السبب
                        $check_student = null;
                        $check_reason = '';
                        
                        // البحث في جميع الطلاب (بغض النظر عن التخصص)
                        if (!empty($input_student_id) && ctype_digit((string)$input_student_id)) {
                            $check_student = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM students s JOIN majors m ON s.major_id = m.id WHERE s.id = ?", [$input_student_id]);
                        } elseif (!empty($student_number)) {
                            $check_student = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM students s JOIN majors m ON s.major_id = m.id WHERE s.student_number = ?", [$student_number]);
                            if (!$check_student && strpos($student_number, '-') !== false) {
                                $parts = explode('-', $student_number, 2);
                                if (count($parts) === 2) {
                                    $alt = trim($parts[1]) . '-' . trim($parts[0]);
                                    $check_student = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM students s JOIN majors m ON s.major_id = m.id WHERE s.student_number = ?", [$alt]);
                                }
                            }
                        } elseif (!empty($national_id)) {
                            $check_student = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM students s JOIN majors m ON s.major_id = m.id WHERE s.national_id = ?", [$national_id]);
                        } elseif (!empty($student_name)) {
                            $check_student = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM students s JOIN majors m ON s.major_id = m.id WHERE s.full_name_ar = ?", [$student_name]);
                        }
                        
                        if ($check_student) {
                            // الطالب موجود لكن في تخصص مختلف أو غير نشط
                            if ($check_student['major_id'] != $course['major_id']) {
                                $check_reason = ' (موجود في تخصص: ' . $check_student['major_name'] . ')';
                            } elseif ($check_student['status'] != 'active') {
                                $check_reason = ' (حالة: ' . $check_student['status'] . ')';
                            }
                        }
                        
                        $not_found[] = $search_value . $check_reason;
                    }
                    }
                    
                    // بناء رسالة النتيجة
                    if ($success_count > 0) {
                        $sc_label = ($success_count === 1) ? 'طالب واحد' : (($success_count === 2) ? 'طالبان' : $success_count . ' طلاب');
                        $message = '<div class="alert alert-success">✓ تم تسجيل ' . $sc_label . ' بنجاح من الملف</div>';
                    }
                    if ($already_registered > 0) {
                        $label = ($already_registered === 1) ? 'طالب واحد كان مسجلاً مسبقاً' : (($already_registered === 2) ? 'طالبان كانا مسجلين مسبقاً' : $already_registered . ' طلاب كانوا مسجلين مسبقاً');
                        $message .= '<div class="alert alert-info">⚠ ' . $label . '</div>';
                    }
                    if ($error_count > 0) {
                        $message .= '<div class="alert alert-warning">⚠ ' . $error_count . ' خطأ في التسجيل</div>';
                    }
                    if (count($not_found) > 0) {
                        $message .= '<div class="alert alert-warning" style="margin-top: 10px;">';
                        $nf_count = count($not_found);
                        $nf_label = ($nf_count === 1) ? 'طالب واحد' : (($nf_count === 2) ? 'طالبان' : $nf_count . ' طلاب');
                        $message .= '<strong>⚠ لم يُعثر على ' . $nf_label . ' في التخصص أو غير نشطين:</strong><br>';
                        $message .= '<div style="margin-top: 8px; padding: 8px; background: #fff3cd; border-radius: 4px; font-family: monospace; font-size: 13px;">';
                        
                        // عرض أول 20 طالب
                        $display_count = min(20, count($not_found));
                        $display_items = array_slice($not_found, 0, $display_count);
                        $message .= implode(', ', $display_items);
                        
                        if (count($not_found) > $display_count) {
                            $remaining = count($not_found) - $display_count;
                            $rem_label = ($remaining === 1) ? 'طالب واحد آخر' : (($remaining === 2) ? 'طالبان آخران' : $remaining . ' طلاب آخرون');
                            $message .= '<br><strong style="color: #856404;">... و' . $rem_label . '</strong>';
                        }
                        
                        $message .= '</div>';
                        $message .= '<div style="background: #e3f2fd; padding: 12px; border-radius: 4px; margin-top: 10px; border-right: 4px solid #2196F3;">';
                        $message .= '<strong style="color: #1976d2;">💡 خطوات التحقق:</strong><br>';
                        $message .= '<ol style="margin-right: 20px; margin-top: 8px; line-height: 1.8; font-size: 0.95em;">';
                        $message .= '<li>تأكد من أن الطلاب موجودين في نفس التخصص: <strong>' . htmlspecialchars($major ? $major['name_ar'] : 'غير محدد') . '</strong></li>';
                        $message .= '<li>تأكد من أن حالة الطلاب هي "نشط" (active) وليس "معلق" (suspended) أو "منسحب" (withdrawn)</li>';
                        $message .= '<li>تحقق من صحة البيانات المدخلة (الرقم الجامعي، الرقم الوطني، الاسم)</li>';
                        $message .= '<li>إذا كان الطالب موجوداً في تخصص آخر (كما هو موضح في القائمة أعلاه)، يجب نقله إلى التخصص الصحيح أولاً من صفحة <a href="students.php?major_id=' . $course['major_id'] . '" style="color: #1976d2; font-weight: bold;">إدارة الطلاب</a></li>';
                        $message .= '</ol>';
                        $message .= '<p style="margin-top: 10px;">';
                        $message .= '<a href="students.php?major_id=' . $course['major_id'] . '" class="btn btn-primary" style="padding: 8px 15px; text-decoration: none; display: inline-block; background: #2196F3; color: white; border-radius: 4px; font-weight: bold;">';
                        $message .= '👥 عرض جميع الطلاب في هذا التخصص';
                        $message .= '</a>';
                        $message .= '</p>';
                        $message .= '</div>';
                        $message .= '</div>';
                    }
                }
            } catch (Exception $e) {
                $error_message = $e->getMessage();
                $message = '<div class="alert alert-error">';
                $message .= '<strong>خطأ في قراءة الملف:</strong><br>';
                $message .= htmlspecialchars($error_message);
                $message .= '<br><br>';
                
                if ($file_ext === 'xlsx' || $file_ext === 'xls') {
                    $message .= '<strong>الحلول المقترحة:</strong><br>';
                    $message .= '1. قم بتحويل الملف إلى CSV من Excel (File → Save As → CSV)<br>';
                    $message .= '2. أو قم بتثبيت مكتبة PhpSpreadsheet:<br>';
                    $message .= '   <code>composer require phpoffice/phpspreadsheet</code><br>';
                    $message .= '3. أو استخدم التسجيل اليدوي من القائمة أدناه';
                } else {
                    $message .= '<strong>يرجى التحقق من:</strong><br>';
                    $message .= '1. أن الملف بتنسيق CSV صحيح<br>';
                    $message .= '2. أن الملف يحتوي على عمود واحد على الأقل (student_id, student_number, national_id, أو student_name)<br>';
                    $message .= '3. أن الملف غير فارغ';
                }
                $message .= '</div>';
                error_log("Excel upload error: " . $error_message . " | File: " . $file['name'] . " | Size: " . $file['size'] . " | Type: " . $file_ext);
            }
        }
    }
}

// معالجة التسجيل
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'register') {
    $student_ids = $_POST['students'] ?? [];
    $success_count = 0;
    $error_count = 0;
    $already_registered_count = 0;
    $not_found_count = 0;
    $errors = [];
    
    if (empty($student_ids)) {
        $message = '<div class="alert alert-warning">⚠️ يرجى اختيار طالب واحد على الأقل للتسجيل</div>';
    } else {
        // بدء معاملة قاعدة البيانات
        try {
            $db->getConnection()->beginTransaction();
            
            foreach ($student_ids as $student_id) {
                $student_id = (int)$student_id;
                
                if ($student_id <= 0) {
                    continue;
                }
                
                try {
                    // التحقق من أن الطالب ينتمي للكلية والتخصص الصحيح
                    $student = $db->fetchOne(
                        "SELECT * FROM students WHERE id = ? AND major_id = ?",
                        [$student_id, $course['major_id']]
                    );
                    
                    if (!$student) {
                        $not_found_count++;
                        $errors[] = "الطالب ID: $student_id غير موجود في التخصص";
                        continue;
                    }
                    
                    // التحقق من الصلاحيات
                    if (!$auth->belongsToCollege($student['college_id']) && !$auth->hasRole('admin')) {
                        $not_found_count++;
                        $errors[] = "الطالب ID: $student_id - ليس لديك صلاحية";
                        continue;
                    }
                    
                    // التحقق من حالة الطالب
                    if ($student['status'] !== 'active') {
                        $error_count++;
                        $errors[] = "الطالب ID: $student_id - غير نشط (حالة: " . $student['status'] . ")";
                        continue;
                    }
                    
                    // التحقق من التسجيل المسبق
                    $existing = $db->fetchOne(
                        "SELECT * FROM student_courses 
                         WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                        [$student_id, $course_id, $semester_id]
                    );
                    
                    if ($existing) {
                        $already_registered_count++;
                        continue;
                    }
                    
                    // محاولة التسجيل
                    // استخدام التاريخ الحالي من PHP
                    $registration_date = date('Y-m-d');
                    
                    $db->query(
                        "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date) 
                         VALUES (?, ?, ?, ?)",
                        [$student_id, $course_id, $semester_id, $registration_date]
                    );
                    $success_count++;
                    
                } catch (PDOException $e) {
                    // قد تكون المادة مسجلة مسبقاً (في حالة race condition)
                    if (strpos($e->getMessage(), 'Duplicate entry') !== false || 
                        strpos($e->getMessage(), 'unique_student_course_semester') !== false ||
                        strpos($e->getMessage(), '1062') !== false) {
                        $already_registered_count++;
                    } else {
                        $error_count++;
                        $errors[] = "الطالب ID: $student_id - " . $e->getMessage();
                        error_log("Error registering student $student_id: " . $e->getMessage());
                    }
                }
            }
            
            // تأكيد المعاملة
            $db->getConnection()->commit();
            
        } catch (Exception $e) {
            // إلغاء المعاملة في حالة الخطأ
            $db->getConnection()->rollBack();
            $message = '<div class="alert alert-error">خطأ في التسجيل: ' . htmlspecialchars($e->getMessage()) . '</div>';
            error_log("Transaction error: " . $e->getMessage());
        }
        
        // بناء رسالة النتيجة
        if ($success_count > 0) {
            $sc_label = ($success_count === 1) ? 'طالب واحد' : (($success_count === 2) ? 'طالبان' : $success_count . ' طلاب');
            $message = '<div class="alert alert-success">✓ تم تسجيل ' . $sc_label . ' في المادة بنجاح</div>';
        }
        if ($already_registered_count > 0) {
            $arc_label = ($already_registered_count === 1) ? 'طالب واحد كان مسجلاً مسبقاً' : (($already_registered_count === 2) ? 'طالبان كانا مسجلين مسبقاً' : $already_registered_count . ' طلاب كانوا مسجلين مسبقاً');
            $message .= '<div class="alert alert-info">⚠ ' . $arc_label . '</div>';
        }
        if ($error_count > 0) {
            $ec_label = ($error_count === 1) ? 'طالب واحد لم يتم تسجيله (قد يكون غير نشط)' : (($error_count === 2) ? 'طالبان لم يتم تسجيلهما' : $error_count . ' طلاب لم يتم تسجيلهم (قد يكونوا غير نشطين)');
            $message .= '<div class="alert alert-warning">⚠ ' . $ec_label . '</div>';
            if (count($errors) > 0 && count($errors) <= 5) {
                $message .= '<div class="alert alert-info" style="margin-top:10px;"><small>' . implode('<br>', array_map('htmlspecialchars', $errors)) . '</small></div>';
            }
        }
        if ($not_found_count > 0) {
            $nfc_label = ($not_found_count === 1) ? 'طالب واحد غير موجود في التخصص' : (($not_found_count === 2) ? 'طالبان غير موجودين في التخصص' : $not_found_count . ' طلاب غير موجودين في التخصص');
            $message .= '<div class="alert alert-warning">⚠ ' . $nfc_label . '</div>';
        }
        
        // إعادة تحميل الصفحة بعد النجاح لتحديث القائمة
        if ($success_count > 0) {
            header("Location: register_students_to_course.php?course_id=" . $course_id . "&semester_id=" . $semester_id . "&msg=success");
            exit;
        }
    }
}

// جلب الطلاب المسجلين في المادة
$registered_students = $db->fetchAll(
    "SELECT s.* FROM students s
     JOIN student_courses sc ON s.id = sc.student_id
     WHERE sc.course_id = ? AND sc.semester_id = ?
     ORDER BY s.full_name_ar",
    [$course_id, $semester_id]
);

$registered_ids = array_column($registered_students, 'id');

// جلب معلومات التخصص (يجب أن يكون قبل استخدام $major في رسائل الخطأ)
$major = $db->fetchOne("SELECT * FROM majors WHERE id = ?", [$course['major_id']]);

// جلب جميع الطلاب في نفس التخصص (بما في ذلك غير النشطين للتشخيص)
if ($auth->hasRole('admin')) {
    $all_students = $db->fetchAll(
        "SELECT s.* FROM students s 
         WHERE s.major_id = ?
         ORDER BY s.full_name_ar",
        [$course['major_id']]
    );
    
    // جلب عدد الطلاب النشطين فقط
    $active_students_count = $db->fetchOne(
        "SELECT COUNT(*) as count FROM students s 
         WHERE s.major_id = ? AND s.status = 'active'",
        [$course['major_id']]
    )['count'];
} else {
    if ($currentCollegeId === null) {
        $all_students = [];
        $active_students_count = 0;
    } else {
        $all_students = $db->fetchAll(
            "SELECT s.* FROM students s 
             WHERE s.major_id = ? AND s.college_id = ?
             ORDER BY s.full_name_ar",
            [$course['major_id'], $currentCollegeId]
        );
        
        // جلب عدد الطلاب النشطين فقط
        $active_students_count = $db->fetchOne(
            "SELECT COUNT(*) as count FROM students s 
             WHERE s.major_id = ? AND s.college_id = ? AND s.status = 'active'",
            [$course['major_id'], $currentCollegeId]
        )['count'];
    }
}

// تصفية الطلاب النشطين فقط للعرض
$active_students = array_filter($all_students, function($s) {
    return $s['status'] === 'active';
});
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تسجيل طلاب في المادة</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>تسجيل طلاب في المادة</h2>
            <div>
                <a href="grades.php?course_id=<?php echo $course_id; ?>&semester_id=<?php echo $semester_id; ?>" class="btn btn-secondary">العودة للدرجات</a>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h3>معلومات المادة</h3>
            </div>
            <p><strong>المادة:</strong> <?php echo htmlspecialchars($course['name_ar']); ?> (<?php echo htmlspecialchars($course['code']); ?>)</p>
            <p><strong>التخصص:</strong> <?php echo $major ? htmlspecialchars($major['name_ar']) : 'غير محدد'; ?></p>
            <p><strong>الفصل:</strong> فصل <?php echo $semester['semester_number']; ?> - <?php echo htmlspecialchars($semester['academic_year']); ?></p>
            <p><strong>الطلاب المسجلين حالياً:</strong> <?php echo count($registered_students); ?> طالب</p>
            <p><strong>إجمالي الطلاب في التخصص:</strong> <?php echo count($all_students); ?> طالب (<?php echo $active_students_count; ?> نشط)</p>
        </div>
        
        <?php echo $message; ?>
        
        <!-- قسم رفع ملف Excel/CSV -->
        <div class="card" style="margin-bottom: 20px; border: 2px solid #2196F3;">
            <div class="card-header" style="background-color: #2196F3; color: white;">
                <h3>📥 تسجيل الطلاب من ملف Excel/CSV</h3>
            </div>
            <div style="padding: 20px;">
                <form method="POST" enctype="multipart/form-data" style="margin-bottom: 15px;" id="uploadForm" onsubmit="return validateFileUpload()">
                    <input type="hidden" name="action" value="upload_excel">
                    
                    <div style="margin-bottom: 15px;">
                        <label for="excel_file" style="display: block; margin-bottom: 8px; font-weight: bold;">
                            اختر ملف Excel أو CSV:
                        </label>
                        <input type="file" 
                               name="excel_file" 
                               id="excel_file" 
                               accept=".csv,.xlsx,.xls"
                               required
                               onchange="validateFileSelection(this)"
                               style="padding: 8px; width: 100%; max-width: 400px; border: 1px solid #ddd; border-radius: 4px;">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            الصيغ المدعومة: CSV, Excel (.xlsx, .xls) - الحد الأقصى: 10MB
                        </small>
                        <div id="fileInfo" style="margin-top: 8px; padding: 8px; background: #f0f0f0; border-radius: 4px; display: none;">
                            <small id="fileName" style="display: block; font-weight: bold;"></small>
                            <small id="fileSize" style="display: block; color: #666;"></small>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="padding: 10px 20px;" id="uploadBtn">
                        📤 رفع الملف وتسجيل الطلاب
                    </button>
                </form>
                
                <script>
                function validateFileSelection(input) {
                    const fileInfo = document.getElementById('fileInfo');
                    const fileName = document.getElementById('fileName');
                    const fileSize = document.getElementById('fileSize');
                    const uploadBtn = document.getElementById('uploadBtn');
                    
                    if (input.files && input.files.length > 0) {
                        const file = input.files[0];
                        const maxSizeInMB = 10;
                        
                        // تنسيق حجم الملف بشكل أفضل
                        let fileSizeText = '';
                        if (file.size < 1024) {
                            fileSizeText = 'الحجم: ' + file.size + ' بايت';
                        } else if (file.size < 1024 * 1024) {
                            const fileSizeInKB = (file.size / 1024).toFixed(2);
                            fileSizeText = 'الحجم: ' + fileSizeInKB + ' KB';
                        } else {
                            const fileSizeInMB = (file.size / (1024 * 1024)).toFixed(2);
                            fileSizeText = 'الحجم: ' + fileSizeInMB + ' MB';
                        }
                        
                        fileName.textContent = 'الملف المحدد: ' + file.name;
                        fileSize.textContent = fileSizeText;
                        
                        if (file.size > (maxSizeInMB * 1024 * 1024)) {
                            fileSize.textContent += ' ⚠ حجم الملف كبير جداً!';
                            fileSize.style.color = '#f44336';
                            uploadBtn.disabled = true;
                            uploadBtn.style.opacity = '0.5';
                            uploadBtn.style.cursor = 'not-allowed';
                        } else {
                            fileSize.style.color = '#666';
                            uploadBtn.disabled = false;
                            uploadBtn.style.opacity = '1';
                            uploadBtn.style.cursor = 'pointer';
                        }
                        
                        // التحقق من نوع الملف
                        const allowedTypes = ['csv', 'xlsx', 'xls'];
                        const fileExt = file.name.split('.').pop().toLowerCase();
                        
                        if (!allowedTypes.includes(fileExt)) {
                            fileName.textContent += ' ⚠ نوع الملف غير مدعوم!';
                            fileName.style.color = '#f44336';
                            uploadBtn.disabled = true;
                            uploadBtn.style.opacity = '0.5';
                            uploadBtn.style.cursor = 'not-allowed';
                        } else {
                            fileName.style.color = '#000';
                        }
                        
                        fileInfo.style.display = 'block';
                    } else {
                        fileInfo.style.display = 'none';
                        uploadBtn.disabled = false;
                        uploadBtn.style.opacity = '1';
                        uploadBtn.style.cursor = 'pointer';
                    }
                }
                
                function validateFileUpload() {
                    const fileInput = document.getElementById('excel_file');
                    
                    if (!fileInput.files || fileInput.files.length === 0) {
                        alert('⚠️ يرجى اختيار ملف أولاً');
                        return false;
                    }
                    
                    const file = fileInput.files[0];
                    const maxSizeInMB = 10;
                    
                    if (file.size > (maxSizeInMB * 1024 * 1024)) {
                        alert('⚠️ حجم الملف كبير جداً. الحد الأقصى 10MB');
                        return false;
                    }
                    
                    const allowedTypes = ['csv', 'xlsx', 'xls'];
                    const fileExt = file.name.split('.').pop().toLowerCase();
                    
                    if (!allowedTypes.includes(fileExt)) {
                        alert('⚠️ نوع الملف غير مدعوم. يرجى اختيار ملف CSV أو Excel (.xlsx, .xls)');
                        return false;
                    }
                    
                    // إظهار رسالة التحميل
                    const uploadBtn = document.getElementById('uploadBtn');
                    uploadBtn.disabled = true;
                    uploadBtn.innerHTML = '⏳ جاري رفع الملف...';
                    
                    return true;
                }
                </script>
                
                <div style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 15px;">
                    <h4 style="margin-top: 0;">📋 تنسيق الملف المطلوب:</h4>
                    <p><strong>يجب أن يحتوي الملف على عمود واحد على الأقل من:</strong></p>
                    <ul style="margin-right: 20px;">
                        <li><strong>معرّف الطالب الداخلي</strong> (id, student_id)</li>
                        <li><strong>الرقم الجامعي</strong> (student_number, رقم_جامعي, الرقم الجامعي)</li>
                        <li><strong>الرقم الوطني</strong> (national_id, nid, رقم_وطني, الرقم الوطني)</li>
                        <li><strong>اسم الطالب</strong> (student_name, name, full_name, الاسم, اسم الطالب)</li>
                    </ul>
                    
                    <div style="background: #e3f2fd; padding: 15px; border-radius: 5px; margin-top: 15px; border-right: 4px solid #2196F3;">
                        <h4 style="margin-top: 0; color: #1976d2;">🔍 كيف أتأكد من أن الطلاب موجودين ونشطين؟</h4>
                        <ol style="margin-right: 20px; line-height: 1.8;">
                            <li><strong>التحقق من التخصص:</strong> تأكد من أن جميع الطلاب في الملف ينتمون إلى نفس التخصص الذي تنتمي إليه المادة (<strong><?php echo htmlspecialchars($major ? $major['name_ar'] : 'غير محدد'); ?></strong>)</li>
                            <li><strong>التحقق من الحالة:</strong> تأكد من أن حالة الطلاب هي "نشط" (active) وليس "معلق" (suspended) أو "منسحب" (withdrawn)</li>
                            <li><strong>التحقق من الكلية:</strong> إذا لم تكن مديراً عاماً، تأكد من أن الطلاب ينتمون إلى نفس الكلية</li>
                            <li><strong>التحقق من البيانات:</strong> تأكد من صحة البيانات المدخلة (الرقم الجامعي، الرقم الوطني، الاسم)</li>
                        </ol>
                        <p style="margin-top: 10px;">
                            <a href="students.php?major_id=<?php echo $course['major_id']; ?>" class="btn btn-primary" style="padding: 8px 15px; text-decoration: none; display: inline-block;">
                                👥 عرض جميع الطلاب في هذا التخصص
                            </a>
                        </p>
                    </div>
                    <p style="margin-top: 10px;"><strong>مثال CSV:</strong></p>
                    <pre style="background: white; padding: 15px; border-radius: 4px; overflow-x: auto; font-family: 'Courier New', monospace; font-size: 13px; line-height: 1.6; border: 1px solid #ddd;">id,student_number,national_id,student_name

12,2021001,123456789012,أحمد محمد علي

13,2021002,234567890123,فاطمة أحمد حسن

,,345678901234,محمد خالد إبراهيم

,2021004,,سارة علي محمود</pre>
                    <p style="margin-top: 10px; font-size: 0.9em; color: #666;">
                        <strong>💡 ملاحظة:</strong> يمكن ترك بعض الحقول فارغة. يجب أن يحتوي كل سطر على قيمة واحدة على الأقل من: id، student_number، national_id، أو student_name.
                    </p>
                    <p style="margin-top: 10px;">
                        <a href="sample_students.csv" download class="btn btn-secondary" style="padding: 8px 15px;">
                            📥 تحميل ملف نموذجي (CSV)
                        </a>
                    </p>
                </div>
                
                <?php
                // عرض معلومات تشخيصية للمسؤولين
                if ($auth->hasRole('admin')): 
                    $upload_max_filesize = ini_get('upload_max_filesize');
                    $post_max_size = ini_get('post_max_size');
                    $max_execution_time = ini_get('max_execution_time');
                    $memory_limit = ini_get('memory_limit');
                ?>
                    <div style="background: #e3f2fd; padding: 15px; border-radius: 5px; margin-top: 15px; border: 1px solid #2196f3;">
                        <h4 style="margin-top: 0; color: #1976d2;">🔧 معلومات إعدادات PHP (للمسؤولين فقط):</h4>
                        <ul style="margin-right: 20px; line-height: 1.8;">
                            <li><strong>الحد الأقصى لحجم الملف (upload_max_filesize):</strong> <?php echo $upload_max_filesize; ?></li>
                            <li><strong>الحد الأقصى لحجم POST (post_max_size):</strong> <?php echo $post_max_size; ?></li>
                            <li><strong>الحد الأقصى لوقت التنفيذ (max_execution_time):</strong> <?php echo $max_execution_time; ?> ثانية</li>
                            <li><strong>حد الذاكرة (memory_limit):</strong> <?php echo $memory_limit; ?></li>
                        </ul>
                        <p style="margin-top: 10px; font-size: 14px;">
                            <strong>💡 ملاحظة:</strong> إذا كان الملف كبيراً، قد تحتاج إلى تعديل إعدادات PHP في ملف php.ini
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (empty($all_students)): ?>
            <div class="alert alert-error">
                <h4>⚠️ لا يوجد طلاب في هذا التخصص</h4>
                <p><strong>التخصص المختار:</strong> <?php echo $major ? htmlspecialchars($major['name_ar']) : 'غير محدد'; ?></p>
                <p><strong>الكلية:</strong> <?php echo $auth->hasRole('admin') ? 'جميع الكليات' : 'كلية المستخدم'; ?></p>
                <hr>
                <p><strong>الحل:</strong></p>
                <ol style="margin-right: 20px;">
                    <li>تأكد من وجود طلاب في التخصص من صفحة <a href="students.php">إدارة الطلاب</a></li>
                    <li>تأكد من أن الطلاب بحالة "نشط" (active)</li>
                    <li>إذا كنت تستخدم بيانات تجريبية، تأكد من تحميلها من <code>load_sample_data.php</code></li>
                </ol>
                <hr>
                <p>
                    <a href="students.php?major_id=<?php echo $course['major_id']; ?>" class="btn btn-primary">
                        ➕ إضافة طلاب في هذا التخصص
                    </a>
                    <a href="check_students.php?course_id=<?php echo $course_id; ?>&semester_id=<?php echo $semester_id; ?>" class="btn btn-info">
                        🔍 فحص تفصيلي للطلاب
                    </a>
                    <?php if (file_exists('load_sample_data.php')): ?>
                        <a href="load_sample_data.php" class="btn btn-secondary" target="_blank">
                            📥 تحميل بيانات تجريبية
                        </a>
                    <?php endif; ?>
                </p>
            </div>
        <?php elseif (empty($active_students)): ?>
            <div class="alert alert-warning">
                <h4>⚠️ لا يوجد طلاب نشطين في هذا التخصص</h4>
                <p>يوجد <?php echo count($all_students); ?> طالب في التخصص لكن جميعهم بحالة غير نشط.</p>
                <p><strong>الحل:</strong> يجب تفعيل الطلاب من صفحة <a href="students.php">إدارة الطلاب</a></p>
            </div>
        <?php else: ?>
        <div class="card" style="margin-top: 20px;">
            <div class="card-header" style="background-color: #4CAF50; color: white;">
                <h3>✏️ تسجيل الطلاب يدوياً</h3>
            </div>
            
            <form method="POST" id="registerForm" onsubmit="return validateForm()">
                <input type="hidden" name="action" value="register">
                
                <div style="margin-bottom: 15px;">
                    <button type="button" onclick="selectAll()" class="btn btn-secondary">تحديد الكل</button>
                    <button type="button" onclick="deselectAll()" class="btn btn-secondary">إلغاء التحديد</button>
                    <span id="selectedCount" style="margin-right: 15px; font-weight: bold; color: #667eea;">0 طالب محدد</span>
                </div>
                
                <div class="table-container">
                    <table>
                        <thead>
                            <tr>
                                <th>اختيار</th>
                                <th>الرقم الجامعي</th>
                                <th>اسم الطالب</th>
                                <th>الحالة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $available_count = 0;
                            foreach ($active_students as $student): 
                                if (!in_array($student['id'], $registered_ids)) {
                                    $available_count++;
                                }
                            ?>
                                <tr>
                                    <td>
                                        <?php if (!in_array($student['id'], $registered_ids)): ?>
                                            <input type="checkbox" 
                                                   name="students[]" 
                                                   value="<?php echo $student['id']; ?>" 
                                                   class="student-checkbox"
                                                   onchange="updateSelectedCount()">
                                        <?php else: ?>
                                            <span style="color: green; font-weight: bold;">✓</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                    <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                    <td>
                                        <?php if (in_array($student['id'], $registered_ids)): ?>
                                            <span style="color: green; font-weight: bold;">✓ مسجل</span>
                                        <?php else: ?>
                                            <span style="color: orange;">غير مسجل</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (count($all_students) > count($active_students)): ?>
                                <tr style="background-color: #fff3cd;">
                                    <td colspan="4" style="text-align: center; padding: 10px;">
                                        <small>⚠️ يوجد <?php echo count($all_students) - count($active_students); ?> طالب غير نشط (غير معروض)</small>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if ($available_count > 0): ?>
                    <div style="margin-top: 15px; padding: 15px; background: #e3f2fd; border-radius: 5px;">
                        <p><strong>عدد الطلاب المتاحين للتسجيل:</strong> <?php echo $available_count; ?> طالب</p>
                    </div>
                    <button type="submit" class="btn btn-primary" style="margin-top: 15px; padding: 12px 30px; font-size: 16px;">
                        ✅ تسجيل الطلاب المحددين
                    </button>
                <?php else: ?>
                    <div style="margin-top: 15px; padding: 15px; background: #f1f8f4; border-radius: 5px;">
                        <p style="color: #4caf50; font-weight: bold;">✓ جميع الطلاب النشطين مسجلون بالفعل في هذه المادة</p>
                    </div>
                <?php endif; ?>
            </form>
        </div>
        <?php endif; ?>
    </div>
    
    <script>
        function selectAll() {
            document.querySelectorAll('.student-checkbox').forEach(cb => cb.checked = true);
            updateSelectedCount();
        }
        
        function deselectAll() {
            document.querySelectorAll('.student-checkbox').forEach(cb => cb.checked = false);
            updateSelectedCount();
        }
        
        function updateSelectedCount() {
            const checked = document.querySelectorAll('.student-checkbox:checked').length;
            const countEl = document.getElementById('selectedCount');
            if (countEl) {
                countEl.textContent = checked + ' طالب محدد';
                countEl.style.color = checked > 0 ? '#4caf50' : '#667eea';
            }
        }
        
        function validateForm() {
            const checked = document.querySelectorAll('.student-checkbox:checked').length;
            if (checked === 0) {
                alert('⚠️ يرجى اختيار طالب واحد على الأقل للتسجيل');
                return false;
            }
            return confirm('هل تريد تسجيل ' + checked + ' طالب في هذه المادة؟');
        }
        
        // تحديث العدد عند تحميل الصفحة
        document.addEventListener('DOMContentLoaded', function() {
            updateSelectedCount();
        });
    </script>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

