<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';
require_once __DIR__ . '/classes/ThorndikeCalibrator.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();

// التحقق من وجود جدول semester_calibrations وإنشائه إذا لم يكن موجوداً
try {
    $table_exists = $db->fetchOne("SHOW TABLES LIKE 'semester_calibrations'");
    if (!$table_exists) {
        $db->query("
            CREATE TABLE IF NOT EXISTS semester_calibrations (
                id INT PRIMARY KEY AUTO_INCREMENT,
                semester_id INT NOT NULL,
                target_mean DECIMAL(5,2) NOT NULL,
                target_std DECIMAL(5,2) NOT NULL,
                calibrated_by INT NOT NULL,
                calibrated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (semester_id) REFERENCES semesters(id) ON DELETE CASCADE,
                FOREIGN KEY (calibrated_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_semester_calibration (semester_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (Exception $e) {
    // تجاهل الخطأ إذا كان الجدول موجوداً بالفعل
    error_log("Calibration table check: " . $e->getMessage());
}

// التحقق من وجود جدول calibrated_grades وإنشائه إذا لم يكن موجوداً
try {
    $table_exists = $db->fetchOne("SHOW TABLES LIKE 'calibrated_grades'");
    if (!$table_exists) {
        $db->query("
            CREATE TABLE IF NOT EXISTS calibrated_grades (
                id INT PRIMARY KEY AUTO_INCREMENT,
                grade_id INT NOT NULL,
                student_id INT NOT NULL,
                course_id INT NOT NULL,
                semester_id INT NOT NULL,
                original_marks DECIMAL(5,2) NOT NULL,
                original_letter_grade VARCHAR(5),
                original_points DECIMAL(3,2),
                calibrated_marks DECIMAL(5,2) NOT NULL,
                calibrated_letter_grade VARCHAR(5),
                calibrated_points DECIMAL(3,2),
                calibration_target_mean DECIMAL(5,2) NOT NULL,
                calibration_target_std DECIMAL(5,2) NOT NULL,
                calibrated_by INT NOT NULL,
                calibrated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (grade_id) REFERENCES grades(id) ON DELETE CASCADE,
                FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
                FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
                FOREIGN KEY (semester_id) REFERENCES semesters(id) ON DELETE CASCADE,
                FOREIGN KEY (calibrated_by) REFERENCES users(id) ON DELETE RESTRICT,
                UNIQUE KEY unique_grade_calibration (grade_id),
                INDEX idx_semester (semester_id),
                INDEX idx_student_course (student_id, course_id, semester_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (Exception $e) {
    error_log("Calibrated grades table check: " . $e->getMessage());
}

// التحقق من وجود عمود is_alternative في جدول courses وإضافته إذا لم يكن موجوداً
try {
    $course_columns = $db->fetchAll("SHOW COLUMNS FROM courses");
    $existing_course_columns = array_column($course_columns, 'Field');
    
    if (!in_array('is_alternative', $existing_course_columns)) {
        $db->query("ALTER TABLE courses ADD COLUMN is_alternative BOOLEAN DEFAULT FALSE AFTER is_required");
    }
} catch (Exception $e) {
    error_log("Courses columns check error: " . $e->getMessage());
}

// التحقق من وجود أعمدة المعايرة في جدول grades وإضافتها إذا لم تكن موجودة
try {
    $columns = $db->fetchAll("SHOW COLUMNS FROM grades");
    $existing_columns = array_column($columns, 'Field');
    
    $columns_to_add = [
        'calibrated_marks' => "ADD COLUMN calibrated_marks DECIMAL(5,2) NULL AFTER points",
        'calibrated_letter_grade' => "ADD COLUMN calibrated_letter_grade VARCHAR(5) NULL AFTER calibrated_marks",
        'calibrated_points' => "ADD COLUMN calibrated_points DECIMAL(3,2) NULL AFTER calibrated_letter_grade",
        'calibration_target_mean' => "ADD COLUMN calibration_target_mean DECIMAL(5,2) NULL AFTER calibrated_points",
        'calibration_target_std' => "ADD COLUMN calibration_target_std DECIMAL(5,2) NULL AFTER calibration_target_mean",
        'calibrated_at' => "ADD COLUMN calibrated_at TIMESTAMP NULL AFTER calibration_target_std",
        'calibrated_by' => "ADD COLUMN calibrated_by INT NULL AFTER calibrated_at"
    ];
    
    foreach ($columns_to_add as $column_name => $sql) {
        if (!in_array($column_name, $existing_columns)) {
            try {
                $db->query("ALTER TABLE grades " . $sql);
            } catch (Exception $e) {
                error_log("Error adding column $column_name: " . $e->getMessage());
            }
        }
    }
    
    // إضافة Foreign Key للعمود calibrated_by إذا لم يكن موجوداً
    if (in_array('calibrated_by', $existing_columns)) {
        try {
            $fks = $db->fetchAll("
                SELECT CONSTRAINT_NAME 
                FROM information_schema.KEY_COLUMN_USAGE 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'grades' 
                AND COLUMN_NAME = 'calibrated_by'
            ");
            
            if (empty($fks)) {
                $db->query("ALTER TABLE grades ADD FOREIGN KEY (calibrated_by) REFERENCES users(id) ON DELETE SET NULL");
            }
        } catch (Exception $e) {
            // تجاهل الخطأ إذا كان Foreign Key موجوداً بالفعل
            error_log("FK check: " . $e->getMessage());
        }
    }
    
    // التحقق من نوع حقل marks وتعديله إذا كان DECIMAL
    if (in_array('marks', $existing_columns)) {
        try {
            $marks_column = null;
            foreach ($columns as $col) {
                if ($col['Field'] == 'marks') {
                    $marks_column = $col;
                    break;
                }
            }
            
            if ($marks_column && (strpos($marks_column['Type'], 'decimal') !== false || strpos($marks_column['Type'], 'DECIMAL') !== false)) {
                // تعديل نوع الحقل إلى VARCHAR للسماح بإدخال حروف
                $db->query("ALTER TABLE grades MODIFY COLUMN marks VARCHAR(10) NULL");
            }
        } catch (Exception $e) {
            error_log("Error modifying marks column: " . $e->getMessage());
        }
    }
} catch (Exception $e) {
    error_log("Columns check error: " . $e->getMessage());
}

$college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$semester_number = isset($_GET['semester_number']) ? (int)$_GET['semester_number'] : 0;
$academic_year = isset($_GET['academic_year']) ? $_GET['academic_year'] : '';
$semester_id = 0;
$message = '';
$view_type = isset($_GET['view']) ? $_GET['view'] : 'original'; // 'original' or 'calibrated'
$target_mean = isset($_GET['target_mean']) ? (float)$_GET['target_mean'] : 70;
$target_std = isset($_GET['target_std']) ? (float)$_GET['target_std'] : 12;

// اسم الجامعة (يمكن تعديله من هنا)
$university_name = 'جامعة الشرق للعلوم والتكنولوجيا: Alsharg University of Science and Technology';

// دالة لتحويل رقم الفصل إلى نص عربي
function getSemesterName($number) {
    switch ((int)$number) {
        case 1:
            return 'الفصل الأول';
        case 2:
            return 'الفصل الثاني';
        case 3:
            return 'الفصل الثالث';
        case 4:
            return 'الفصل الرابع';
        case 5:
            return 'الفصل الخامس';
        case 6:
            return 'الفصل السادس';
        case 7:
            return 'الفصل السابع';
        case 8:
            return 'الفصل الثامن';
        default:
            return 'فصل ' . (int)$number;
    }
}

// دالة لتحويل المعدل الفصلي إلى تقدير حرفي
// دالة لتحويل الدرجة الرقمية إلى تقدير حرفي
// التقدير الجديد: A (80-100), B (70-79), C (60-69), D (50-59), F (<50)
function getLetterGradeFromMarks($marks) {
    if ($marks >= 80 && $marks <= 100) {
        return 'A';
    } elseif ($marks >= 70 && $marks <= 79) {
        return 'B';
    } elseif ($marks >= 60 && $marks <= 69) {
        return 'C';
    } elseif ($marks >= 50 && $marks <= 59) {
        return 'D';
    } else {
        return 'F';
    }
}

// دالة لتحويل المعدل إلى تقدير حرفي (للتوافق مع الكود القديم)
function getLetterGradeFromGPA($gpa) {
    // تحويل GPA من 5 إلى درجة رقمية تقريبية (GPA * 20)
    $estimated_marks = $gpa * 20;
    return getLetterGradeFromMarks($estimated_marks);
}

// دالة لحساب التقدير النصي بناءً على المعدل الفصلي وحالة الطالب (المعدل من 5)
// القواعد: >=4 ممتاز، >=3.5 جيد جداً، >=3 جيد، >=2.5 بدون رسوب مقبول، >=2.5 مع رسوب ملاحق، <2.5 إعادة السنة، <1.5 فصل بسبب المعدل
function getTextGrade($gpa, $status = '', $failed_credits = 0) {
    // معدل أقل من 1.5: فصل بسبب المعدل
    if ($gpa < 1.5) {
        return 'فصل';
    }
    // معدل أقل من 2.5: إعادة السنة
    if ($gpa < 2.5) {
        return 'إعادة السنة';
    }
    // معدل >= 2.5 و < 3: مقبول إذا لم يرسب، أو ملاحق إذا رسب في بعض المواد
    if ($gpa >= 2.5 && $gpa < 3) {
        if ($failed_credits > 0 || strpos($status, 'ملاحق') !== false) {
            return 'ملاحق';
        }
        return 'مقبول';
    }
    // معدل >= 3 و < 3.5: جيد
    if ($gpa >= 3 && $gpa < 3.5) {
        return 'جيد';
    }
    // معدل >= 3.5 و < 4: جيد جداً
    if ($gpa >= 3.5 && $gpa < 4) {
        return 'جيد جداً';
    }
    // معدل >= 4: ممتاز
    if ($gpa >= 4 && $gpa <= 5) {
        return 'ممتاز';
    }
    return '-';
}

// دالة لحساب حالة الطالب وملاحظاته بناءً على القواعد الأكاديمية
function calculateStudentStatus($student_grades, $courses_list, $total_semester_credits, $is_supplementary_student = false) {
    $failed_credits = 0;
    $failed_courses = [];
    $no_grade_courses = []; // المواد التي تحتوي على "بد"
    $is_frozen = false; // التحقق من وجود "ت" (تجميد)
    $has_cheating = false; // التحقق من وجود "غش"
    
    // حساب الساعات المعتمدة الراسبة والمواد الراسبة
    foreach ($courses_list as $course) {
        $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
        
        if ($grade) {
            $letter_grade = isset($grade['letter_grade']) ? trim($grade['letter_grade']) : '';
            $marks = isset($grade['marks']) ? $grade['marks'] : null;
            
            // معالجة الدرجات الخاصة
            $is_failed = false;
            
            // التحقق من وجود "ت" (تجميد) في الدرجة
            $marks_str = $marks !== null ? (string)$marks : '';
            $letter_grade_str = $letter_grade !== null ? (string)$letter_grade : '';
            if (strpos($marks_str, 'ت') !== false || strpos($letter_grade_str, 'ت') !== false) {
                $is_frozen = true;
            }
            
            // التحقق من القيم النصية في marks أولاً (إذا كانت marks نصية)
            if ($marks !== null && !is_numeric($marks)) {
                $marks_text = trim($marks);
                
                // إذا كانت "ب" (حرف عربي)، تعامل معها كـ "بد"
                if ($marks_text == 'ب') {
                    $marks_text = 'بد';
                }
                
                if ($marks_text == 'بد') {
                    // "بد" - لا تُحسب كراسبة
                    $is_failed = false;
                    // إضافة المادة إلى قائمة المواد بدون درجة
                    $no_grade_courses[] = [
                        'name_ar' => $course['name_ar'],
                        'code' => $course['code']
                    ];
                } elseif ($marks_text == 'غ' || $marks_text == 'غياب') {
                    // "غ" (غياب) - تُحسب كراسبة
                    $is_failed = true;
                } elseif ($marks_text == 'غش' || $marks_text == 'غشش') {
                    // "غش" (غش) - تُحسب كراسبة
                    $is_failed = true;
                    $has_cheating = true;
                } elseif (strpos($marks_text, 'ت') !== false) {
                    // "ت" (تجميد) - لا تُحسب كراسبة، ولكن يتم التعامل معها في حالة الطالب
                    $is_failed = false;
                    $is_frozen = true;
                } elseif ($marks_text == 'مع' || strpos($marks_text, 'مع') !== false) {
                    // "مع" (معفى) - لا تُحسب كراسبة ولا تدخل في حساب المعدل
                    $is_failed = false;
                } else {
                    // نص آخر غير معروف - تحقق من letter_grade إذا كان موجوداً
                    if ($letter_grade) {
                         $letter_grade_trimmed = trim($letter_grade);
                        if ($letter_grade_trimmed == 'ب') {
                            $letter_grade_trimmed = 'بد';
                        }
                        
                        if ($letter_grade_trimmed == 'بد') {
                            $is_failed = false;
                            // إضافة المادة إلى قائمة المواد بدون درجة
                            $no_grade_courses[] = [
                                'name_ar' => $course['name_ar'],
                                'code' => $course['code']
                            ];
                        } elseif ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب') {
                            $is_failed = true;
                } elseif ($letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش') {
                    $is_failed = true;
                    $has_cheating = true;
                            $has_cheating = true;
                        } elseif (strpos($letter_grade_trimmed, 'ت') !== false) {
                            // "ت" (تجميد) - لا تُحسب كراسبة، ولكن يتم التعامل معها في حالة الطالب
                            $is_failed = false;
                            $is_frozen = true;
                        } elseif ($letter_grade_trimmed == 'مع' || strpos($letter_grade_trimmed, 'مع') !== false) {
                            // "مع" (معفى) - لا تُحسب كراسبة ولا تدخل في حساب المعدل
                            $is_failed = false;
                        } elseif ($letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                            $is_failed = true;
                        } else {
                            // نص غير معروف - لا تُحسب كراسبة افتراضياً
                            $is_failed = false;
                        }
                    } else {
                        // لا marks نصية ولا letter_grade - لا تُحسب كراسبة
                        $is_failed = false;
                    }
                }
            }
            // التحقق من letter_grade إذا كانت marks رقمية أو null
            elseif ($letter_grade) {
                $letter_grade_trimmed = trim($letter_grade);
                // إذا كانت "ب" (حرف عربي)، تعامل معها كـ "بد"
                if ($letter_grade_trimmed == 'ب') {
                    $letter_grade_trimmed = 'بد';
                }
                
                if ($letter_grade_trimmed == 'بد') {
                    // "بد" - لا تُحسب كراسبة
                    $is_failed = false;
                    // إضافة المادة إلى قائمة المواد بدون درجة
                    $no_grade_courses[] = [
                        'name_ar' => $course['name_ar'],
                        'code' => $course['code']
                    ];
                } elseif ($letter_grade_trimmed == 'غ' || $letter_grade_trimmed == 'غياب') {
                    // "غ" (غياب) - تُحسب كراسبة
                    $is_failed = true;
                } elseif ($letter_grade_trimmed == 'غش' || $letter_grade_trimmed == 'غشش') {
                    // "غش" (غش) - تُحسب كراسبة
                    $is_failed = true;
                } elseif (strpos($letter_grade_trimmed, 'ت') !== false) {
                    // "ت" (تجميد) - لا تُحسب كراسبة، ولكن يتم التعامل معها في حالة الطالب
                    $is_failed = false;
                    $is_frozen = true;
                } elseif ($letter_grade_trimmed == 'مع' || strpos($letter_grade_trimmed, 'مع') !== false) {
                    // "مع" (معفى) - لا تُحسب كراسبة ولا تدخل في حساب المعدل
                    $is_failed = false;
                } elseif ($letter_grade_trimmed == 'F' || $letter_grade_trimmed == 'راسب') {
                    // "F" (راسب) - تُحسب كراسبة
                    $is_failed = true;
                }
            }
            // إذا كانت marks رقمية
            elseif ($marks !== null && is_numeric($marks)) {
                $is_failed = ((float)$marks < 50);
            }
            
            if ($is_failed) {
                $failed_credits += (float)$course['credit_hours'];
                $failed_courses[] = [
                    'name_ar' => $course['name_ar'],
                    'code' => $course['code'],
                    'credits' => (float)$course['credit_hours']
                ];
            }
        }
    }
    
    // إذا كان غياباً في كل المواد: فصل بسبب الغياب
    $total_courses = count($courses_list);
    $absence_count = 0;
    if ($total_courses > 0) {
        foreach ($courses_list as $course) {
            $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
            if (!$grade) continue;
            $letter_grade = isset($grade['letter_grade']) ? trim((string)$grade['letter_grade']) : '';
            $marks = isset($grade['marks']) ? $grade['marks'] : null;
            $is_absence = false;
            if ($marks !== null && !is_numeric($marks)) {
                $t = trim((string)$marks);
                if ($t === 'غ' || $t === 'غياب') $is_absence = true;
            }
            if (!$is_absence && $letter_grade !== '' && ($letter_grade === 'غ' || $letter_grade === 'غياب')) $is_absence = true;
            if ($is_absence) $absence_count++;
        }
        if ($absence_count === $total_courses) {
            return [
                'status' => 'فصل',
                'notes' => 'فصل بسبب الغياب',
                'failed_credits' => $failed_credits,
                'failed_courses' => $failed_courses
            ];
        }
    }
    
    $notes = '';
    $status = 'ناجح';
    
    // التحقق من وجود "ت" (تجميد) - إذا كان موجوداً، تكون حالة الطالب "تجميد"
    if ($is_frozen) {
        $status = 'تجميد';
        $notes = 'الطالب مجمد الفصل الدراسي';
        // إضافة "تحقيق" إذا كان هناك غش
        if ($has_cheating) {
            $notes .= ' | تحقيق';
        }
        return ['status' => $status, 'notes' => $notes, 'failed_credits' => $failed_credits, 'failed_courses' => $failed_courses];
    }
    
    // إضافة معلومات مواد "بد" في الملحوظات
    if (!empty($no_grade_courses)) {
        $no_grade_names = array_map(function($course) {
            return $course['name_ar'];
        }, $no_grade_courses);
        $notes = 'بديل: ' . implode('، ', $no_grade_names);
    }
    
    // تطبيق القواعد
    if ($failed_credits > 0) {
        $one_third = $total_semester_credits / 3;
        $two_thirds = ($total_semester_credits * 2) / 3;
        $one_sixth = $total_semester_credits / 6;
        
        if ($is_supplementary_student) {
            // للطلاب الجالسين لامتحانات الملاحق - النتيجة الفصلية: لا تظهر قرارات الإعادة
            if ($failed_credits > 0) {
                // إذا كان هناك مواد راسبة بعد الملاحق: يحمل المواد الراسبة
                $status = 'ناجح (يحمل المواد الراسبة)';
                $failed_course_names = array_map(function($course) {
                    return $course['name_ar'];
                }, $failed_courses);
                $notes = 'يحمل المواد التالية: ' . implode('، ', $failed_course_names);
                
                // إضافة معلومات مواد "بد" إذا كانت موجودة
                if (!empty($no_grade_courses)) {
                    $no_grade_names = array_map(function($course) {
                        return $course['name_ar'];
                    }, $no_grade_courses);
                    $notes .= ' | بديل: ' . implode('، ', $no_grade_names);
                }
                
                // إضافة "تحقيق" إذا كان هناك غش
                if ($has_cheating) {
                    $notes .= ' | تحقيق';
                }
            } else {
                // لا توجد مواد راسبة: ناجح
                $status = 'ناجح';
            }
        } else {
            // للطلاب العاديين - النتيجة الفصلية: لا تظهر قرارات الإعادة أو الفصل
            // فقط "ناجح" أو "ملاحق" بناءً على وجود مواد راسبة
            if ($failed_credits > 0) {
                // إذا كان هناك مواد راسبة: ملاحق
                $status = 'ملاحق';
                $failed_course_names = array_map(function($course) {
                    return $course['name_ar'];
                }, $failed_courses);
                
                // بناء الملحوظات: بديل + ملاحق
                $notes_parts = [];
                if (!empty($no_grade_courses)) {
                    $no_grade_names = array_map(function($course) {
                        return $course['name_ar'];
                    }, $no_grade_courses);
                    $notes_parts[] = 'بديل: ' . implode('، ', $no_grade_names);
                }
                if (!empty($failed_course_names)) {
                    $notes_parts[] = 'ملاحق في: ' . implode('، ', $failed_course_names);
                }
                $notes = implode(' + ', $notes_parts);
                
                // إضافة "تحقيق" إذا كان هناك غش
                if ($has_cheating) {
                    $notes .= ' | تحقيق';
                }
            } else {
                // لا توجد مواد راسبة: ناجح
                $status = 'ناجح';
            }
        }
    } else {
        // إذا لم يكن هناك مواد راسبة، أضف معلومات مواد "بد" فقط
        if (!empty($no_grade_courses)) {
            $no_grade_names = array_map(function($course) {
                return $course['name_ar'];
            }, $no_grade_courses);
            $notes = 'بديل: ' . implode('، ', $no_grade_names);
        }
        
        // إضافة "تحقيق" إذا كان هناك غش (حتى لو لم يكن هناك مواد راسبة)
        if ($has_cheating) {
            if (!empty($notes)) {
                $notes .= ' | تحقيق';
            } else {
                $notes = 'تحقيق';
            }
        }
    }
    
    return [
        'status' => $status,
        'notes' => $notes,
        'failed_credits' => $failed_credits,
        'failed_courses' => $failed_courses
    ];
}

/**
 * دالة لترتيب الطلاب حسب الحالة
 * الترتيب:
 * 1. ناجح (بدون ملاحق ولا بديل)
 * 2. ملاحق وبدائل معاً
 * 3. ملاحق فقط
 * 4. إعادة الفصل الدراسي
 * 5. حالات غش
 * 6. تجميد الفصل الدراسي
 */
function getStudentSortOrder($student, $use_calibrated = false) {
    $status = $use_calibrated ? ($student['calibrated_status'] ?? $student['status'] ?? '') : ($student['status'] ?? '');
    $notes = $use_calibrated ? ($student['calibrated_notes'] ?? $student['notes'] ?? '') : ($student['notes'] ?? '');
    $grades = $use_calibrated ? ($student['calibrated_grades'] ?? $student['grades'] ?? []) : ($student['grades'] ?? []);
    
    // التحقق من وجود غش في الدرجات
    $has_cheating = false;
    foreach ($grades as $grade) {
        if ($grade) {
            $letter_grade = isset($grade['letter_grade']) ? trim($grade['letter_grade']) : '';
            $marks = isset($grade['marks']) ? $grade['marks'] : null;
            
            if (($marks !== null && !is_numeric($marks) && (trim($marks) == 'غش' || trim($marks) == 'غشش')) ||
                ($letter_grade == 'غش' || $letter_grade == 'غشش')) {
                $has_cheating = true;
                break;
            }
        }
    }
    
    // التحقق من تجميد الفصل الدراسي (من حالة الطالب في قاعدة البيانات)
    $is_frozen = false;
    if (isset($student['status_db']) && ($student['status_db'] == 'suspended' || $student['status_db'] == 'frozen')) {
        $is_frozen = true;
    }
    
    // تحديد ترتيب الحالة
    if ($is_frozen) {
        return 6; // تجميد الفصل الدراسي
    } elseif ($has_cheating) {
        return 5; // حالات غش
    } elseif ($status == 'إعادة الفصل' || strpos($status, 'إعادة') !== false) {
        return 4; // إعادة الفصل الدراسي
    } elseif ($status == 'ملاحق' || strpos($status, 'ملاحق') !== false) {
        // التحقق من وجود بديل
        $has_alternative = strpos($notes, 'بديل') !== false || strpos($notes, 'بد') !== false;
        if ($has_alternative) {
            return 2; // ملاحق وبدائل معاً
        } else {
            return 3; // ملاحق فقط
        }
    } elseif ($status == 'ناجح' || $status == 'ناجح (يحمل المواد الراسبة)') {
        // التحقق من وجود بديل أو ملاحق
        $has_alternative = strpos($notes, 'بديل') !== false || strpos($notes, 'بد') !== false;
        $has_supplementary = strpos($notes, 'ملاحق') !== false;
        
        if ($has_alternative && $has_supplementary) {
            return 2; // ملاحق وبدائل معاً
        } elseif ($has_alternative || $has_supplementary) {
            if ($has_alternative) {
                return 2; // بديل فقط (نعتبره ملاحق وبدائل)
            } else {
                return 3; // ملاحق فقط
            }
        } else {
            return 1; // ناجح بدون ملاحق ولا بديل
        }
    } elseif ($status == 'فصل') {
        return 4; // فصل (نعتبره مثل إعادة الفصل)
    } else {
        // حالة غير معروفة - نضعها في النهاية
        return 7;
    }
}

/**
 * ترتيب الطلاب أبجدياً حسب الاسم دون تصنيف حسب النتيجة (نجاح/ملاحق/إلخ)
 */
function sortStudentsByStatus($students_data, $use_calibrated = false) {
    usort($students_data, function($a, $b) {
        $name_a = trim($a['full_name_ar'] ?? '');
        $name_b = trim($b['full_name_ar'] ?? '');
        return strcoll($name_a, $name_b);
    });
    return $students_data;
}

// جلب معلومات الفصل الدراسي
$semester_info = null;
$college_info = null;
$major_info = null;
$students_data = [];
$courses_list = [];

// إذا تم اختيار جميع المعايير، جلب معرف الفصل الدراسي
if ($college_id && $major_id && $semester_number && $academic_year) {
    $semester = $db->fetchOne(
        "SELECT id FROM semesters 
         WHERE major_id = ? AND semester_number = ? AND academic_year = ?",
        [$major_id, $semester_number, $academic_year]
    );
    if ($semester) {
        $semester_id = $semester['id'];
    }
}

// معالجة طلب حفظ الدرجات المعايرة (يجب أن يكون بعد تحديد semester_id)
$save_calibration_result = null;

if ($semester_id) {
    // معالجة طلب حفظ الدرجات المعايرة
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['save_calibration'])) {
        $calibrator = new ThorndikeCalibrator();
        $user = $auth->getCurrentUser();
        
        // جلب إعدادات المواد
        $course_settings = [];
        $global_min_grade = isset($_POST['global_min_grade']) ? (float)$_POST['global_min_grade'] : 0;
        $calibrate_all = isset($_POST['calibrate_all']) && $_POST['calibrate_all'] == '1';
        
        $courses_json = isset($_POST['courses']) ? $_POST['courses'] : '';
        $courses_decoded = !empty($courses_json) ? json_decode($courses_json, true) : [];
        if (is_array($courses_decoded)) {
            foreach ($courses_decoded as $course_id => $settings) {
                $course_id = (int)$course_id;
                $course_settings[$course_id] = [
                    'enabled' => isset($settings['enabled']) && $settings['enabled'] == '1',
                    'target_mean' => isset($settings['target_mean']) && $settings['target_mean'] !== '' ? (float)$settings['target_mean'] : null,
                    'target_std' => isset($settings['target_std']) && $settings['target_std'] !== '' ? (float)$settings['target_std'] : null,
                    'min_grade' => isset($settings['min_grade']) && $settings['min_grade'] !== '' ? (float)$settings['min_grade'] : $global_min_grade
                ];
            }
        }
        if (empty($course_settings) && isset($courses_list)) {
            foreach ($courses_list as $course) {
                $course_settings[$course['id']] = [
                    'enabled' => true,
                    'target_mean' => null,
                    'target_std' => null,
                    'min_grade' => $global_min_grade
                ];
            }
        }
        
        $save_calibration_result = $calibrator->saveCalibratedGrades(
            $semester_id,
            (float)$_POST['target_mean'],
            (float)$_POST['target_std'],
            $user['id'],
            $course_settings,
            $global_min_grade
        );
        
        if ($save_calibration_result['success']) {
            // تسجيل نشاط المعايرة
            require_once __DIR__ . '/classes/ActivityLogger.php';
            $logger = new ActivityLogger();
            $semester_info = $db->fetchOne("SELECT s.*, m.name_ar as major_name FROM semesters s JOIN majors m ON s.major_id = m.id WHERE s.id = ?", [$semester_id]);
            $logger->log('calibrate', 'semester', $semester_id, 'معايرة درجات الفصل الدراسي: ' . ($semester_info['major_name'] ?? '') . ' - ' . getSemesterName($semester_info['semester_number'] ?? 0) . ' - ' . ($semester_info['academic_year'] ?? ''));
            
            $message = '<div class="alert alert-success">✅ ' . htmlspecialchars($save_calibration_result['message']) . '</div>';
        } else {
            $message = '<div class="alert alert-danger">❌ ' . htmlspecialchars($save_calibration_result['message']) . '</div>';
        }
    }
    
    // معالجة طلب إلغاء المعايرة
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['cancel_calibration'])) {
        try {
            $db->getConnection()->beginTransaction();
            
            // حذف الدرجات المعايرة من الجدول المنفصل
            $db->query("DELETE FROM calibrated_grades WHERE semester_id = ?", [$semester_id]);
            
            // حذف معلومات المعايرة
            $db->query("DELETE FROM semester_calibrations WHERE semester_id = ?", [$semester_id]);
            
            // إلغاء المعايرة من جدول grades (اختياري - يمكن الاحتفاظ بها للتاريخ)
            // $db->query("UPDATE grades SET calibrated_marks = NULL, calibrated_letter_grade = NULL, calibrated_points = NULL, calibrated_at = NULL, calibrated_by = NULL WHERE semester_id = ?", [$semester_id]);
            
            $db->getConnection()->commit();
            $message = '<div class="alert alert-success">✅ تم إلغاء المعايرة بنجاح</div>';
        } catch (Exception $e) {
            $db->getConnection()->rollBack();
            $message = '<div class="alert alert-danger">❌ حدث خطأ أثناء إلغاء المعايرة: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // جلب معلومات الفصل الدراسي والكلية والتخصص
    $semester_info = $db->fetchOne(
        "SELECT s.*, m.name_ar as major_name, m.id as major_id, 
                c.name_ar as college_name, c.id as college_id
         FROM semesters s
         JOIN majors m ON s.major_id = m.id
         JOIN colleges c ON m.college_id = c.id
         WHERE s.id = ?",
        [$semester_id]
    );
    
    if ($semester_info) {
        // التحقق من الصلاحية
        if (!$auth->belongsToCollege($semester_info['college_id']) && !$auth->hasRole('admin')) {
            header('Location: semesters.php');
            exit;
        }
        
        $college_info = [
            'id' => $semester_info['college_id'],
            'name_ar' => $semester_info['college_name']
        ];
        
        $major_info = [
            'id' => $semester_info['major_id'],
            'name_ar' => $semester_info['major_name']
        ];
        
        // جلب جميع المواد في هذا الفصل
        $courses_list = $db->fetchAll(
            "SELECT c.id, c.code, c.name_ar, c.credit_hours, c.is_alternative
             FROM courses c
             WHERE c.semester_id = ?
             ORDER BY c.code",
            [$semester_id]
        );
        
        // حساب إجمالي الساعات المعتمدة للفصل الدراسي
        $total_semester_credits = 0;
        foreach ($courses_list as $course) {
            $total_semester_credits += (float)$course['credit_hours'];
        }
        
        // جلب جميع الطلاب المسجلين في هذا الفصل
        $students = $db->fetchAll(
            "SELECT DISTINCT s.id, s.student_number, s.full_name_ar
             FROM students s
             JOIN student_courses sc ON s.id = sc.student_id
             WHERE sc.semester_id = ?
             ORDER BY s.full_name_ar",
            [$semester_id]
        );
        
        // جلب الدرجات والمعدلات لكل طالب (الدرجات الأصلية)
        foreach ($students as $student) {
            $student_grades = [];
            
            // جلب درجات الطالب في جميع المواد
            foreach ($courses_list as $course) {
                $grade = $db->fetchOne(
                    "SELECT g.marks, g.letter_grade, g.points
                     FROM grades g
                     WHERE g.student_id = ? AND g.course_id = ? AND g.semester_id = ?",
                    [$student['id'], $course['id'], $semester_id]
                );
                
                $student_grades[$course['id']] = $grade ? [
                    'marks' => $grade['marks'],
                    'letter_grade' => $grade['letter_grade'],
                    'points' => $grade['points']
                ] : null;
            }
            
            // حساب المعدل الفصلي بالمنطق الموحد: مجموع (الدرجة/20 × الساعات) / مجموع الساعات
            $gpa_data = $gradeCalculator->calculateSemesterGPAFromGrades($student_grades, $courses_list);
            $gpa = $gpa_data['gpa'];
            $gradeCalculator->calculateSemesterGPA($student['id'], $semester_id);
            
            $letter_grade = $gpa > 0 ? getLetterGradeFromGPA($gpa) : '-';
            
            // التحقق من كون الطالب جالس لامتحانات الملاحق
            // يمكن تحديد ذلك من خلال البحث في فصول سابقة إذا كان الطالب لديه مواد راسبة في الفصل السابق
            // أو من خلال جدول منفصل لتتبع الطلاب الجالسين لامتحانات الملاحق
            // حالياً سنفترض أن الطالب ليس جالس لامتحانات الملاحق إلا إذا كان لديه مواد راسبة في الفصل السابق
            $is_supplementary_student = false;
            
            // يمكن إضافة منطق هنا للتحقق من الفصول السابقة أو من جدول منفصل
            // مثال: البحث في الفصل السابق للطالب للتحقق من وجود مواد راسبة
            
            // حساب حالة الطالب وملاحظاته
            $student_status = calculateStudentStatus(
                $student_grades, 
                $courses_list, 
                $total_semester_credits, 
                $is_supplementary_student
            );
            
            // حساب التقدير النصي بناءً على المعدل الفصلي وحالة الطالب
            $failed_credits = isset($student_status['failed_credits']) ? $student_status['failed_credits'] : 0;
            $text_grade = getTextGrade($gpa, $student_status['status'], $failed_credits);
            
            // جمع مواد البديل التي يجلس لها الطالب
            $alternative_courses = [];
            foreach ($courses_list as $course) {
                if (isset($course['is_alternative']) && $course['is_alternative']) {
                    $grade = isset($student_grades[$course['id']]) ? $student_grades[$course['id']] : null;
                    if ($grade && $grade['letter_grade'] && $grade['letter_grade'] !== 'بد') {
                        $alternative_courses[] = $course['name_ar'];
                    }
                }
            }
            
            // إضافة معلومات مواد البديل في عمود التقدير
            if (!empty($alternative_courses)) {
                $text_grade .= ' (بديل: ' . implode('، ', $alternative_courses) . ')';
            }
            
            $students_data[] = [
                'id' => $student['id'],
                'student_number' => $student['student_number'],
                'full_name_ar' => $student['full_name_ar'],
                'grades' => $student_grades,
                'gpa' => $gpa,
                'letter_grade' => $letter_grade,
                'text_grade' => $text_grade,
                'status' => $student_status['status'],
                'notes' => $student_status['notes'],
                'failed_credits' => $student_status['failed_credits'] ?? 0,
                'failed_courses' => $student_status['failed_courses'] ?? [],
                'status_db' => $student['status_db'] ?? null
            ];
        }
        
        // معايرة الدرجات باستخدام نظام ثوراندويك
        $calibrator = new ThorndikeCalibrator();
        
        // التحقق من وجود درجات معايرة محفوظة
        $has_saved_calibration = $calibrator->hasCalibratedGrades($semester_id);
        $calibration_info = $calibrator->getCalibrationInfo($semester_id);
        
        // إذا تم حفظ المعايرة للتو، استخدم القيم المحفوظة
        if ($save_calibration_result && $save_calibration_result['success']) {
            $has_saved_calibration = true;
            $calibration_info = $calibrator->getCalibrationInfo($semester_id);
        }
        
        // دائماً حساب الدرجات المعايرة للعرض (سواء كانت محفوظة أم لا)
        if ($view_type == 'calibrated') {
            // إذا كانت هناك درجات معايرة محفوظة، استخدمها
            if ($has_saved_calibration) {
                $saved_grades = $calibrator->getSavedCalibratedGrades($semester_id);
                $saved_grades_map = [];
                
                foreach ($saved_grades as $saved_grade) {
                    $key = $saved_grade['student_id'] . '_' . $saved_grade['course_id'];
                    $saved_grades_map[$key] = $saved_grade;
                }
                
                // استخدام الدرجات المحفوظة
                foreach ($students_data as &$student) {
                    $calibrated_grades = [];
                    
                    foreach ($courses_list as $course) {
                        $key = $student['id'] . '_' . $course['id'];
                        if (isset($saved_grades_map[$key])) {
                            $saved_grade = $saved_grades_map[$key];
                            
                            $calibrated_grades[$course['id']] = [
                                'marks' => $saved_grade['calibrated_marks'],
                                'letter_grade' => $saved_grade['calibrated_letter_grade'],
                                'points' => $saved_grade['calibrated_points'],
                                'original_marks' => $saved_grade['original_marks']
                            ];
                        } else {
                            $original_grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                            if ($original_grade) {
                                $calibrated_grades[$course['id']] = [
                                    'marks' => $original_grade['marks'],
                                    'letter_grade' => $original_grade['letter_grade'],
                                    'points' => $original_grade['points'],
                                    'original_marks' => $original_grade['marks']
                                ];
                            } else {
                                $calibrated_grades[$course['id']] = null;
                            }
                        }
                    }
                    
                    $student['calibrated_grades'] = $calibrated_grades;
                    $cal_gpa_data = $gradeCalculator->calculateSemesterGPAFromGrades($calibrated_grades, $courses_list);
                    $student['calibrated_gpa'] = $cal_gpa_data['gpa'];
                    $student['calibrated_letter_grade'] = $student['calibrated_gpa'] > 0 ? getLetterGradeFromGPA($student['calibrated_gpa']) : '-';
                    
                    // حساب حالة الطالب بناءً على الدرجات المعايرة
                    $calibrated_status = calculateStudentStatus(
                        $calibrated_grades, 
                        $courses_list, 
                        $total_semester_credits, 
                        false
                    );
                    $student['calibrated_status'] = $calibrated_status['status'];
                    $student['calibrated_notes'] = $calibrated_status['notes'];
                    $student['calibrated_failed_courses'] = $calibrated_status['failed_courses'] ?? [];
                    
                    // حساب التقدير النصي للدرجات المعايرة
                    $calibrated_failed_credits = isset($calibrated_status['failed_credits']) ? $calibrated_status['failed_credits'] : 0;
                    $student['calibrated_text_grade'] = getTextGrade($student['calibrated_gpa'], $calibrated_status['status'], $calibrated_failed_credits);
                    
                    // جمع مواد البديل التي يجلس لها الطالب (للدرجات المعايرة)
                    $alternative_courses_cal = [];
                    foreach ($courses_list as $course) {
                        if (isset($course['is_alternative']) && $course['is_alternative']) {
                            $cal_grade = isset($calibrated_grades[$course['id']]) ? $calibrated_grades[$course['id']] : null;
                            if ($cal_grade && $cal_grade['letter_grade'] && $cal_grade['letter_grade'] !== 'بد') {
                                $alternative_courses_cal[] = $course['name_ar'];
                            }
                        }
                    }
                    
                    // إضافة معلومات مواد البديل في عمود التقدير
                    if (!empty($alternative_courses_cal)) {
                        $student['calibrated_text_grade'] .= ' (بديل: ' . implode('، ', $alternative_courses_cal) . ')';
                    }
                }
                unset($student);
                
                // حساب الإحصائيات من الدرجات المحفوظة
                $calibration_data = ['statistics' => []];
                if ($calibration_info) {
                    $target_mean = $calibration_info['target_mean'];
                    $target_std = $calibration_info['target_std'];
                }
            } else {
                // حساب الدرجات المعايرة (غير محفوظة) - للعرض فقط
                // استخدام إعدادات افتراضية للعرض
                $course_settings = [];
                $global_min_grade = isset($_GET['global_min_grade']) ? (float)$_GET['global_min_grade'] : 0;
                $calibrate_all = !isset($_GET['calibrate_all']) || $_GET['calibrate_all'] == '1';
                $courses_param = isset($_GET['courses']) ? $_GET['courses'] : '';
                if (is_string($courses_param)) {
                    $courses_param = json_decode($courses_param, true);
                }
                if (is_array($courses_param)) {
                    foreach ($courses_param as $course_id => $settings) {
                        $course_id = (int)$course_id;
                        $course_settings[$course_id] = [
                            'enabled' => isset($settings['enabled']) && $settings['enabled'] == '1',
                            'target_mean' => isset($settings['target_mean']) && $settings['target_mean'] !== '' ? (float)$settings['target_mean'] : null,
                            'target_std' => isset($settings['target_std']) && $settings['target_std'] !== '' ? (float)$settings['target_std'] : null,
                            'min_grade' => isset($settings['min_grade']) && $settings['min_grade'] !== '' ? (float)$settings['min_grade'] : $global_min_grade
                        ];
                    }
                }
                
                $calibration_data = $calibrator->calibrateSemesterGrades($semester_id, null, $target_mean, $target_std, $course_settings, $global_min_grade);
                $calibrated_grades_map = [];
                
                // إنشاء خريطة للدرجات المعايرة
                if (!empty($calibration_data['calibrated_grades'])) {
                    foreach ($calibration_data['calibrated_grades'] as $cal_grade) {
                        $key = $cal_grade['student_id'] . '_' . $cal_grade['course_id'];
                        $calibrated_grades_map[$key] = $cal_grade;
                    }
                }
                
                // إضافة الدرجات المعايرة والمعدلات المعايرة للطلاب
                foreach ($students_data as &$student) {
                    $calibrated_grades = [];
                    
                    foreach ($courses_list as $course) {
                        $key = $student['id'] . '_' . $course['id'];
                        if (isset($calibrated_grades_map[$key])) {
                            $cal_grade = $calibrated_grades_map[$key];
                            
                            if (isset($cal_grade['was_calibrated']) && !$cal_grade['was_calibrated']) {
                                $original_grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                                if ($original_grade) {
                                    $calibrated_grades[$course['id']] = [
                                        'marks' => $original_grade['marks'],
                                        'letter_grade' => $original_grade['letter_grade'],
                                        'points' => $original_grade['points'],
                                        'original_marks' => $original_grade['marks'],
                                        'was_calibrated' => false,
                                        'calibration_reason' => $cal_grade['calibration_reason'] ?? 'غير معروف'
                                    ];
                                } else {
                                    $calibrated_grades[$course['id']] = null;
                                }
                            } else {
                                $grade_info = $calibrator->calculateGradeFromMarks($cal_grade['calibrated_marks']);
                                $calibrated_grades[$course['id']] = [
                                    'marks' => $cal_grade['calibrated_marks'],
                                    'letter_grade' => $grade_info['letter'],
                                    'points' => $grade_info['points'],
                                    'original_marks' => $cal_grade['original_marks'],
                                    'was_calibrated' => true
                                ];
                            }
                        } else {
                            $original_grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                            if ($original_grade) {
                                $calibrated_grades[$course['id']] = [
                                    'marks' => $original_grade['marks'],
                                    'letter_grade' => $original_grade['letter_grade'],
                                    'points' => $original_grade['points'],
                                    'original_marks' => $original_grade['marks']
                                ];
                            } else {
                                $calibrated_grades[$course['id']] = null;
                            }
                        }
                    }
                    
                    $student['calibrated_grades'] = $calibrated_grades;
                    $cal_gpa_data = $gradeCalculator->calculateSemesterGPAFromGrades($calibrated_grades, $courses_list);
                    $student['calibrated_gpa'] = $cal_gpa_data['gpa'];
                    $student['calibrated_letter_grade'] = $student['calibrated_gpa'] > 0 ? getLetterGradeFromGPA($student['calibrated_gpa']) : '-';
                    
                    // حساب حالة الطالب بناءً على الدرجات المعايرة
                    $calibrated_status = calculateStudentStatus(
                        $calibrated_grades, 
                        $courses_list, 
                        $total_semester_credits, 
                        false
                    );
                    $student['calibrated_status'] = $calibrated_status['status'];
                    $student['calibrated_notes'] = $calibrated_status['notes'];
                    $student['calibrated_failed_courses'] = $calibrated_status['failed_courses'] ?? [];
                    
                    // حساب التقدير النصي للدرجات المعايرة
                    $calibrated_failed_credits = isset($calibrated_status['failed_credits']) ? $calibrated_status['failed_credits'] : 0;
                    $student['calibrated_text_grade'] = getTextGrade($student['calibrated_gpa'], $calibrated_status['status'], $calibrated_failed_credits);
                    
                    // جمع مواد البديل التي يجلس لها الطالب (للدرجات المعايرة)
                    $alternative_courses_cal2 = [];
                    foreach ($courses_list as $course) {
                        if (isset($course['is_alternative']) && $course['is_alternative']) {
                            $cal_grade = isset($calibrated_grades[$course['id']]) ? $calibrated_grades[$course['id']] : null;
                            if ($cal_grade && $cal_grade['letter_grade'] && $cal_grade['letter_grade'] !== 'بد') {
                                $alternative_courses_cal2[] = $course['name_ar'];
                            }
                        }
                    }
                    
                    // إضافة معلومات مواد البديل في عمود التقدير
                    if (!empty($alternative_courses_cal2)) {
                        $student['calibrated_text_grade'] .= ' (بديل: ' . implode('، ', $alternative_courses_cal2) . ')';
                    }
                }
                unset($student);
            }
        } else {
            // للعرض الأصلي، لا نحتاج لمعالجة إضافية
            foreach ($students_data as &$student) {
                $student['calibrated_grades'] = [];
                $student['calibrated_gpa'] = 0;
                $student['calibrated_letter_grade'] = '-';
            }
            unset($student);
            $calibration_data = ['statistics' => []];
        }
    }
}

// جلب الكليات
$colleges = [];
if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $user = $auth->getCurrentUser();
    $colleges = $db->fetchAll(
        "SELECT id, name_ar FROM colleges WHERE id = ?",
        [$user['college_id']]
    );
}

// جلب التخصصات حسب الكلية المختارة
$majors = [];
if ($college_id) {
    $majors = $db->fetchAll(
        "SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
}

// جلب الفصول الدراسية حسب التخصص المختار
$semesters = [];
if ($major_id) {
    $semesters = $db->fetchAll(
        "SELECT DISTINCT semester_number FROM semesters 
         WHERE major_id = ? ORDER BY semester_number",
        [$major_id]
    );
}

// جلب السنوات الدراسية حسب التخصص والفصل المختارين
$academic_years = [];
if ($major_id && $semester_number) {
    $academic_years = $db->fetchAll(
        "SELECT DISTINCT academic_year FROM semesters 
         WHERE major_id = ? AND semester_number = ? ORDER BY academic_year DESC",
        [$major_id, $semester_number]
    );
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>استخراج نتائج الفصل الدراسي</title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .results-header {
            text-align: center;
            margin: 30px 0;
            padding: 20px;
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .results-header h1 {
            font-size: 1.8em;
            margin: 10px 0;
            color: #333;
        }
        .results-header h2 {
            font-size: 1.4em;
            margin: 8px 0;
            color: #667eea;
        }
        .results-header h3 {
            font-size: 1.2em;
            margin: 6px 0;
            color: #666;
        }
        .basmala {
            font-size: 1.5em;
            color: #667eea;
            margin-bottom: 20px;
            font-weight: bold;
        }
        .results-table {
            margin-top: 20px;
        }
        .results-table table {
            font-size: 0.8em;
            border-collapse: collapse;
            border: 2px solid #000;
        }
        .results-table th {
            background: #667eea;
            color: white;
            padding: 6px 5px;
            text-align: center;
            font-weight: bold;
            border: 2px solid #000;
        }
        .results-table td {
            padding: 5px 4px;
            text-align: center;
            border: 2px solid #000;
        }
        .results-table tr:nth-child(even) {
            background: #f9f9f9;
        }
        .gpa-cell {
            font-weight: bold;
            color: #667eea;
        }
        .letter-grade-cell {
            font-weight: bold;
            font-size: 1.05em;
        }
        .print-btns-group {
            display: inline-flex;
            gap: 8px;
            flex-wrap: wrap;
            margin-bottom: 10px;
        }
        .print-btns-group .print-header-btn {
            position: static;
        }
        .semester-selector {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .print-btn {
            margin-top: 20px;
            text-align: center;
        }
        .view-selector {
            background: white;
            padding: 15px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            text-align: center;
        }
        .view-selector .btn-group {
            display: inline-flex;
            gap: 10px;
            margin: 10px 0;
        }
        .view-selector .btn {
            padding: 10px 20px;
            border: 2px solid #667eea;
            background: white;
            color: #667eea;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .view-selector .btn.active {
            background: #667eea;
            color: white;
        }
        .view-selector .btn:hover {
            background: #667eea;
            color: white;
        }
        .calibration-params {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid #ddd;
        }
        .calibration-params label {
            margin: 0 10px;
        }
        .calibration-params input {
            width: 80px;
            padding: 5px;
            margin: 0 5px;
        }
        .results-section {
            margin-bottom: 40px;
        }
        .results-header {
            position: relative;
        }
        .print-header-btn {
            position: absolute;
            top: 10px;
            left: 10px;
            padding: 10px 18px;
            border: 1px solid #667eea;
            background: white;
            color: #667eea;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1em;
            font-weight: bold;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            box-shadow: 0 1px 4px rgba(0,0,0,0.1);
        }
        .print-header-btn:hover {
            background: #667eea;
            color: white;
        }
        .results-section h3 {
            background: #667eea;
            color: white;
            padding: 15px;
            margin: 0;
            border-radius: 10px 10px 0 0;
            text-align: center;
        }
        @media print {
            @page {
                size: A4;
                margin: 12mm;
                @bottom-center {
                    content: "صفحة " counter(page) " من " counter(pages);
                    font-size: 9pt;
                    font-family: inherit;
                }
            }
            @page landscape {
                size: A4 landscape;
                margin: 12mm;
                @bottom-center {
                    content: "صفحة " counter(page) " من " counter(pages);
                    font-size: 9pt;
                    font-family: inherit;
                }
            }
            body.print-landscape .results-section {
                page: landscape;
            }
            .semester-selector, .print-btn, .main-header, .main-footer, .view-selector, .no-print,
            .page-header, .container > .alert {
                display: none !important;
            }
            body { margin: 0; padding: 0; }
            .container { padding: 0; }
            .results-section {
                margin: 0;
                box-shadow: none;
            }
            .results-header {
                page-break-after: avoid;
            }
            .results-table thead {
                page-break-inside: avoid;
            }
            .results-table table {
                font-size: 8pt;
            }
            .results-table th,
            .results-table td {
                padding: 3px 4px;
            }
            .results-header .basmala {
                font-size: 14px;
            }
            .results-header h1 {
                font-size: 14px !important;
            }
            .results-header h2,
            .results-header h3 {
                font-size: 12px !important;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header no-print">
            <h2>استخراج نتائج الفصل الدراسي</h2>
        </div>
        
        <div class="semester-selector">
            <form method="GET" action="semester_results.php" id="selectionForm">
                <div class="form-group">
                    <label>اختر الكلية:</label>
                    <select name="college_id" id="college_select" required onchange="this.form.submit()">
                        <option value="">-- اختر الكلية --</option>
                        <?php foreach ($colleges as $college): ?>
                            <option value="<?php echo $college['id']; ?>" 
                                    <?php echo ($college_id == $college['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($college['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($college_id): ?>
                <div class="form-group">
                    <label>اختر التخصص:</label>
                    <select name="major_id" id="major_select" required onchange="this.form.submit()">
                        <option value="">-- اختر التخصص --</option>
                        <?php foreach ($majors as $major): ?>
                            <option value="<?php echo $major['id']; ?>" 
                                    <?php echo ($major_id == $major['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($major['name_ar']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($major_id): ?>
                <div class="form-group">
                    <label>اختر الفصل الدراسي:</label>
                    <select name="semester_number" id="semester_select" required onchange="this.form.submit()">
                        <option value="">-- اختر الفصل الدراسي --</option>
                        <?php foreach ($semesters as $sem): ?>
                            <option value="<?php echo $sem['semester_number']; ?>" 
                                    <?php echo ($semester_number == $sem['semester_number']) ? 'selected' : ''; ?>>
                                <?php echo getSemesterName($sem['semester_number']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($major_id && $semester_number): ?>
                <div class="form-group">
                    <label>اختر السنة الدراسية:</label>
                    <select name="academic_year" id="academic_year_select" required onchange="this.form.submit()">
                        <option value="">-- اختر السنة الدراسية --</option>
                        <?php foreach ($academic_years as $year): ?>
                            <option value="<?php echo htmlspecialchars($year['academic_year']); ?>" 
                                    <?php echo ($academic_year == $year['academic_year']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($year['academic_year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
            </form>
        </div>
        
        <script>
        // إزالة القيم التابعة من النموذج قبل الإرسال
        document.getElementById('college_select')?.addEventListener('change', function() {
            // عند تغيير الكلية، إزالة التخصص والفصل والسنة من النموذج
            const majorSelect = document.getElementById('major_select');
            const semesterSelect = document.getElementById('semester_select');
            const yearSelect = document.getElementById('academic_year_select');
            
            if (majorSelect) {
                majorSelect.removeAttribute('name');
                majorSelect.value = '';
            }
            if (semesterSelect) {
                semesterSelect.removeAttribute('name');
                semesterSelect.value = '';
            }
            if (yearSelect) {
                yearSelect.removeAttribute('name');
                yearSelect.value = '';
            }
        });
        
        document.getElementById('major_select')?.addEventListener('change', function() {
            // عند تغيير التخصص، إزالة الفصل والسنة من النموذج
            const semesterSelect = document.getElementById('semester_select');
            const yearSelect = document.getElementById('academic_year_select');
            
            if (semesterSelect) {
                semesterSelect.removeAttribute('name');
                semesterSelect.value = '';
            }
            if (yearSelect) {
                yearSelect.removeAttribute('name');
                yearSelect.value = '';
            }
        });
        
        document.getElementById('semester_select')?.addEventListener('change', function() {
            // عند تغيير الفصل، إزالة السنة من النموذج
            const yearSelect = document.getElementById('academic_year_select');
            if (yearSelect) {
                yearSelect.removeAttribute('name');
                yearSelect.value = '';
            }
        });
        </script>
        
        <?php if ($message): ?>
            <?php echo $message; ?>
        <?php endif; ?>
        
        <?php if ($semester_info && !empty($students_data)): ?>
            <div class="view-selector">
                <div class="btn-group">
                    <a href="?college_id=<?php echo $college_id; ?>&major_id=<?php echo $major_id; ?>&semester_number=<?php echo $semester_number; ?>&academic_year=<?php echo urlencode($academic_year); ?>&view=original&target_mean=<?php echo $target_mean; ?>&target_std=<?php echo $target_std; ?>" 
                       class="btn <?php echo $view_type == 'original' ? 'active' : ''; ?>">أصلية</a>
                    <a href="?college_id=<?php echo $college_id; ?>&major_id=<?php echo $major_id; ?>&semester_number=<?php echo $semester_number; ?>&academic_year=<?php echo urlencode($academic_year); ?>&view=calibrated&target_mean=<?php echo $target_mean; ?>&target_std=<?php echo $target_std; ?>" 
                       class="btn <?php echo $view_type == 'calibrated' ? 'active' : ''; ?>">
                        النتائج بعد المعايرة (ثوراندويك)
                    </a>
                </div>
                <?php if ($view_type == 'calibrated'): ?>
                <!-- شرح قانون ثوراندويك -->
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 10px; margin-bottom: 20px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 15px 0; color: white; font-size: 20px;">📐 قانون ثوراندويك (Thorndike Curve)</h3>
                    <div style="background: rgba(255,255,255,0.15); padding: 15px; border-radius: 8px; backdrop-filter: blur(10px);">
                        <p style="margin: 0 0 10px 0; font-size: 16px; line-height: 1.6;">
                            <strong>القانون الأساسي:</strong>
                        </p>
                        <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 5px; font-family: 'Courier New', monospace; font-size: 18px; text-align: center; margin: 10px 0; direction: ltr;">
                            <strong>الدرجة المعايرة = المتوسط المستهدف + (الدرجة الأصلية - المتوسط الأصلي) × (الانحراف المعياري المستهدف ÷ الانحراف المعياري الأصلي)</strong>
                        </div>
                        <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 5px; font-family: 'Courier New', monospace; font-size: 16px; text-align: center; margin: 10px 0; direction: ltr;">
                            <strong>Calibrated = μ<sub>target</sub> + (Original - μ<sub>original</sub>) × (σ<sub>target</sub> / σ<sub>original</sub>)</strong>
                        </div>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid rgba(255,255,255,0.3);">
                            <p style="margin: 0 0 8px 0;"><strong>📊 شرح المكونات:</strong></p>
                            <ul style="margin: 0; padding-right: 20px; line-height: 1.8;">
                                <li><strong>الدرجة المعايرة:</strong> الدرجة الجديدة بعد المعايرة</li>
                                <li><strong>الدرجة الأصلية:</strong> الدرجة الأصلية للطالب</li>
                                <li><strong>المتوسط الأصلي (μ<sub>original</sub>):</strong> متوسط درجات جميع الطلاب في المادة</li>
                                <li><strong>المتوسط المستهدف (μ<sub>target</sub>):</strong> المتوسط الذي نريد تحقيقه (افتراضي: 70)</li>
                                <li><strong>الانحراف المعياري الأصلي (σ<sub>original</sub>):</strong> قياس التشتت في الدرجات الأصلية</li>
                                <li><strong>الانحراف المعياري المستهدف (σ<sub>target</sub>):</strong> قياس التشتت المطلوب (افتراضي: 12)</li>
                            </ul>
                        </div>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid rgba(255,255,255,0.3);">
                            <p style="margin: 0 0 8px 0;"><strong>💡 كيفية العمل:</strong></p>
                            <ul style="margin: 0; padding-right: 20px; line-height: 1.8;">
                                <li>النظام يحسب المتوسط والانحراف المعياري للدرجات الأصلية في كل مادة</li>
                                <li>ثم يطبق القانون على كل درجة لجعل المتوسط = المتوسط المستهدف</li>
                                <li>ويجعل الانحراف المعياري = الانحراف المعياري المستهدف</li>
                                <li>الدرجة المعايرة تُحدّد بين 0 و 100 (لا تزيد ولا تقل عن ذلك)</li>
                            </ul>
                        </div>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid rgba(255,255,255,0.3);">
                            <p style="margin: 0 0 8px 0;"><strong>📝 مثال توضيحي:</strong></p>
                            <div style="background: rgba(255,255,255,0.15); padding: 12px; border-radius: 5px; font-size: 14px; line-height: 1.8;">
                                <p style="margin: 0 0 5px 0;"><strong>البيانات:</strong></p>
                                <ul style="margin: 0; padding-right: 20px;">
                                    <li>درجة طالب أصلية: 85</li>
                                    <li>المتوسط الأصلي للمادة: 75</li>
                                    <li>الانحراف المعياري الأصلي: 10</li>
                                    <li>المتوسط المستهدف: 70</li>
                                    <li>الانحراف المعياري المستهدف: 12</li>
                                </ul>
                                <p style="margin: 10px 0 5px 0;"><strong>الحساب:</strong></p>
                                <div style="background: rgba(255,255,255,0.2); padding: 10px; border-radius: 5px; font-family: 'Courier New', monospace; text-align: center; direction: ltr;">
                                    الدرجة المعايرة = 70 + (85 - 75) × (12 / 10)<br>
                                    = 70 + 10 × 1.2<br>
                                    = 70 + 12<br>
                                    <strong>= 82</strong>
                                </div>
                                <p style="margin: 10px 0 0 0; font-size: 13px;">
                                    <em>ملاحظة: الدرجة 82 ضمن النطاق المسموح (0-100)، لذلك تم قبولها</em>
                                </p>
                            </div>
                        </div>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid rgba(255,255,255,0.3); background: rgba(255,255,255,0.1); padding: 10px; border-radius: 5px;">
                            <p style="margin: 0; font-size: 14px;">
                                <strong>⚠️ ملاحظة:</strong> لا يمكن معايرة الدرجات إذا كانت جميع الدرجات متساوية (الانحراف المعياري = 0) أو إذا كان عدد الطلاب أقل من 2
                            </p>
                        </div>
                    </div>
                </div>
                
                <!-- إعدادات المعايرة -->
                <?php 
                $courses_from_get = [];
                if (isset($_GET['courses']) && is_string($_GET['courses'])) {
                    $dec = json_decode($_GET['courses'], true);
                    if (is_array($dec)) $courses_from_get = $dec;
                }
                $global_min_from_get = isset($_GET['global_min_grade']) ? (float)$_GET['global_min_grade'] : 0;
                $calibrate_all_from_get = !isset($_GET['calibrate_all']) || $_GET['calibrate_all'] == '1';
                ?>
                <div class="calibration-params">
                    <?php if ($has_saved_calibration && $calibration_info): ?>
                        <div style="background: #e8f5e9; padding: 10px; border-radius: 5px; margin-bottom: 10px;">
                            <strong>✅ تم حفظ المعايرة مسبقاً</strong><br>
                            <small>
                                المتوسط المستهدف: <?php echo $calibration_info['target_mean']; ?> | 
                                الانحراف المعياري: <?php echo $calibration_info['target_std']; ?><br>
                                تمت المعايرة بواسطة: <?php echo htmlspecialchars($calibration_info['calibrated_by_name'] ?? 'غير معروف'); ?> | 
                                التاريخ: <?php echo date('Y-m-d H:i', strtotime($calibration_info['calibrated_at'])); ?>
                            </small>
                        </div>
                    <?php endif; ?>
                    
                    <!-- اختيار المواد المراد معايرتها -->
                    <div style="margin-bottom: 20px; padding: 15px; background: #e3f2fd; border-radius: 8px; border: 1px solid #2196F3;">
                        <strong style="display: block; margin-bottom: 10px;">📋 اختر نطاق المعايرة:</strong>
                        <label style="display: block; margin-bottom: 8px; cursor: pointer;">
                            <input type="radio" name="calibration_mode" id="calibrate_all" value="all" <?php echo $calibrate_all_from_get ? 'checked' : ''; ?> onchange="toggleCourseSettings()">
                            معايرة جميع المواد
                        </label>
                        <label style="display: block; margin-bottom: 0; cursor: pointer;">
                            <input type="radio" name="calibration_mode" id="calibrate_some" value="some" <?php echo !$calibrate_all_from_get ? 'checked' : ''; ?> onchange="toggleCourseSettings()">
                            معايرة بعض المواد فقط (اختر من جدول المواد أدناه)
                        </label>
                    </div>
                    
                    <div style="margin-bottom: 20px;">
                        <label>المتوسط المستهدف:</label>
                        <input type="number" id="target_mean" value="<?php echo $target_mean; ?>" min="0" max="100" step="0.1">
                        <label>الانحراف المعياري المستهدف:</label>
                        <input type="number" id="target_std" value="<?php echo $target_std; ?>" min="0" max="50" step="0.1">
                        <button onclick="updateCalibration()" class="btn btn-primary" style="margin-right: 10px;">تحديث المعايرة</button>
                    </div>
                    
                    <!-- إعدادات المواد - لكل مادة: وسط مستهدف، انحراف معياري مستهدف، الحد الأدنى -->
                    <div style="background: #f9f9f9; padding: 20px; border-radius: 10px; margin-top: 20px; border: 2px solid #667eea;">
                        <h3 style="margin-top: 0; color: #2c3e50; margin-bottom: 15px;">⚙️ إعدادات معايرة المواد (لكل مادة)</h3>
                        <p style="color: #666; margin-bottom: 15px;">حدد لكل مادة: الوسط المستهدف، الانحراف المعياري المستهدف، والحد الأدنى للدرجات (الدرجات الأقل منه لا تدخل في المعايرة). <strong>إذا اخترت "معايرة بعض المواد فقط"</strong> استخدم عمود «معايرة هذه المادة» في الجدول (☑ نعم / ☐ لا) لتحديد المواد المراد معايرتها.</p>
                        
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 8px; font-weight: bold;">الحد الأدنى العام (افتراضي لكل المواد عند ترك الحقل فارغاً):</label>
                            <input type="number" id="global_min_grade" value="<?php echo $global_min_from_get; ?>" min="0" max="100" step="1" style="width: 80px; padding: 8px;">
                            <small style="color: #666; margin-right: 10px;">(الدرجات الأقل من هذا الحد لا تدخل في المعايرة)</small>
                        </div>
                        
                        <div id="course_settings_section" style="margin-top: 15px;">
                            <h4 style="margin-bottom: 15px; color: #555;">إعدادات كل مادة:</h4>
                            <p style="margin-bottom: 10px; padding: 8px; background: #fff3e0; border: 1px solid #ff9800; border-radius: 5px; font-size: 14px;">
                                <strong>💡 لاختيار المواد المراد معايرتها:</strong> اختر «معايرة بعض المواد فقط» أعلاه، ثم في الجدول أدناه استخدم عمود <strong>«معايرة هذه المادة»</strong> — حدد ☑ للمواد التي تريد معايرتها وأزل التحديد ☐ عن التي لا تريد معايرتها.
                            </p>
                            <div style="max-height: 450px; overflow: auto; border: 1px solid #ddd; padding: 15px; border-radius: 5px; background: white;">
                                <table style="width: 100%; border-collapse: collapse; min-width: 600px;">
                                    <thead>
                                        <tr style="background: #667eea; color: white;">
                                            <th style="padding: 10px; text-align: right; border: 1px solid #555;">اسم المادة</th>
                                            <th style="padding: 10px; text-align: center; border: 1px solid #555; background: #5a6fd6; min-width: 140px;">معايرة هذه المادة<br><small>☑ نعم / ☐ لا</small></th>
                                            <th style="padding: 10px; text-align: center; border: 1px solid #555;">الوسط المستهدف</th>
                                            <th style="padding: 10px; text-align: center; border: 1px solid #555;">الانحراف المعياري المستهدف</th>
                                            <th style="padding: 10px; text-align: center; border: 1px solid #555;">الحد الأدنى<br><small>(لا تدخل الدرجات الأقل)</small></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($courses_list as $course): 
                                            $c_settings = $courses_from_get[$course['id']] ?? [];
                                            $c_target_mean = isset($c_settings['target_mean']) ? $c_settings['target_mean'] : $target_mean;
                                            $c_target_std = isset($c_settings['target_std']) ? $c_settings['target_std'] : $target_std;
                                            $c_min_grade = isset($c_settings['min_grade']) ? $c_settings['min_grade'] : $global_min_from_get;
                                            $c_enabled = !isset($c_settings['enabled']) || $c_settings['enabled'] == '1';
                                        ?>
                                            <tr>
                                                <td style="padding: 8px; text-align: right; border: 1px solid #ddd;">
                                                    <?php echo htmlspecialchars($course['name_ar']); ?>
                                                </td>
                                                <td style="padding: 8px; text-align: center; border: 1px solid #ddd; background: #f5f5f5;">
                                                    <input type="checkbox" 
                                                           class="course_enable" 
                                                           data-course-id="<?php echo $course['id']; ?>" 
                                                           <?php echo $c_enabled ? 'checked' : ''; ?>
                                                           onchange="updateCourseRow(this)"
                                                           title="حدد لمعايرة هذه المادة">
                                                    <span style="font-size: 12px; color: #666;">معايرة</span>
                                                </td>
                                                <td style="padding: 8px; text-align: center; border: 1px solid #ddd;">
                                                    <input type="number" 
                                                           class="course_target_mean" 
                                                           data-course-id="<?php echo $course['id']; ?>" 
                                                           value="<?php echo htmlspecialchars($c_target_mean); ?>" 
                                                           min="0" max="100" step="0.1" 
                                                           style="width: 75px; padding: 5px;"
                                                           placeholder="وسط">
                                                </td>
                                                <td style="padding: 8px; text-align: center; border: 1px solid #ddd;">
                                                    <input type="number" 
                                                           class="course_target_std" 
                                                           data-course-id="<?php echo $course['id']; ?>" 
                                                           value="<?php echo htmlspecialchars($c_target_std); ?>" 
                                                           min="0" max="50" step="0.1" 
                                                           style="width: 75px; padding: 5px;"
                                                           placeholder="انحراف">
                                                </td>
                                                <td style="padding: 8px; text-align: center; border: 1px solid #ddd;">
                                                    <input type="number" 
                                                           class="course_min_grade" 
                                                           data-course-id="<?php echo $course['id']; ?>" 
                                                           value="<?php echo htmlspecialchars($c_min_grade); ?>" min="0" max="100" step="1" 
                                                           style="width: 70px; padding: 5px;"
                                                           placeholder="حد أدنى">
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php if (empty($courses_list)): ?>
                            <p style="margin-top: 10px; color: #d32f2f;">لا توجد مواد مسجلة لهذا الفصل (المواد مرتبطة بالفصل في قاعدة البيانات). لا يمكن عرض إعدادات المعايرة لكل مادة.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if (!$has_saved_calibration || ($has_saved_calibration && $calibration_info && ($target_mean != $calibration_info['target_mean'] || $target_std != $calibration_info['target_std']))): ?>
                        <form method="POST" id="saveCalibrationForm" style="margin-top: 20px;" onsubmit="return prepareCalibrationForm() && confirm('هل أنت متأكد من حفظ الدرجات المعايرة؟ سيتم تحديث جميع الدرجات المعايرة المحفوظة مسبقاً.');">
                            <input type="hidden" name="target_mean" id="form_target_mean" value="<?php echo $target_mean; ?>">
                            <input type="hidden" name="target_std" id="form_target_std" value="<?php echo $target_std; ?>">
                            <input type="hidden" name="global_min_grade" id="form_global_min_grade" value="0">
                            <input type="hidden" name="calibrate_all" id="form_calibrate_all" value="1">
                            <input type="hidden" name="courses" id="form_courses" value="">
                            <input type="hidden" name="save_calibration" value="1">
                            <button type="submit" class="btn" style="background: #4caf50; color: white; border: none; padding: 12px 30px; font-size: 16px;">
                                💾 حفظ الدرجات المعايرة
                            </button>
                        </form>
                    <?php endif; ?>
                    
                    <?php if ($has_saved_calibration): ?>
                        <form method="POST" style="display: inline-block; margin-right: 10px; margin-top: 20px;" onsubmit="return confirm('هل أنت متأكد من إلغاء المعايرة؟ سيتم حذف جميع الدرجات المعايرة المحفوظة.');">
                            <input type="hidden" name="cancel_calibration" value="1">
                            <button type="submit" class="btn" style="background: #f44336; color: white; border: none; padding: 12px 30px; font-size: 16px;">
                                🗑️ إلغاء المعايرة
                            </button>
                        </form>
                    <?php endif; ?>
                </div>
                
                <script>
                function toggleCourseSettings() {
                    const calibrateAll = document.getElementById('calibrate_all').checked;
                    document.querySelectorAll('.course_enable').forEach(function(cb) {
                        cb.disabled = calibrateAll;
                        if (calibrateAll) cb.checked = true;
                        const cid = cb.getAttribute('data-course-id');
                        document.querySelectorAll('.course_target_mean[data-course-id="' + cid + '"], .course_target_std[data-course-id="' + cid + '"], .course_min_grade[data-course-id="' + cid + '"]').forEach(function(inp) {
                            inp.disabled = !calibrateAll && !cb.checked;
                        });
                    });
                }
                document.addEventListener('DOMContentLoaded', function() { toggleCourseSettings(); });
                
                function updateCourseRow(checkbox) {
                    const courseId = checkbox.getAttribute('data-course-id');
                    document.querySelectorAll('.course_target_mean[data-course-id="' + courseId + '"], .course_target_std[data-course-id="' + courseId + '"], .course_min_grade[data-course-id="' + courseId + '"]').forEach(function(inp) {
                        inp.disabled = !checkbox.checked;
                    });
                }
                
                function prepareCalibrationForm() {
                    const calibrateAll = document.getElementById('calibrate_all').checked;
                    const targetMean = document.getElementById('target_mean').value;
                    const targetStd = document.getElementById('target_std').value;
                    const globalMinGrade = document.getElementById('global_min_grade').value;
                    
                    document.getElementById('form_target_mean').value = targetMean;
                    document.getElementById('form_target_std').value = targetStd;
                    document.getElementById('form_global_min_grade').value = globalMinGrade;
                    document.getElementById('form_calibrate_all').value = calibrateAll ? '1' : '0';
                    
                    const courses = {};
                    document.querySelectorAll('.course_enable').forEach(function(checkbox) {
                        const courseId = checkbox.getAttribute('data-course-id');
                        const tMean = document.querySelector('.course_target_mean[data-course-id="' + courseId + '"]');
                        const tStd = document.querySelector('.course_target_std[data-course-id="' + courseId + '"]');
                        const minGrade = document.querySelector('.course_min_grade[data-course-id="' + courseId + '"]');
                        courses[courseId] = {
                            enabled: (calibrateAll || checkbox.checked) ? '1' : '0',
                            target_mean: (tMean && tMean.value !== '') ? tMean.value : targetMean,
                            target_std: (tStd && tStd.value !== '') ? tStd.value : targetStd,
                            min_grade: (minGrade && minGrade.value !== '') ? minGrade.value : globalMinGrade
                        };
                    });
                    document.getElementById('form_courses').value = JSON.stringify(courses);
                    
                    return true;
                }
                
                // تحديث الحد الأدنى العام لجميع الحقول الفارغة
                document.getElementById('global_min_grade')?.addEventListener('input', function() {
                    const v = this.value;
                    document.querySelectorAll('.course_min_grade').forEach(function(input) {
                        if (input.value === '' || input.value === '0') input.value = v;
                    });
                });
                </script>
                
                <?php if ($has_saved_calibration && $view_type == 'calibrated'): ?>
                <!-- جدول الدرجات المعايرة المحفوظة -->
                <div style="margin-top: 30px; background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
                    <h3 style="color: #2c3e50; margin-bottom: 20px; border-bottom: 2px solid #667eea; padding-bottom: 10px;">
                        📊 جدول الدرجات المعايرة المحفوظة
                    </h3>
                    <div class="table-responsive">
                        <table class="data-table" style="width: 100%;">
                            <thead>
                                <tr>
                                    <th>الطالب</th>
                                    <th>المادة</th>
                                    <th>الدرجة الأصلية</th>
                                    <th>التقدير الأصلي</th>
                                    <th>الدرجة المعايرة</th>
                                    <th>التقدير المعاير</th>
                                    <th>النقاط المعايرة</th>
                                    <th>تاريخ المعايرة</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $calibrated_grades_list = $calibrator->getSavedCalibratedGrades($semester_id);
                                if (empty($calibrated_grades_list)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">لا توجد درجات معايرة محفوظة</td>
                                    </tr>
                                <?php else: 
                                    foreach ($calibrated_grades_list as $cal_grade):
                                        $student_info = $db->fetchOne("SELECT student_number, full_name_ar FROM students WHERE id = ?", [$cal_grade['student_id']]);
                                        $course_info = $db->fetchOne("SELECT code, name_ar FROM courses WHERE id = ?", [$cal_grade['course_id']]);
                                ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($student_info['student_number'] . ' - ' . $student_info['full_name_ar']); ?></td>
                                        <td><?php echo htmlspecialchars($course_info['name_ar']); ?></td>
                                        <td><?php echo is_numeric($cal_grade['original_marks']) ? number_format((float)$cal_grade['original_marks'], 2) : htmlspecialchars($cal_grade['original_marks']); ?></td>
                                        <td><?php echo htmlspecialchars(cleanLetterGrade($cal_grade['original_letter_grade'] ?? '-')); ?></td>
                                        <td><strong><?php echo number_format((float)$cal_grade['calibrated_marks'], 2); ?></strong></td>
                                        <td><strong><?php echo htmlspecialchars(cleanLetterGrade($cal_grade['calibrated_letter_grade'] ?? '-')); ?></strong></td>
                                        <td><?php echo $cal_grade['calibrated_points'] !== null ? number_format((float)$cal_grade['calibrated_points'], 2) : '-'; ?></td>
                                        <td><?php echo $cal_grade['calibrated_at'] ? date('Y-m-d H:i', strtotime($cal_grade['calibrated_at'])) : '-'; ?></td>
                                    </tr>
                                <?php 
                                    endforeach;
                                endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endif; ?>
                
                <script>
                function updateCalibration() {
                    const targetMean = document.getElementById('target_mean').value;
                    const targetStd = document.getElementById('target_std').value;
                    const globalMinGrade = document.getElementById('global_min_grade').value;
                    const calibrateAll = document.getElementById('calibrate_all').checked;
                    const url = new URL(window.location.href);
                    url.searchParams.set('target_mean', targetMean);
                    url.searchParams.set('target_std', targetStd);
                    url.searchParams.set('global_min_grade', globalMinGrade);
                    url.searchParams.set('calibrate_all', calibrateAll ? '1' : '0');
                    
                    const courses = {};
                    document.querySelectorAll('.course_enable').forEach(function(checkbox) {
                        const courseId = checkbox.getAttribute('data-course-id');
                        const tMean = document.querySelector('.course_target_mean[data-course-id="' + courseId + '"]');
                        const tStd = document.querySelector('.course_target_std[data-course-id="' + courseId + '"]');
                        const minGrade = document.querySelector('.course_min_grade[data-course-id="' + courseId + '"]');
                        courses[courseId] = {
                            enabled: (calibrateAll || checkbox.checked) ? '1' : '0',
                            target_mean: (tMean && tMean.value !== '') ? tMean.value : targetMean,
                            target_std: (tStd && tStd.value !== '') ? tStd.value : targetStd,
                            min_grade: (minGrade && minGrade.value !== '') ? minGrade.value : globalMinGrade
                        };
                    });
                    url.searchParams.set('courses', JSON.stringify(courses));
                    
                    window.location.href = url.toString();
                }
                </script>
                <?php endif; ?>
            </div>
            
            <?php if ($view_type == 'original'): ?>
                <div class="results-section">
                    <div class="results-header">
                        <div class="print-btns-group no-print">
                            <button type="button" class="print-header-btn" onclick="printResults('portrait')" title="طباعة بالطول">🖨️ طباعة بالطول</button>
                            <button type="button" class="print-header-btn" onclick="printResults('landscape')" title="طباعة بالعرض">🖨️ طباعة بالعرض</button>
                        </div>
                        <div class="basmala">بِسْمِ اللَّهِ الرَّحْمَنِ الرَّحِيمِ</div>
                        <h1 style="font-size: 18px;"><?php echo htmlspecialchars($university_name); ?></h1>
                        <h2><?php echo htmlspecialchars($college_info['name_ar']); ?></h2>
                        <?php if (trim($major_info['name_ar'] ?? '') !== 'الكلية'): ?><h3><?php echo htmlspecialchars($major_info['name_ar']); ?></h3><?php endif; ?>
                        <h3>نتائج <?php echo getSemesterName($semester_info['semester_number']); ?> - <?php echo htmlspecialchars($semester_info['academic_year']); ?></h3>
                    </div>
                    
                    <div class="results-table table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>م</th>
                                    <th>الرقم الجامعي</th>
                                    <th>اسم الطالب</th>
                                    <?php foreach ($courses_list as $course): ?>
                                        <th><?php echo htmlspecialchars($course['name_ar']); ?></th>
                                    <?php endforeach; ?>
                                    <th>المعدل الفصلي</th>
                                    <th>التقدير</th>
                                    <th>ملاحظات</th>
                                </tr>
                                <tr>
                                    <th></th>
                                    <th></th>
                                    <th>الساعات</th>
                                    <?php foreach ($courses_list as $course): ?>
                                        <th><?php echo (float)$course['credit_hours']; ?></th>
                                    <?php endforeach; ?>
                                    <th></th>
                                    <th></th>
                                    <th></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                // ترتيب الطلاب حسب الحالة (قبل المعايرة)
                                $sorted_students = sortStudentsByStatus($students_data, false);
                                $serial = 1;
                                foreach ($sorted_students as $student): 
                                ?>
                                    <tr>
                                        <td><?php echo $serial++; ?></td>
                                        <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                        <?php foreach ($courses_list as $course): ?>
                                            <td>
                                                <?php 
                                                // النتيجة الأصلية: عمود المادة والدرجة فقط (بدون بعد المعايرة)
                                                $grade = $student['grades'][$course['id']];
                                                if ($grade) {
                                                    if (isset($grade['marks']) && $grade['marks'] !== null && $grade['marks'] !== '') {
                                                        echo is_numeric($grade['marks']) ? number_format((float)$grade['marks'], 0) : htmlspecialchars(trim($grade['marks']));
                                                    } elseif (isset($grade['letter_grade']) && trim($grade['letter_grade']) !== '') {
                                                        echo htmlspecialchars(cleanLetterGrade($grade['letter_grade']));
                                                    } else {
                                                        echo '-';
                                                    }
                                                } else {
                                                    echo '-';
                                                }
                                                ?>
                                            </td>
                                        <?php endforeach; ?>
                                        <td class="gpa-cell">
                                            <?php echo $student['gpa'] > 0 ? number_format($student['gpa'], 2) : '-'; ?>
                                        </td>
                                        <td class="letter-grade-cell">
                                            <?php echo isset($student['text_grade']) ? htmlspecialchars($student['text_grade']) : '-'; ?>
                                        </td>
                                        <td style="text-align: right; font-size: 0.9em;">
                                            <?php 
                                            // النتيجة الفصلية: يجب أن تكون متسقة مع التقدير - لا يمكن أن يكون التقدير "فصل" والملاحظات "ناجح"
                                            if (!empty($student['notes']) || !empty($student['status'])) {
                                                $status_color = '';
                                                $display_status = $student['status'] ?? 'ناجح';
                                                $text_grade_val = $student['text_grade'] ?? '';
                                                
                                                // إذا كان التقدير "فصل" أو "إعادة السنة"، لا نعرض "ناجح" أبداً - يجب أن تكون النتيجة متسقة مع التقدير
                                                if (strpos($text_grade_val, 'فصل') !== false || strpos($text_grade_val, 'إعادة السنة') !== false) {
                                                    if (strpos($text_grade_val, 'فصل') !== false) {
                                                        $display_status = 'فصل';
                                                        $status_color = 'color: #f44336; font-weight: bold;';
                                                        $display_notes = (isset($student['notes']) && strpos($student['notes'], 'فصل بسبب الغياب') !== false) ? 'فصل بسبب الغياب' : 'فصل بسبب انخفاض المعدل';
                                                    } else {
                                                        $display_status = 'إعادة السنة';
                                                        $status_color = 'color: #ff9800; font-weight: bold;';
                                                        $display_notes = 'إعادة السنة بسبب انخفاض المعدل';
                                                    }
                                                } else {
                                                    // لا نعرض "ناجح" للطالب الذي حالته إعادة أو فصل - نبقى على الحالة الفعلية
                                                    if (strpos($display_status, 'إعادة') !== false || strpos($display_status, 'فصل') !== false) {
                                                        if (!empty($student['notes']) && strpos($student['notes'], 'ملاحق') !== false) {
                                                            $display_status = 'ملاحق';
                                                            $status_color = 'color: #2196F3; font-weight: bold;';
                                                        } else {
                                                            if (strpos($display_status, 'إعادة') !== false) {
                                                                $status_color = 'color: #ff9800; font-weight: bold;';
                                                            } elseif (strpos($display_status, 'فصل') !== false) {
                                                                $status_color = 'color: #f44336; font-weight: bold;';
                                                            }
                                                        }
                                                    }
                                                    if (strpos($display_status, 'ملاحق') !== false) {
                                                        $status_color = 'color: #2196F3; font-weight: bold;';
                                                        $failed_names = array_map(function($c) { return $c['name_ar']; }, $student['failed_courses'] ?? []);
                                                        $display_notes = !empty($failed_names) ? 'ملاحق: ' . implode('، ', $failed_names) : 'ملاحق';
                                                    } elseif (strpos($display_status, 'تجميد') !== false) {
                                                        $status_color = 'color: #9C27B0; font-weight: bold;';
                                                    }
                                                    if (strpos($display_status, 'ملاحق') === false) {
                                                        $display_notes = $student['notes'] ?? '';
                                                        $display_notes = preg_replace('/راسب في أكثر من.*?\)/', '', $display_notes);
                                                        $display_notes = preg_replace('/إعادة.*?الفصل/', '', $display_notes);
                                                        $display_notes = preg_replace('/فصل.*?الساعات/', '', $display_notes);
                                                        $display_notes = trim($display_notes, ' |');
                                                    }
                                                }
                                                
                                                // إذا كان الطالب إعادة أو فصل: يترك عمود الملاحظات خالياً
                                                if (strpos($display_status, 'فصل') !== false || strpos($display_status, 'إعادة') !== false) {
                                                    echo '';
                                                } else {
                                                    echo '<div style="' . $status_color . '">' . htmlspecialchars($display_status) . '</div>';
                                                    if (!empty($display_notes)) {
                                                        echo '<div style="margin-top: 5px; color: #666;">' . htmlspecialchars($display_notes) . '</div>';
                                                    }
                                                }
                                            } else {
                                                echo '-';
                                            }
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php else: ?>
                <!-- النتائج بعد المعايرة -->
                <div class="results-section">
                    <h3 class="no-print">النتائج بعد المعايرة (نظام ثوراندويك)</h3>
                    <div class="results-header">
                        <div class="print-btns-group no-print">
                            <button type="button" class="print-header-btn" onclick="printResults('portrait')" title="طباعة بالطول">🖨️ طباعة بالطول</button>
                            <button type="button" class="print-header-btn" onclick="printResults('landscape')" title="طباعة بالعرض">🖨️ طباعة بالعرض</button>
                        </div>
                        <div class="basmala">بِسْمِ اللَّهِ الرَّحْمَنِ الرَّحِيمِ</div>
                        <h1 style="font-size: 18px;"><?php echo htmlspecialchars($university_name); ?></h1>
                        <h2><?php echo htmlspecialchars($college_info['name_ar']); ?></h2>
                        <?php if (trim($major_info['name_ar'] ?? '') !== 'الكلية'): ?><h3><?php echo htmlspecialchars($major_info['name_ar']); ?></h3><?php endif; ?>
                        <h3>نتائج <?php echo getSemesterName($semester_info['semester_number']); ?> - <?php echo htmlspecialchars($semester_info['academic_year']); ?></h3>
                        <h4 class="no-print" style="color: #999; margin-top: 10px;">النتائج بعد المعايرة باستخدام نظام ثوراندويك</h4>
                        <p class="no-print" style="color: #666; margin-top: 10px;">
                            المتوسط المستهدف: <?php echo $target_mean; ?> | 
                            الانحراف المعياري المستهدف: <?php echo $target_std; ?>
                        </p>
                        <?php 
                        // عرض معلومات تشخيصية عن المعايرة
                        if (!empty($calibration_data['statistics'])):
                            $has_changes = false;
                            foreach ($calibration_data['statistics'] as $stats) {
                                if (abs($stats['original_mean'] - $stats['calibrated_mean']) > 0.01 || 
                                    abs($stats['original_std'] - $stats['calibrated_std']) > 0.01) {
                                    $has_changes = true;
                                    break;
                                }
                            }
                            if (!$has_changes && !empty($calibration_data['statistics'])):
                        ?>
                            <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; margin-top: 15px; border-radius: 5px;">
                                <strong>ℹ️ ملاحظة:</strong> الدرجات المعايرة قريبة جداً من الأصلية لأن:
                                <ul style="margin-right: 20px; margin-top: 10px;">
                                    <li>المتوسط الأصلي قريب من المتوسط المستهدف (<?php echo $target_mean; ?>)</li>
                                    <li>الانحراف المعياري الأصلي قريب من المستهدف (<?php echo $target_std; ?>)</li>
                                </ul>
                                <p style="margin-top: 10px;"><strong>لرؤية تغييرات أكبر:</strong> غيّر المتوسط المستهدف أو الانحراف المعياري المستهدف</p>
                            </div>
                        <?php 
                            endif;
                        endif;
                        ?>
                        <?php 
                        // التحقق من وجود درجات أصلية
                        $has_any_grades = false;
                        $grades_count = 0;
                        foreach ($students_data as $student) {
                            foreach ($courses_list as $course) {
                                if (isset($student['grades'][$course['id']]) && $student['grades'][$course['id']] !== null) {
                                    $has_any_grades = true;
                                    $grades_count++;
                                }
                            }
                        }
                        
                        // التحقق من وجود درجات معايرة
                        $has_any_calibrated = false;
                        $calibrated_count = 0;
                        foreach ($students_data as $student) {
                            if (isset($student['calibrated_grades'])) {
                                foreach ($courses_list as $course) {
                                    if (isset($student['calibrated_grades'][$course['id']]) && $student['calibrated_grades'][$course['id']] !== null) {
                                        $has_any_calibrated = true;
                                        $calibrated_count++;
                                    }
                                }
                            }
                        }
                        
                        if (!$has_any_grades): 
                        ?>
                            <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; margin-top: 15px; border-radius: 5px;">
                                <strong>⚠️ تنبيه:</strong> لا توجد درجات أصلية للطلاب في هذا الفصل الدراسي. يجب إدخال الدرجات الأصلية أولاً قبل إمكانية معايرتها.
                                <div style="margin-top: 10px;">
                                    <a href="enter_grades_simple.php" class="btn btn-primary">➡️ اذهب إلى إدخال الدرجات</a>
                                </div>
                            </div>
                        <?php elseif (!$has_any_calibrated && $has_any_grades): 
                            // جمع معلومات عن أسباب عدم المعايرة من الإحصائيات
                            $calibration_stats = $calibration_data['statistics'] ?? [];
                            $not_calibrated_courses = [];
                            foreach ($calibration_stats as $course_id => $stats) {
                                if (isset($stats['can_calibrate']) && !$stats['can_calibrate']) {
                                    $course_name = '';
                                    foreach ($courses_list as $course) {
                                        if ($course['id'] == $course_id) {
                                            $course_name = $course['name_ar'];
                                            break;
                                        }
                                    }
                                    $not_calibrated_courses[] = [
                                        'name' => $course_name,
                                        'reason' => $stats['calibration_reason'] ?? 'غير معروف',
                                        'student_count' => $stats['student_count'] ?? 0
                                    ];
                                }
                            }
                        ?>
                            <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 15px; margin-top: 15px; border-radius: 5px;">
                                <strong>⚠️ تنبيه:</strong> تم العثور على <?php echo $grades_count; ?> درجة أصلية، لكن لم يتم معايرتها. 
                                <?php if (!empty($not_calibrated_courses)): ?>
                                    <div style="margin-top: 15px; background: #fff; padding: 10px; border-radius: 4px; border: 1px solid #ddd;">
                                        <strong>المواد التي لم تُعاير:</strong>
                                        <ul style="margin: 10px 0; padding-right: 20px;">
                                            <?php foreach ($not_calibrated_courses as $course_info): ?>
                                                <li>
                                                    <strong><?php echo htmlspecialchars($course_info['name']); ?></strong>: 
                                                    <?php echo htmlspecialchars($course_info['reason']); ?>
                                                    <?php if ($course_info['student_count'] > 0): ?>
                                                        (<?php echo $course_info['student_count']; ?> طالب)
                                                    <?php endif; ?>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                <ul style="margin-right: 20px; margin-top: 10px;">
                                    <li><strong>الأسباب الشائعة لعدم المعايرة:</strong>
                                        <ul style="margin: 5px 0; padding-right: 20px;">
                                            <li>عدد الطلاب أقل من 2 (يحتاج على الأقل طالبين لحساب الانحراف المعياري)</li>
                                            <li>جميع الدرجات متساوية (الانحراف المعياري = 0)</li>
                                        </ul>
                                    </li>
                                </ul>
                                <div style="margin-top: 10px;">
                                    <p><strong>💡 ملاحظة:</strong> حتى لو لم يتم معايرة بعض المواد، ستظهر الدرجات الأصلية في الجدول مع علامة تحذير. يمكنك تمرير الماوس على العلامة لمعرفة السبب.</p>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="no-print" style="background: #e8f5e9; border: 1px solid #4caf50; padding: 15px; margin-top: 15px; border-radius: 5px;">
                                <strong>✅ تم العثور على:</strong> <?php echo $grades_count; ?> درجة أصلية و <?php echo $calibrated_count; ?> درجة معايرة
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="results-table table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>م</th>
                                    <th>الرقم الجامعي</th>
                                    <th>اسم الطالب</th>
                                    <?php foreach ($courses_list as $course): ?>
                                        <th><?php echo htmlspecialchars($course['name_ar']); ?></th>
                                    <?php endforeach; ?>
                                    <th>المعدل الفصلي</th>
                                    <th>التقدير</th>
                                    <th>ملاحظات</th>
                                </tr>
                                <tr>
                                    <th></th>
                                    <th></th>
                                    <th>الساعات</th>
                                    <?php foreach ($courses_list as $course): ?>
                                        <th><?php echo (float)$course['credit_hours']; ?></th>
                                    <?php endforeach; ?>
                                    <th></th>
                                    <th></th>
                                    <th></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                // ترتيب الطلاب: عند عدم وجود معايرة محفوظة نرتب حسب الأصل، وإلا حسب المعايرة
                                $sorted_students_cal = sortStudentsByStatus($students_data, $has_saved_calibration);
                                $serial = 1;
                                foreach ($sorted_students_cal as $student): 
                                ?>
                                    <tr>
                                        <td><?php echo $serial++; ?></td>
                                        <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                                        <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                                        <?php foreach ($courses_list as $course): ?>
                                            <td>
                                                <?php 
                                                // إذا لم تجرِ معايرة محفوظة: اعرض الدرجة الأصلية فقط
                                                if (!$has_saved_calibration):
                                                    $orig = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                                                    if ($orig) {
                                                        if (isset($orig['marks']) && $orig['marks'] !== null && $orig['marks'] !== '') {
                                                            echo is_numeric($orig['marks']) ? number_format((float)$orig['marks'], 0) : htmlspecialchars(trim($orig['marks']));
                                                        } elseif (isset($orig['letter_grade']) && trim($orig['letter_grade']) !== '') {
                                                            echo htmlspecialchars(cleanLetterGrade($orig['letter_grade']));
                                                        } else {
                                                            echo '-';
                                                        }
                                                    } else {
                                                        echo '-';
                                                    }
                                                else:
                                                    // التحقق من وجود درجات معايرة
                                                    if (isset($student['calibrated_grades']) && isset($student['calibrated_grades'][$course['id']])):
                                                        $cal_grade = $student['calibrated_grades'][$course['id']];
                                                        if ($cal_grade):
                                                            if (isset($cal_grade['was_calibrated']) && !$cal_grade['was_calibrated']):
                                                                if (isset($cal_grade['letter_grade'])):
                                                                    echo '<span style="color: #999;">' . htmlspecialchars(cleanLetterGrade($cal_grade['letter_grade'])) . '</span>';
                                                                else:
                                                                    echo '-';
                                                                endif;
                                                            else:
                                                                if (isset($cal_grade['letter_grade'])):
                                                                    echo htmlspecialchars(cleanLetterGrade($cal_grade['letter_grade']));
                                                                else:
                                                                    echo '-';
                                                                endif;
                                                            endif;
                                                        else:
                                                            echo '-';
                                                        endif;
                                                    else:
                                                        $original_grade = isset($student['grades'][$course['id']]) ? $student['grades'][$course['id']] : null;
                                                        if ($original_grade && isset($original_grade['letter_grade'])):
                                                            echo '<span style="color: #999;">' . htmlspecialchars(cleanLetterGrade($original_grade['letter_grade'])) . '</span>';
                                                        else:
                                                            echo '-';
                                                        endif;
                                                    endif;
                                                endif;
                                                ?>
                                            </td>
                                        <?php endforeach; ?>
                                        <td class="gpa-cell">
                                            <?php 
                                            if (!$has_saved_calibration) {
                                                echo $student['gpa'] > 0 ? number_format($student['gpa'], 2) : '-';
                                            } else {
                                                echo $student['calibrated_gpa'] > 0 ? number_format($student['calibrated_gpa'], 2) : '-';
                                            }
                                            ?>
                                        </td>
                                        <td class="letter-grade-cell">
                                            <?php 
                                            if (!$has_saved_calibration) {
                                                echo isset($student['text_grade']) ? htmlspecialchars($student['text_grade']) : '-';
                                            } else {
                                                echo isset($student['calibrated_text_grade']) ? htmlspecialchars($student['calibrated_text_grade']) : '-';
                                            }
                                            ?>
                                        </td>
                                        <td style="text-align: right; font-size: 0.9em;">
                                            <?php 
                                            // إذا لم تجرِ معايرة محفوظة: استخدم الحالة والملاحظات الأصلية
                                            $notes_status = $has_saved_calibration ? ($student['calibrated_status'] ?? '') : ($student['status'] ?? '');
                                            $notes_text = $has_saved_calibration ? ($student['calibrated_notes'] ?? '') : ($student['notes'] ?? '');
                                            $notes_text_grade = $has_saved_calibration ? ($student['calibrated_text_grade'] ?? '') : ($student['text_grade'] ?? '');
                                            if (!empty($notes_text) || !empty($notes_status)) {
                                                $status_color = '';
                                                $display_status = $notes_status ?: 'ناجح';
                                                $text_grade_val = $notes_text_grade;
                                                
                                                if (strpos($text_grade_val, 'فصل') !== false || strpos($text_grade_val, 'إعادة السنة') !== false) {
                                                    if (strpos($text_grade_val, 'فصل') !== false) {
                                                        $display_status = 'فصل';
                                                        $status_color = 'color: #f44336; font-weight: bold;';
                                                        $display_notes = (isset($student['notes']) && strpos($student['notes'], 'فصل بسبب الغياب') !== false) ? 'فصل بسبب الغياب' : 'فصل بسبب انخفاض المعدل';
                                                    } else {
                                                        $display_status = 'إعادة السنة';
                                                        $status_color = 'color: #ff9800; font-weight: bold;';
                                                        $display_notes = 'إعادة السنة بسبب انخفاض المعدل';
                                                    }
                                                } else {
                                                    if (strpos($display_status, 'إعادة') !== false || strpos($display_status, 'فصل') !== false) {
                                                        if (!empty($notes_text) && strpos($notes_text, 'ملاحق') !== false) {
                                                            $display_status = 'ملاحق';
                                                            $status_color = 'color: #2196F3; font-weight: bold;';
                                                        } else {
                                                            if (strpos($display_status, 'إعادة') !== false) {
                                                                $status_color = 'color: #ff9800; font-weight: bold;';
                                                            } elseif (strpos($display_status, 'فصل') !== false) {
                                                                $status_color = 'color: #f44336; font-weight: bold;';
                                                            } else {
                                                                $status_color = '';
                                                            }
                                                        }
                                                    } elseif (strpos($display_status, 'ملاحق') !== false) {
                                                        $status_color = 'color: #2196F3; font-weight: bold;';
                                                        $failed_courses_cal = $student['calibrated_failed_courses'] ?? [];
                                                        $failed_names_cal = array_map(function($c) { return $c['name_ar']; }, $failed_courses_cal);
                                                        $display_notes = !empty($failed_names_cal) ? 'ملاحق: ' . implode('، ', $failed_names_cal) : 'ملاحق';
                                                    } elseif (strpos($display_status, 'تجميد') !== false) {
                                                        $status_color = 'color: #9C27B0; font-weight: bold;';
                                                    } else {
                                                        $status_color = '';
                                                    }
                                                    if (strpos($display_status, 'ملاحق') === false) {
                                                        $display_notes = $notes_text;
                                                        $display_notes = preg_replace('/راسب في أكثر من.*?\)/', '', $display_notes);
                                                        $display_notes = preg_replace('/إعادة.*?الفصل/', '', $display_notes);
                                                        $display_notes = preg_replace('/فصل.*?الساعات/', '', $display_notes);
                                                        $display_notes = trim($display_notes, ' |');
                                                    }
                                                }
                                                
                                                // إذا كان الطالب إعادة أو فصل: يترك عمود الملاحظات خالياً
                                                if (strpos($display_status, 'فصل') !== false || strpos($display_status, 'إعادة') !== false) {
                                                    echo '';
                                                } else {
                                                    echo '<div style="' . $status_color . '">' . htmlspecialchars($display_status) . '</div>';
                                                    if (!empty($display_notes)) {
                                                        echo '<div style="margin-top: 5px; color: #666;">' . htmlspecialchars($display_notes) . '</div>';
                                                    }
                                                }
                                            } else {
                                                echo '-';
                                            }
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <?php if (!empty($calibration_data['statistics'])): ?>
                    <div class="card" style="margin-top: 20px;">
                        <div class="card-header">
                            <h3>إحصائيات المعايرة</h3>
                        </div>
                        <div class="table-container">
                            <table>
                                <thead>
                                    <tr>
                                        <th>المادة</th>
                                        <th>عدد الطلاب</th>
                                        <th>المتوسط الأصلي</th>
                                        <th>الانحراف المعياري الأصلي</th>
                                        <th>المتوسط بعد المعايرة</th>
                                        <th>الانحراف المعياري بعد المعايرة</th>
                                        <th>الفرق في المتوسط</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($courses_list as $course): ?>
                                        <?php if (isset($calibration_data['statistics'][$course['id']])): ?>
                                            <?php 
                                            $stats = $calibration_data['statistics'][$course['id']];
                                            $mean_diff = $stats['calibrated_mean'] - $stats['original_mean'];
                                            $std_diff = $stats['calibrated_std'] - $stats['original_std'];
                                            
                                            // حساب عدد الطلاب في هذه المادة
                                            $student_count = 0;
                                            foreach ($students_data as $student) {
                                                if (isset($student['grades'][$course['id']]) && $student['grades'][$course['id']] !== null) {
                                                    $student_count++;
                                                }
                                            }
                                            ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($course['name_ar']); ?></td>
                                                <td><?php echo $student_count; ?></td>
                                                <td><?php echo number_format($stats['original_mean'], 2); ?></td>
                                                <td><?php echo number_format($stats['original_std'], 2); ?></td>
                                                <td style="color: <?php echo abs($mean_diff) > 0.01 ? '#4caf50' : '#999'; ?>;">
                                                    <?php echo number_format($stats['calibrated_mean'], 2); ?>
                                                </td>
                                                <td style="color: <?php echo abs($std_diff) > 0.01 ? '#4caf50' : '#999'; ?>;">
                                                    <?php echo number_format($stats['calibrated_std'], 2); ?>
                                                </td>
                                                <td style="color: <?php echo abs($mean_diff) > 0.01 ? ($mean_diff > 0 ? '#4caf50' : '#f44336') : '#999'; ?>;">
                                                    <?php 
                                                    if (abs($mean_diff) > 0.01) {
                                                        echo ($mean_diff > 0 ? '+' : '') . number_format($mean_diff, 2);
                                                    } else {
                                                        echo '≈ 0';
                                                    }
                                                    ?>
                                                </td>
                                            </tr>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <div style="padding: 15px; background: #f5f5f5; border-top: 1px solid #ddd;">
                            <p><strong>ملاحظات:</strong></p>
                            <ul style="margin-right: 20px;">
                                <li>إذا كان الفرق في المتوسط ≈ 0، فهذا يعني أن المتوسط الأصلي قريب جداً من المستهدف</li>
                                <li>إذا كان الانحراف المعياري الأصلي = 0، لا يمكن المعايرة (جميع الطلاب لديهم نفس الدرجة)</li>
                                <li>لرؤية تغييرات أكبر، غيّر المتوسط المستهدف أو الانحراف المعياري المستهدف</li>
                            </ul>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <div class="print-btn no-print">
                <button type="button" onclick="printResults('portrait')" class="btn btn-primary">🖨️ طباعة بالطول</button>
                <button type="button" onclick="printResults('landscape')" class="btn btn-primary">🖨️ طباعة بالعرض</button>
            </div>
            <script>
            function printResults(orientation) {
                if (orientation === 'landscape') {
                    document.body.classList.add('print-landscape');
                } else {
                    document.body.classList.remove('print-landscape');
                }
                window.print();
            }
            window.addEventListener('afterprint', function() {
                document.body.classList.remove('print-landscape');
            });
            </script>
            
        <?php elseif ($semester_id && empty($students_data)): ?>
            <div class="alert alert-info">
                لا توجد نتائج للطلاب في هذا الفصل الدراسي.
            </div>
        <?php endif; ?>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

