<?php
// عدم عرض أخطاء PHP للمستخدم (الأمان) — تسجيلها في السجل فقط
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('log_errors', '1');

try {
    require_once __DIR__ . '/config/config.php';
    require_once __DIR__ . '/classes/Auth.php';
} catch (Exception $e) {
    die('خطأ في تحميل الملفات: ' . htmlspecialchars($e->getMessage()));
}

try {
    $auth = new Auth();
} catch (Exception $e) {
    die('خطأ في تهيئة Auth: ' . htmlspecialchars($e->getMessage()));
}

if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin') && !$auth->hasRole('registrar'))) {
    header('Location: index.php');
    exit;
}

try {
    $db = Database::getInstance();
} catch (Exception $e) {
    die('خطأ في الاتصال بقاعدة البيانات: ' . htmlspecialchars($e->getMessage()));
}

// دالة لتحويل رقم الفصل إلى نص عربي (الفصل الأول، الثاني، ...)
function getSemesterName($number) {
    switch ((int)$number) {
        case 1: return 'الفصل الأول';
        case 2: return 'الفصل الثاني';
        case 3: return 'الفصل الثالث';
        case 4: return 'الفصل الرابع';
        case 5: return 'الفصل الخامس';
        case 6: return 'الفصل السادس';
        case 7: return 'الفصل السابع';
        case 8: return 'الفصل الثامن';
        default: return 'فصل ' . (int)$number;
    }
}

// فحص وإصلاح AUTO_INCREMENT إذا لزم الأمر
try {
    // التحقق من وجود سجل ب id=0 وحذفه
    $zero_record = $db->fetchOne("SELECT id FROM students WHERE id = 0");
    if ($zero_record) {
        $db->query("DELETE FROM students WHERE id = 0");
    }
    
    // التحقق من أن حقل id لديه خاصية AUTO_INCREMENT
    $columns = $db->fetchAll("SHOW COLUMNS FROM students WHERE Field = 'id'");
    $has_auto_increment = false;
    if (!empty($columns)) {
        $has_auto_increment = (strpos($columns[0]['Extra'], 'auto_increment') !== false);
    }
    
    // إذا لم يكن لديه AUTO_INCREMENT، قم بإضافته
    if (!$has_auto_increment) {
        // جلب أعلى ID موجود
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        $new_auto_increment = max($max_id + 1, 1);
        
        // إضافة AUTO_INCREMENT للحقل
        $db->getConnection()->exec("ALTER TABLE students MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
        $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)$new_auto_increment);
    } else {
        // التحقق من أن AUTO_INCREMENT مضبوط بشكل صحيح
        $status = $db->fetchOne("SHOW TABLE STATUS LIKE 'students'");
        $auto_increment = $status['Auto_increment'] ?? null;
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        
        // إذا كان AUTO_INCREMENT أقل من أو يساوي أعلى ID موجود، قم بتحديثه
        if ($auto_increment !== null && $auto_increment <= $max_id) {
            $new_auto_increment = max($max_id + 1, 1);
            $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)$new_auto_increment);
        }
    }
} catch (PDOException $e) {
    // تجاهل الأخطاء في الفحص الاستباقي
    error_log("Auto-increment check error: " . $e->getMessage());
}

// معالجة AJAX requests
if (isset($_GET['action']) && $_GET['action'] == 'get_majors' && isset($_GET['college_id'])) {
    header('Content-Type: application/json');
    $college_id = (int)$_GET['college_id'];
    $majors = $db->fetchAll(
        "SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar",
        [$college_id]
    );
    // الكليات التي ليس بها تشعيب (مثل كلية القانون): إذا لم يكن لها أي تخصص، إنشاء تخصص افتراضي "الكلية"
    if (empty($majors) && $college_id > 0) {
        $college = $db->fetchOne("SELECT id, specialization_year FROM colleges WHERE id = ?", [$college_id]);
        if ($college && (empty($college['specialization_year']) || (int)$college['specialization_year'] <= 0)) {
            try {
                $has_is_general = $db->fetchOne("SHOW COLUMNS FROM majors WHERE Field = 'is_general'");
                if (!$has_is_general) {
                    @$db->getConnection()->exec("ALTER TABLE majors ADD COLUMN is_general TINYINT(1) NOT NULL DEFAULT 0 AFTER total_semesters");
                }
                $db->query(
                    "INSERT INTO majors (college_id, name_ar, name_en, code, total_semesters, is_general) VALUES (?, ?, ?, ?, ?, 0)",
                    [$college_id, 'الكلية', 'College', 'DEFAULT', 8]
                );
                $new_id = (int)$db->getConnection()->lastInsertId();
                $majors = [['id' => $new_id, 'name_ar' => 'الكلية']];
            } catch (PDOException $e) {
                // إن كان التخصص الافتراضي موجوداً مسبقاً (مثلاً رمز DEFAULT مكرر)، جلب القائمة مرة أخرى
                $majors = $db->fetchAll("SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar", [$college_id]);
            }
        }
    }
    echo json_encode($majors);
    exit;
}

$major_id = isset($_GET['major_id']) ? (int)$_GET['major_id'] : 0;
$filter_college_id = isset($_GET['college_id']) ? (int)$_GET['college_id'] : 0;
$message = '';

// معالجة رفع ملف Excel/CSV
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_excel') {
    if (isset($_FILES['excel_file']) && $_FILES['excel_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['excel_file'];
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        // التحقق من حجم الملف (10MB)
        $max_size = 10 * 1024 * 1024; // 10MB
        if ($file['size'] > $max_size) {
            $message = '<div class="alert alert-error">خطأ: حجم الملف كبير جداً. الحد الأقصى 10MB</div>';
        }
        // التحقق من نوع الملف
        elseif (!in_array($file_ext, ['csv', 'xlsx', 'xls'])) {
            $message = '<div class="alert alert-error">خطأ: نوع الملف غير مدعوم. يرجى رفع ملف CSV أو Excel (.xlsx, .xls)</div>';
        } else {
            try {
                require_once __DIR__ . '/classes/ExcelReader.php';
                $excelReader = new ExcelReader();
                
                $students_data = $excelReader->readFile($file['tmp_name'], $file_ext);
                
                if ($students_data === false || empty($students_data)) {
                    $message = '<div class="alert alert-error">خطأ: فشل قراءة الملف أو الملف فارغ. يرجى التحقق من صحة الملف والتنسيق</div>';
                } else {
                    $success_count = 0;
                    $error_count = 0;
                    $duplicate_count = 0;
                    $duplicates = []; // قائمة الطلاب المتخطين (موجودون مسبقاً) لعرض أسمائهم
                    $errors = [];
                    
                    // فحص سريع: عدد الأرقام الجامعية المختلفة في الملف (لتنبيه المستخدم إذا كان الملف مكرراً)
                    $file_student_numbers = [];
                    $cellStrForScan = function($val) {
                        if ($val === null || $val === '') return '';
                        if (is_numeric($val)) return (string)(int)(float)$val;
                        return trim((string)$val);
                    };
                    foreach ($students_data as $row) {
                        $row = array_combine(array_map('trim', array_keys($row)), array_values($row));
                        $sn = '';
                        foreach (['student_number', 'رقم_جامعي', 'الرقم الجامعي', 'student_id'] as $key) {
                            if (isset($row[$key])) {
                                $v = $cellStrForScan($row[$key]);
                                if ($v !== '') { $sn = $v; break; }
                            }
                        }
                        $fn = '';
                        foreach (['full_name_ar', 'name_ar', 'اسم_عربي', 'الاسم العربي', 'اسم', 'الاسم', 'student_name', 'name', 'full_name'] as $key) {
                            if (isset($row[$key])) {
                                $v = $cellStrForScan($row[$key]);
                                if ($v !== '') { $fn = $v; break; }
                            }
                        }
                        if ($sn !== '' || $fn !== '') {
                            $file_student_numbers[] = $sn;
                        }
                    }
                    $unique_in_file = count(array_unique(array_filter($file_student_numbers)));
                    $rows_with_data = count($file_student_numbers);
                    
                    // الحصول على الكلية الافتراضية
                    $default_college_id = $auth->hasRole('admin') ? (int)($_POST['college_id'] ?? 0) : ($_SESSION['college_id'] ?? 0);
                    
                    if ($default_college_id <= 0) {
                        $message = '<div class="alert alert-error">خطأ: يجب تحديد كلية</div>';
                    } else {
                        // بدء معاملة قاعدة البيانات
                        try {
                            $db->getConnection()->beginTransaction();
                            
                            foreach ($students_data as $row_index => $row) {
                                // تطبيع المفاتيح (إزالة مسافات زائدة) - Excel قد يرجّع مفاتيح بمسافات
                                $row = array_combine(array_map('trim', array_keys($row)), array_values($row));
                                
                                // دالة مساعدة لاستخراج قيمة نصية (Excel قد يرجّع أرقاماً أو float)
                                $cellStr = function($val) {
                                    if ($val === null || $val === '') return '';
                                    if (is_numeric($val)) return (string)(int)(float)$val;
                                    return trim((string)$val);
                                };
                                
                                // قراءة الحقول - البحث في عدة مفاتيح محتملة
                                $student_number = '';
                                foreach (['student_number', 'رقم_جامعي', 'الرقم الجامعي', 'student_id'] as $key) {
                                    if (isset($row[$key])) {
                                        $v = $cellStr($row[$key]);
                                        if ($v !== '') { $student_number = $v; break; }
                                    }
                                }
                                
                                $full_name_ar = '';
                                foreach (['full_name_ar', 'name_ar', 'اسم_عربي', 'الاسم العربي', 'اسم', 'الاسم', 'student_name', 'name', 'full_name'] as $key) {
                                    if (isset($row[$key])) {
                                        $v = $cellStr($row[$key]);
                                        if ($v !== '') { $full_name_ar = $v; break; }
                                    }
                                }
                                
                                // تخطي الصفوف الفارغة بالكامل (لا تُحسب كخطأ)
                                if ($student_number === '' && $full_name_ar === '') {
                                    continue;
                                }
                                
                                $national_id_raw = trim($row['national_id'] ?? $row['رقم_وطني'] ?? $row['الرقم الوطني'] ?? '');
                                $national_id = ($national_id_raw !== '') ? $national_id_raw : null;
                                $full_name_en = trim($row['full_name_en'] ?? $row['name_en'] ?? $row['اسم_إنجليزي'] ?? $row['الاسم الإنجليزي'] ?? '');
                                $email = trim($row['email'] ?? $row['بريد'] ?? $row['البريد الإلكتروني'] ?? '');
                                $phone = trim($row['phone'] ?? $row['هاتف'] ?? $row['الهاتف'] ?? '');
                                $major_code = trim($row['major_code'] ?? $row['رمز_تخصص'] ?? $row['رمز التخصص'] ?? '');
                                $major_id_from_file = isset($row['major_id']) ? (int)$row['major_id'] : 0;
                                $enrollment_date_raw = trim($row['enrollment_date'] ?? $row['تاريخ_تسجيل'] ?? $row['تاريخ التسجيل'] ?? date('Y-m-d'));
                                $enrollment_ts = strtotime($enrollment_date_raw);
                                $enrollment_date = ($enrollment_ts !== false) ? date('Y-m-d', $enrollment_ts) : date('Y-m-d');
                                $address = trim($row['address'] ?? $row['العنوان'] ?? '');
                                $graduation_date_raw = trim($row['graduation_date'] ?? $row['تاريخ_التخرج'] ?? $row['تاريخ التخرج'] ?? '');
                                $graduation_date = null;
                                if ($graduation_date_raw !== '') {
                                    $gt = strtotime($graduation_date_raw);
                                    $graduation_date = ($gt !== false) ? date('Y-m-d', $gt) : null;
                                }
                                // قبول "نشط"، "نشط لا زال"، "الكلية active"، "active" وغيرها
                                $status_raw = trim($row['status'] ?? $row['الحالة'] ?? 'active');
                                if (preg_match('/(active|graduated|suspended|withdrawn|نشط|خريج|موقوف|منسحب)/iu', $status_raw, $m)) {
                                    $status_raw = $m[1];
                                }
                                $status_raw = trim($status_raw);
                                $status_raw = mb_strtolower($status_raw, 'UTF-8');
                                if ($status_raw === 'نشط') $status_raw = 'active';
                                if (in_array($status_raw, ['خريج', 'خريجة'])) $status_raw = 'graduated';
                                if (in_array($status_raw, ['موقوف', 'موقوفة'])) $status_raw = 'suspended';
                                if (in_array($status_raw, ['منسحب', 'منسحبة'])) $status_raw = 'withdrawn';
                                $status_allowed = ['active', 'graduated', 'suspended', 'withdrawn'];
                                $status = in_array($status_raw, $status_allowed) ? $status_raw : 'active';
                                
                                // التحقق من الحقول المطلوبة
                                if (empty($student_number) || empty($full_name_ar)) {
                                    $error_count++;
                                    $row_number = $row_index + 2; // +2 لأن الفهرس يبدأ من 0 والسطر الأول هو العناوين
                                    
                                    // إنشاء رسالة خطأ تفصيلية للمساعدة في التصحيح
                                    $error_msg = "السطر $row_number: ";
                                    
                                    if (empty($student_number) && empty($full_name_ar)) {
                                        $error_msg .= "لا يوجد رقم جامعي ولا اسم";
                                    } elseif (empty($student_number)) {
                                        $error_msg .= "لا يوجد رقم جامعي (الاسم: " . htmlspecialchars($full_name_ar) . ")";
                                    } else {
                                        $error_msg .= "لا يوجد اسم (الرقم الجامعي: " . htmlspecialchars($student_number) . ")";
                                    }
                                    
                                    // في وضع التطوير، أضف معلومات إضافية
                                    if (count($errors) < 3) {
                                        $available_keys = array_keys($row);
                                        $error_msg .= " | الحقول الموجودة في السطر: " . implode(', ', array_slice($available_keys, 0, 10));
                                        if (count($available_keys) > 10) {
                                            $error_msg .= "... (و" . (count($available_keys) - 10) . " حقول أخرى)";
                                        }
                                    }
                                    
                                    $errors[] = $error_msg;
                                    continue;
                                }
                                
                                // البحث عن التخصص
                                $selected_major_id = 0;
                                if ($major_id_from_file > 0) {
                                    $selected_major_id = $major_id_from_file;
                                } elseif (!empty($major_code)) {
                                    $major = $db->fetchOne(
                                        "SELECT * FROM majors WHERE code = ? AND college_id = ?",
                                        [$major_code, $default_college_id]
                                    );
                                    if ($major) {
                                        $selected_major_id = $major['id'];
                                    }
                                }
                                
                                // إذا لم يتم العثور على التخصص، استخدام التخصص المحدد من النموذج
                                if ($selected_major_id <= 0) {
                                    $selected_major_id = (int)($_POST['major_id'] ?? 0);
                                }
                                
                                // الكليات التي ليس بها تشعيب (مثل كلية القانون): إذا لم يُحدد تخصص، استخدام التخصص الافتراضي "الكلية"
                                if ($selected_major_id <= 0) {
                                    $college_row = $db->fetchOne("SELECT id, specialization_year FROM colleges WHERE id = ?", [$default_college_id]);
                                    if ($college_row && (empty($college_row['specialization_year']) || (int)$college_row['specialization_year'] <= 0)) {
                                        $default_major = $db->fetchOne("SELECT id FROM majors WHERE college_id = ? ORDER BY id ASC LIMIT 1", [$default_college_id]);
                                        if (!$default_major) {
                                            try {
                                                $has_ig = $db->fetchOne("SHOW COLUMNS FROM majors WHERE Field = 'is_general'");
                                                if (!$has_ig) {
                                                    @$db->getConnection()->exec("ALTER TABLE majors ADD COLUMN is_general TINYINT(1) NOT NULL DEFAULT 0 AFTER total_semesters");
                                                }
                                                $db->query(
                                                    "INSERT INTO majors (college_id, name_ar, name_en, code, total_semesters, is_general) VALUES (?, ?, ?, ?, ?, 0)",
                                                    [$default_college_id, 'الكلية', 'College', 'DEFAULT', 8]
                                                );
                                                $selected_major_id = (int)$db->getConnection()->lastInsertId();
                                            } catch (PDOException $e) {
                                                $default_major = $db->fetchOne("SELECT id FROM majors WHERE college_id = ? ORDER BY id ASC LIMIT 1", [$default_college_id]);
                                                if ($default_major) {
                                                    $selected_major_id = (int)$default_major['id'];
                                                }
                                            }
                                        } else {
                                            $selected_major_id = (int)$default_major['id'];
                                        }
                                    }
                                }
                                
                                if ($selected_major_id <= 0) {
                                    $error_count++;
                                    $errors[] = "الطالب $student_number: لم يتم العثور على التخصص";
                                    continue;
                                }
                                
                                // التحقق من الصلاحيات
                                if (!$auth->belongsToCollege($default_college_id) && !$auth->hasRole('admin')) {
                                    $error_count++;
                                    $errors[] = "الطالب $student_number: ليس لديك صلاحية لإضافة طالب في هذه الكلية";
                                    continue;
                                }
                                
                                // التحقق من عدم وجود طالب بنفس الرقم الجامعي
                                $existing = $db->fetchOne(
                                    "SELECT s.*, c.name_ar as college_name FROM students s LEFT JOIN colleges c ON c.id = s.college_id WHERE s.student_number = ?",
                                    [$student_number]
                                );
                                
                                if ($existing) {
                                    $duplicate_count++;
                                    $duplicates[] = [
                                        'student_number' => $student_number,
                                        'full_name_ar'   => $full_name_ar,
                                        'college_name'   => $existing['college_name'] ?? '—'
                                    ];
                                    continue;
                                }
                                
                                // التحقق من الرقم الوطني إذا كان موجوداً
                                if (!empty($national_id)) {
                                    $existing_nid = $db->fetchOne(
                                        "SELECT s.*, c.name_ar as college_name FROM students s LEFT JOIN colleges c ON c.id = s.college_id WHERE s.national_id = ? AND s.national_id != ''",
                                        [$national_id]
                                    );
                                    if ($existing_nid) {
                                        $duplicate_count++;
                                        $duplicates[] = [
                                            'student_number' => $student_number,
                                            'full_name_ar'   => $full_name_ar,
                                            'college_name'   => $existing_nid['college_name'] ?? '—'
                                        ];
                                        continue;
                                    }
                                }
                                
                                // إنشاء مستخدم للطالب أو استخدام الموجود إن وُجد (تجنب فشل الإضافة بسبب تكرار اسم المستخدم)
                                $username = 'STU' . $student_number;
                                $password = password_hash($student_number, PASSWORD_DEFAULT);
                                
                                try {
                                    $existing_user = $db->fetchOne("SELECT id FROM users WHERE username = ?", [$username]);
                                    if ($existing_user) {
                                        $user_id = (int)$existing_user['id'];
                                    } else {
                                        $db->query(
                                            "INSERT INTO users (college_id, username, password, full_name_ar, full_name_en, email, role) 
                                             VALUES (?, ?, ?, ?, ?, ?, 'student')",
                                            [$default_college_id, $username, $password, $full_name_ar, $full_name_en, $email]
                                        );
                                        $user_id = (int)$db->getConnection()->lastInsertId();
                                    }
                                    
                                    // التأكد من عدم وجود سجل ب id=0 قبل الإدراج
                                    $zero_record = $db->fetchOne("SELECT id FROM students WHERE id = 0");
                                    if ($zero_record) {
                                        $db->query("DELETE FROM students WHERE id = 0");
                                    }
                                    
                                    // التأكد من أن AUTO_INCREMENT يعمل بشكل صحيح
                                    $columns = $db->fetchAll("SHOW COLUMNS FROM students WHERE Field = 'id'");
                                    $has_auto_increment = false;
                                    if (!empty($columns)) {
                                        $has_auto_increment = (strpos($columns[0]['Extra'], 'auto_increment') !== false);
                                    }
                                    
                                    if (!$has_auto_increment) {
                                        // إضافة AUTO_INCREMENT إذا لم يكن موجوداً
                                        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
                                        $max_id = $max_id_result['max_id'] ?? 0;
                                        $new_auto_increment = max($max_id + 1, 1);
                                        $db->getConnection()->exec("ALTER TABLE students MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
                                        $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)$new_auto_increment);
                                    }
                                    
                                    // إضافة الطالب (بدون تحديد id ليتم توليده تلقائياً)
                                    $db->query(
                                        "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, address, enrollment_date, graduation_date, user_id, status) 
                                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                                        [$default_college_id, $selected_major_id, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $address ?: null, $enrollment_date, $graduation_date, $user_id, $status]
                                    );
                                    
                                    // الحصول على ID المولّد تلقائياً
                                    $student_id = $db->getConnection()->lastInsertId();
                                    
                                    // إذا كان ID = 0 أو false، فهناك مشكلة، يجب إصلاحها
                                    if ($student_id == 0 || $student_id === false || $student_id === '0') {
                                        // حذف أي سجل ب id=0
                                        $db->query("DELETE FROM students WHERE id = 0");
                                        
                                        // جلب أعلى ID موجود
                                        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
                                        $max_id = $max_id_result['max_id'] ?? 0;
                                        $new_auto_increment = max($max_id + 1, 1);
                                        
                                        // إصلاح AUTO_INCREMENT
                                        $db->getConnection()->exec("ALTER TABLE students MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
                                        $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)$new_auto_increment);
                                        
                                        // إعادة إدراج السجل
                                        $db->query(
                                            "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, address, enrollment_date, graduation_date, user_id, status) 
                                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                                            [$default_college_id, $selected_major_id, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $address ?: null, $enrollment_date, $graduation_date, $user_id, $status]
                                        );
                                        $student_id = $db->getConnection()->lastInsertId();
                                        
                                        // التحقق مرة أخرى
                                        if ($student_id == 0 || $student_id === false || $student_id === '0') {
                                            throw new Exception("فشل في توليد ID تلقائياً للطالب $student_number");
                                        }
                                    }
                                    
                                    $success_count++;
                                } catch (PDOException $e) {
                                    $error_count++;
                                    $error_msg = $e->getMessage();
                                    if (strpos($error_msg, 'Duplicate entry') !== false || strpos($error_msg, '1062') !== false) {
                                        $duplicate_count++;
                                        $dup_college = $db->fetchOne("SELECT c.name_ar FROM students s LEFT JOIN colleges c ON c.id = s.college_id WHERE s.student_number = ?", [$student_number]);
                                        $duplicates[] = [
                                            'student_number' => $student_number,
                                            'full_name_ar'   => $full_name_ar,
                                            'college_name'   => ($dup_college && isset($dup_college['name_ar'])) ? $dup_college['name_ar'] : '—'
                                        ];
                                    } else {
                                        $errors[] = "الطالب $student_number: " . $error_msg;
                                        error_log("Error adding student from Excel: " . $error_msg);
                                    }
                                } catch (Exception $e) {
                                    $error_count++;
                                    $errors[] = "الطالب $student_number: " . $e->getMessage();
                                    error_log("Error adding student from Excel: " . $e->getMessage());
                                }
                            }
                            
                            // تأكيد المعاملة
                            $db->getConnection()->commit();
                            
                        } catch (Exception $e) {
                            // إلغاء المعاملة في حالة الخطأ
                            $db->getConnection()->rollBack();
                            $message = '<div class="alert alert-error">خطأ في المعاملة: ' . htmlspecialchars($e->getMessage()) . '</div>';
                            error_log("Transaction error: " . $e->getMessage());
                        }
                        
                        // بناء رسالة النتيجة
                        if ($success_count > 0) {
                            $success_label = ($success_count === 1) ? 'طالب واحد' : $success_count . ' طلاب';
                            $message = '<div class="alert alert-success">✓ تم إضافة ' . $success_label . ' بنجاح من الملف</div>';
                            $message .= '<p style="margin:6px 0 0 0; font-size:13px; color:#555;">عدد الصفوف في الملف: <strong>' . $rows_with_data . '</strong>، أرقام جامعية مختلفة: <strong>' . $unique_in_file . '</strong></p>';
                        }
                        if ($success_count === 0 && $duplicate_count === 0 && $error_count === 0 && !empty($students_data)) {
                            $message = '<div class="alert alert-warning">لم يتم إضافة أي طالب. تحقق من: (1) اختيار الكلية والتخصص في النموذج، (2) أن الملف يحتوي على أعمدة الرقم الجامعي والاسم العربي، (3) أن الصفوف غير فارغة.</div>';
                        }
                        if ($duplicate_count > 0) {
                            $dup_label = ($duplicate_count === 1) ? 'طالب واحد موجود مسبقاً' : $duplicate_count . ' طلاب موجودون مسبقاً';
                            $message .= '<div class="alert alert-info">⚠ ' . $dup_label . ' (تم تخطيهم)';
                            if ($unique_in_file === 1 && $rows_with_data > 1) {
                                $message .= '<p style="margin:8px 0 0 0;"><strong>سبب ظهور طالب واحد فقط:</strong> الملف الذي رفعته يحتوي على <strong>نفس الرقم الجامعي في كل الصفوف</strong> (' . $rows_with_data . ' صف). النظام أضاف الطالب مرة واحدة وتخطى الباقي. لإضافة عدة طلاب: تأكد أن عمود الرقم الجامعي يحتوي على قيمة <strong>مختلفة في كل صف</strong>. إذا حفظت الملف من Excel، قد يُحوّل العمود إلى رقم ويُكرر القيمة — استخدم ملف CSV بصيغة UTF-8 أو أنشئ القالب من صفحة <a href="paste_students_list.php">لصق القائمة وتوليد القالب</a>.</p>';
                            } else {
                                $message .= '<p style="margin:8px 0 0 0;">الطلاب المتخطون مسجلون مسبقاً في جدول الطلاب (التكرار يُحسب حسب <strong>الرقم الجامعي</strong> في كل النظام). إذا ظهر طالب واحد فقط في القائمة الحالية، فالـ' . $duplicate_count . ' متخطون مسجلون في <strong>كلية أخرى</strong>. لعرضهم: اضغط «إلغاء التصفية» أو اختر الكلية الأخرى من فلتر «تصفية القائمة» أعلى الجدول.';
                            }
                            if ($default_college_id > 0) {
                                $message .= ' <a href="students.php?college_id=' . (int)$default_college_id . '" class="btn btn-secondary" style="display:inline-block; margin-top:8px; padding:6px 12px;">عرض طلاب هذه الكلية</a>';
                            }
                            $message .= ' <a href="students.php" class="btn btn-secondary" style="display:inline-block; margin-top:8px; padding:6px 12px;">عرض الكل (إلغاء التصفية)</a>';
                            $message .= '</p>';
                            if (!empty($duplicates)) {
                                $max_show_dup = 50;
                                $dup_to_show = array_slice($duplicates, 0, $max_show_dup);
                                $message .= '<br><strong>الطلاب المتخطون (ومكان تسجيلهم):</strong>';
                                $message .= '<div style="margin-top:8px; max-height:280px; overflow-y:auto; background:rgba(255,255,255,0.6); padding:10px; border-radius:6px; font-size:14px;">';
                                $message .= '<table style="width:100%; border-collapse:collapse;"><thead><tr style="border-bottom:1px solid #ddd;"><th style="text-align:right; padding:6px;">الرقم الجامعي</th><th style="text-align:right; padding:6px;">الاسم</th><th style="text-align:right; padding:6px;">مسجل في الكلية</th></tr></thead><tbody>';
                                foreach ($dup_to_show as $d) {
                                    $message .= '<tr style="border-bottom:1px solid #eee;"><td style="padding:6px;">' . htmlspecialchars($d['student_number']) . '</td><td style="padding:6px;">' . htmlspecialchars($d['full_name_ar']) . '</td><td style="padding:6px;">' . htmlspecialchars($d['college_name'] ?? '—') . '</td></tr>';
                                }
                                $message .= '</tbody></table>';
                                if (count($duplicates) > $max_show_dup) {
                                    $message .= '<p style="margin:8px 0 0 0; color:#666;"><em>... و ' . (count($duplicates) - $max_show_dup) . ' طالب آخر. إجمالي المتخطين: ' . count($duplicates) . '</em></p>';
                                }
                                $message .= '</div>';
                            }
                            $message .= '</div>';
                        }
                        if ($error_count > 0) {
                            $message .= '<div class="alert alert-warning">⚠ ' . $error_count . ' خطأ في الإضافة</div>';
                            if (count($errors) > 0) {
                                $max_show = 25;
                                $errors_to_show = array_slice($errors, 0, $max_show);
                                $message .= '<div class="alert alert-info" style="margin-top:10px; max-height:320px; overflow-y:auto;"><strong>تفاصيل الأخطاء:</strong><br><small>' . implode('<br>', array_map('htmlspecialchars', $errors_to_show)) . '</small>';
                                if (count($errors) > $max_show) {
                                    $message .= '<br><br><em>... و ' . (count($errors) - $max_show) . ' خطأ آخر. راجع تنسيق الملف (رقم جامعي، الاسم العربي، التخصص) ودليل رفع الطلاب.</em>';
                                }
                                $message .= '</div>';
                            }
                        }
                    }
                }
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '<br>يرجى استخدام ملف CSV أو تثبيت مكتبة PhpSpreadsheet لقراءة ملفات Excel</div>';
            }
        }
    } elseif (isset($_FILES['excel_file'])) {
        $error_msg = 'خطأ غير معروف';
        switch ($_FILES['excel_file']['error']) {
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $error_msg = 'حجم الملف كبير جداً';
                break;
            case UPLOAD_ERR_PARTIAL:
                $error_msg = 'تم رفع جزء من الملف فقط';
                break;
            case UPLOAD_ERR_NO_FILE:
                $error_msg = 'لم يتم اختيار ملف';
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $error_msg = 'مجلد الملفات المؤقتة غير موجود';
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $error_msg = 'فشل كتابة الملف على القرص';
                break;
        }
        $message = '<div class="alert alert-error">' . $error_msg . '</div>';
    } else {
        $message = '<div class="alert alert-error">خطأ: لم يتم رفع الملف</div>';
    }
}

// معالجة الإضافة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $student_number = trim($_POST['student_number'] ?? '');
    $national_id_raw = trim($_POST['national_id'] ?? '');
    $national_id = ($national_id_raw !== '') ? $national_id_raw : null;
    $full_name_ar = trim($_POST['full_name_ar'] ?? '');
    $full_name_en = trim($_POST['full_name_en'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $college_id = (int)($_POST['college_id'] ?? $_SESSION['college_id']);
    $major_id_post = (int)($_POST['major_id'] ?? 0);
    $enrollment_date_raw = $_POST['enrollment_date'] ?? date('Y-m-d');
    $enrollment_ts = strtotime($enrollment_date_raw);
    $enrollment_date = ($enrollment_ts !== false) ? date('Y-m-d', $enrollment_ts) : date('Y-m-d');
    
    if ($student_number && $full_name_ar && $major_id_post) {
        // التحقق من الصلاحية
        if (!$auth->belongsToCollege($college_id) && !$auth->hasRole('admin')) {
            $message = '<div class="alert alert-error">ليس لديك صلاحية لإضافة طالب في هذه الكلية</div>';
        } else {
            try {
                // التحقق من عدم تكرار الرقم الجامعي
                $existing = $db->fetchOne("SELECT id FROM students WHERE student_number = ?", [$student_number]);
                if ($existing) {
                    $message = '<div class="alert alert-error">الرقم الجامعي موجود مسبقاً. لا يمكن إضافة طالب مكرر.</div>';
                } elseif ($national_id !== null) {
                    $existing_nid = $db->fetchOne("SELECT id FROM students WHERE national_id = ? AND national_id IS NOT NULL AND national_id != ''", [$national_id]);
                    if ($existing_nid) {
                        $message = '<div class="alert alert-error">الرقم الوطني موجود مسبقاً لطالب آخر.</div>';
                    } else {
                        $message = addOneStudent($db, $college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date);
                    }
                } else {
                    $message = addOneStudent($db, $college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date);
                }
            } catch (PDOException $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

/**
 * إضافة طالب واحد (من النموذج) — التحقق من المستخدم وإصلاح AUTO_INCREMENT ثم INSERT
 * @return string رسالة HTML للمستخدم
 */
function addOneStudent($db, $college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date) {
    // التأكد من عدم وجود سجل ب id=0 قبل الإدراج
    $zero_record = $db->fetchOne("SELECT id FROM students WHERE id = 0");
    if ($zero_record) {
        $db->query("DELETE FROM students WHERE id = 0");
    }
    
    $columns = $db->fetchAll("SHOW COLUMNS FROM students WHERE Field = 'id'");
    $has_auto_increment = false;
    if (!empty($columns)) {
        $has_auto_increment = (strpos($columns[0]['Extra'], 'auto_increment') !== false);
    }
    if (!$has_auto_increment) {
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        $new_auto_increment = max($max_id + 1, 1);
        $db->getConnection()->exec("ALTER TABLE students MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
        $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)$new_auto_increment);
    }
    
    $username = 'STU' . $student_number;
    $password = password_hash($student_number, PASSWORD_DEFAULT);
    $existing_user = $db->fetchOne("SELECT id FROM users WHERE username = ?", [$username]);
    if ($existing_user) {
        $user_id = (int)$existing_user['id'];
    } else {
        $db->query(
            "INSERT INTO users (college_id, username, password, full_name_ar, full_name_en, email, role) 
             VALUES (?, ?, ?, ?, ?, ?, 'student')",
            [$college_id, $username, $password, $full_name_ar, $full_name_en, $email]
        );
        $user_id = (int)$db->getConnection()->lastInsertId();
    }
    
    $db->query(
        "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, address, enrollment_date, graduation_date, user_id, status) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, NULL, ?, NULL, ?, 'active')",
        [$college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date, $user_id]
    );
    $student_id = $db->getConnection()->lastInsertId();
    
    if ($student_id == 0 || $student_id === false || $student_id === '0') {
        $db->query("DELETE FROM students WHERE id = 0");
        $max_id_result = $db->fetchOne("SELECT MAX(id) as max_id FROM students WHERE id > 0");
        $max_id = $max_id_result['max_id'] ?? 0;
        $db->getConnection()->exec("ALTER TABLE students MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
        $db->getConnection()->exec("ALTER TABLE students AUTO_INCREMENT = " . (int)max($max_id + 1, 1));
        $db->query(
            "INSERT INTO students (college_id, major_id, student_number, national_id, full_name_ar, full_name_en, email, phone, address, enrollment_date, graduation_date, user_id, status) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, NULL, ?, NULL, ?, 'active')",
            [$college_id, $major_id_post, $student_number, $national_id, $full_name_ar, $full_name_en, $email, $phone, $enrollment_date, $user_id]
        );
        $student_id = $db->getConnection()->lastInsertId();
        if ($student_id == 0 || $student_id === false || $student_id === '0') {
            throw new Exception("فشل في توليد ID تلقائياً. يرجى التحقق من إعدادات قاعدة البيانات.");
        }
    }
    
    require_once __DIR__ . '/classes/ActivityLogger.php';
    $logger = new ActivityLogger();
    $logger->log('create', 'student', $student_id, 'إضافة طالب جديد: ' . $full_name_ar . ' (الرقم الجامعي: ' . $student_number . ')');
    
    return '<div class="alert alert-success">تم إضافة الطالب بنجاح</div>';
}

// معالجة الحذف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $student_id = (int)($_POST['student_id'] ?? 0);
    
    if ($student_id > 0) {
        try {
            // جلب معلومات الطالب للتحقق من الصلاحيات
            $student = $db->fetchOne(
                "SELECT s.* 
                 FROM students s 
                 WHERE s.id = ?",
                [$student_id]
            );
            
            if (!$student) {
                $message = '<div class="alert alert-error">الطالب غير موجود</div>';
            } else {
                // التحقق من الصلاحية
                if (!$auth->belongsToCollege($student['college_id']) && !$auth->hasRole('admin')) {
                    $message = '<div class="alert alert-error">ليس لديك صلاحية لحذف طالب من هذه الكلية</div>';
                } else {
                    // بدء معاملة قاعدة البيانات
                    $db->getConnection()->beginTransaction();
                    
                    try {
                        // حذف المستخدم المرتبط إذا كان موجوداً
                        if ($student['user_id']) {
                            $db->query("DELETE FROM users WHERE id = ?", [$student['user_id']]);
                        }
                        
                        // حذف الطالب (سيتم حذف البيانات المرتبطة تلقائياً بسبب CASCADE)
                        $db->query("DELETE FROM students WHERE id = ?", [$student_id]);
                        
                        // تسجيل نشاط حذف الطالب
                        require_once __DIR__ . '/classes/ActivityLogger.php';
                        $logger = new ActivityLogger();
                        $logger->log('delete', 'student', $student_id, 'حذف طالب: ' . $student['full_name_ar'] . ' (الرقم الجامعي: ' . $student['student_number'] . ')');
                        
                        // تأكيد المعاملة
                        $db->getConnection()->commit();
                        
                        $message = '<div class="alert alert-success">تم حذف الطالب بنجاح</div>';
                    } catch (Exception $e) {
                        // إلغاء المعاملة في حالة الخطأ
                        $db->getConnection()->rollBack();
                        throw $e;
                    }
                }
            }
        } catch (PDOException $e) {
            $message = '<div class="alert alert-error">خطأ في حذف الطالب: ' . htmlspecialchars($e->getMessage()) . '</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    } else {
        $message = '<div class="alert alert-error">معرف الطالب غير صحيح</div>';
    }
}

// معالجة حذف جميع الطلاب (فقط للـ admin)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_all') {
    // التحقق من الصلاحيات - فقط admin يمكنه حذف جميع الطلاب
    if (!$auth->hasRole('admin')) {
        $message = '<div class="alert alert-error">⚠️ خطأ: ليس لديك صلاحية لحذف جميع الطلاب. هذه العملية متاحة فقط للمدير العام.</div>';
    } else {
        // التحقق من تأكيد الحذف
        $confirm_text = trim($_POST['confirm_text'] ?? '');
        $expected_text = 'حذف جميع الطلاب';
        
        if ($confirm_text !== $expected_text) {
            $message = '<div class="alert alert-error">⚠️ خطأ: لم يتم تأكيد العملية. يجب كتابة "حذف جميع الطلاب" للتأكيد.</div>';
        } else {
            try {
                // جلب عدد الطلاب قبل الحذف
                $total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
                
                if ($total_students == 0) {
                    $message = '<div class="alert alert-info">لا يوجد طلاب لحذفهم</div>';
                } else {
                    // بدء معاملة قاعدة البيانات
                    $db->getConnection()->beginTransaction();
                    
                    try {
                        // جلب جميع معرفات المستخدمين المرتبطين بالطلاب
                        $students_with_users = $db->fetchAll(
                            "SELECT DISTINCT user_id FROM students WHERE user_id IS NOT NULL"
                        );
                        
                        // حذف جميع المستخدمين المرتبطين بالطلاب
                        if (!empty($students_with_users)) {
                            $user_ids = array_column($students_with_users, 'user_id');
                            $placeholders = implode(',', array_fill(0, count($user_ids), '?'));
                            $db->query("DELETE FROM users WHERE id IN ($placeholders)", $user_ids);
                        }
                        
                        // حذف جميع الطلاب (سيتم حذف البيانات المرتبطة تلقائياً بسبب CASCADE)
                        // البيانات المرتبطة تشمل: student_courses, grades, semester_gpas, cumulative_gpas, etc.
                        $db->query("DELETE FROM students");
                        
                        // تسجيل نشاط حذف جميع الطلاب
                        require_once __DIR__ . '/classes/ActivityLogger.php';
                        $logger = new ActivityLogger();
                        $logger->log('delete', 'student', null, 'حذف جميع الطلاب (' . $total_students . ' طالب) - تم بواسطة: ' . ($_SESSION['username'] ?? 'غير معروف'));
                        
                        // تأكيد المعاملة
                        $db->getConnection()->commit();
                        
                        $message = '<div class="alert alert-success">✓ تم حذف جميع الطلاب بنجاح (' . $total_students . ' طالب)</div>';
                        $message .= '<div class="alert alert-info">⚠️ تم حذف جميع البيانات المرتبطة (التسجيلات، الدرجات، المعدلات، إلخ)</div>';
                    } catch (Exception $e) {
                        // إلغاء المعاملة في حالة الخطأ
                        $db->getConnection()->rollBack();
                        throw $e;
                    }
                }
            } catch (PDOException $e) {
                $message = '<div class="alert alert-error">خطأ في حذف جميع الطلاب: ' . htmlspecialchars($e->getMessage()) . '</div>';
            } catch (Exception $e) {
                $message = '<div class="alert alert-error">خطأ: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
}

// جلب الطلاب مع معالجة الأخطاء
try {
    if ($auth->hasRole('admin')) {
        $admin_where = [];
        $admin_params = [];
        if ($filter_college_id > 0) {
            $admin_where[] = 's.college_id = ?';
            $admin_params[] = $filter_college_id;
        }
        if ($major_id > 0) {
            $admin_where[] = 's.major_id = ?';
            $admin_params[] = $major_id;
        }
        $admin_where_sql = empty($admin_where) ? '' : ' WHERE ' . implode(' AND ', $admin_where);
        // عند عدم وجود فلتر: استعلام بسيط + LEFT JOIN لضمان ظهور كل الطلاب (حتى من لهم college_id/major_id غير مطابق)
        if (empty($admin_where)) {
            $students = $db->fetchAll(
                "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                        (SELECT sem.semester_number FROM semesters sem 
                         JOIN student_courses sc ON sem.id = sc.semester_id 
                         WHERE sc.student_id = s.id 
                         ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                         LIMIT 1) as semester_number,
                        (SELECT sem.academic_year FROM semesters sem 
                         JOIN student_courses sc ON sem.id = sc.semester_id 
                         WHERE sc.student_id = s.id 
                         ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                         LIMIT 1) as academic_year
                 FROM students s 
                 LEFT JOIN majors m ON s.major_id = m.id 
                 LEFT JOIN colleges c ON s.college_id = c.id 
                 ORDER BY s.full_name_ar"
            );
        } else {
            $students = $db->fetchAll(
                "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 LEFT JOIN majors m ON s.major_id = m.id 
                 LEFT JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 $admin_where_sql
                 ORDER BY s.full_name_ar",
                $admin_params
            );
        }
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        if ($major_id) {
            $students = $db->fetchAll(
                "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 LEFT JOIN majors m ON s.major_id = m.id 
                 LEFT JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 WHERE s.college_id = ? AND s.major_id = ?
                 ORDER BY s.full_name_ar",
                [$college_id, $major_id]
            );
        } else {
            $students = $db->fetchAll(
                "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                        sem.semester_number, sem.academic_year
                 FROM students s 
                 LEFT JOIN majors m ON s.major_id = m.id 
                 LEFT JOIN colleges c ON s.college_id = c.id 
                 LEFT JOIN (
                     SELECT sc.student_id, s.semester_number, s.academic_year,
                            ROW_NUMBER() OVER (PARTITION BY sc.student_id ORDER BY s.academic_year DESC, s.semester_number DESC) as rn
                     FROM student_courses sc
                     JOIN semesters s ON sc.semester_id = s.id
                 ) sem ON sem.student_id = s.id AND sem.rn = 1
                 WHERE s.college_id = ?
                 ORDER BY s.full_name_ar",
                [$college_id]
            );
        }
    }
} catch (Exception $e) {
    // في حالة عدم دعم ROW_NUMBER أو أي خطأ، استخدام استعلام بديل
    try {
        if ($auth->hasRole('admin')) {
            $admin_where = [];
            $admin_params = [];
            if ($filter_college_id > 0) {
                $admin_where[] = 's.college_id = ?';
                $admin_params[] = $filter_college_id;
            }
            if ($major_id > 0) {
                $admin_where[] = 's.major_id = ?';
                $admin_params[] = $major_id;
            }
            $admin_where_sql = empty($admin_where) ? '' : ' WHERE ' . implode(' AND ', $admin_where);
            $students = $db->fetchAll(
                "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                        (SELECT sem.semester_number FROM semesters sem 
                         JOIN student_courses sc ON sem.id = sc.semester_id 
                         WHERE sc.student_id = s.id 
                         ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                         LIMIT 1) as semester_number,
                        (SELECT sem.academic_year FROM semesters sem 
                         JOIN student_courses sc ON sem.id = sc.semester_id 
                         WHERE sc.student_id = s.id 
                         ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                         LIMIT 1) as academic_year
                 FROM students s 
                 LEFT JOIN majors m ON s.major_id = m.id 
                 LEFT JOIN colleges c ON s.college_id = c.id 
                 $admin_where_sql
                 ORDER BY s.full_name_ar",
                $admin_params
            );
        } else {
            $college_id = $_SESSION['college_id'] ?? 0;
            if ($major_id) {
                $students = $db->fetchAll(
                    "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     LEFT JOIN majors m ON s.major_id = m.id 
                     LEFT JOIN colleges c ON s.college_id = c.id 
                     WHERE s.college_id = ? AND s.major_id = ?
                     ORDER BY s.full_name_ar",
                    [$college_id, $major_id]
                );
            } else {
                $students = $db->fetchAll(
                    "SELECT s.*,
                        COALESCE(m.name_ar, '-') as major_name,
                        COALESCE(c.name_ar, '-') as college_name,
                            (SELECT sem.semester_number FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as semester_number,
                            (SELECT sem.academic_year FROM semesters sem 
                             JOIN student_courses sc ON sem.id = sc.semester_id 
                             WHERE sc.student_id = s.id 
                             ORDER BY sem.academic_year DESC, sem.semester_number DESC 
                             LIMIT 1) as academic_year
                     FROM students s 
                     LEFT JOIN majors m ON s.major_id = m.id 
                     LEFT JOIN colleges c ON s.college_id = c.id 
                     WHERE s.college_id = ?
                     ORDER BY s.full_name_ar",
                    [$college_id]
                );
            }
        }
    } catch (Exception $e2) {
        $students = [];
        $message = '<div class="alert alert-error">خطأ في جلب بيانات الطلاب: ' . htmlspecialchars($e2->getMessage()) . '</div>';
    }
}

// جلب التخصصات - للـ admin نبدأ بقائمة فارغة، سيتم تحميلها حسب الكلية المختارة
try {
    if ($auth->hasRole('admin')) {
        // للـ admin، نبدأ بقائمة فارغة - سيتم تحميل التخصصات ديناميكياً حسب الكلية المختارة
        $majors = [];
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        $majors = $db->fetchAll(
            "SELECT m.* FROM majors m 
             JOIN colleges c ON m.college_id = c.id 
             WHERE c.id = ?
             ORDER BY m.name_ar",
            [$college_id]
        );
    }
} catch (Exception $e) {
    $majors = [];
    $message .= '<div class="alert alert-error">خطأ في جلب التخصصات: ' . htmlspecialchars($e->getMessage()) . '</div>';
}

// جلب الكليات
try {
    if ($auth->hasRole('admin')) {
        $colleges = $db->fetchAll("SELECT * FROM colleges ORDER BY name_ar");
    } else {
        $college_id = $_SESSION['college_id'] ?? 0;
        $college_name = $_SESSION['college_name'] ?? 'غير محدد';
        $colleges = [['id' => $college_id, 'name_ar' => $college_name]];
    }
} catch (Exception $e) {
    $colleges = [];
    $message .= '<div class="alert alert-error">خطأ في جلب الكليات: ' . htmlspecialchars($e->getMessage()) . '</div>';
}

// تخصصات الفلتر (للأدمن عند اختيار كلية)
$filter_majors = [];
if ($auth->hasRole('admin') && $filter_college_id > 0) {
    try {
        $filter_majors = $db->fetchAll("SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar", [$filter_college_id]);
    } catch (Exception $e) {
        $filter_majors = [];
    }
}

// إجمالي الطلاب في قاعدة البيانات (للأدمن — للتشخيص)
$total_students_in_db = null;
if ($auth->hasRole('admin')) {
    try {
        $total_students_in_db = (int) $db->fetchOne("SELECT COUNT(*) as cnt FROM students")['cnt'];
    } catch (Exception $e) {
        $total_students_in_db = null;
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة الطلاب</title>
    <?php include __DIR__ . '/includes/compatibility.php'; ?>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>إدارة الطلاب</h2>
            <div>
                <a href="add_students_bulk.php" class="btn btn-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; margin-left: 10px;">➕ إضافة طلاب دفعة واحدة</a>
                <button onclick="document.getElementById('addForm').style.display='block'" class="btn btn-primary">إضافة طالب</button>
                <a href="paste_students_list.php" class="btn btn-success" style="margin-left:8px;">📋 لصق القائمة وتوليد القالب (موصى به — يتجنب مشاكل الترميز)</a>
                <button onclick="document.getElementById('uploadForm').style.display='block'" class="btn btn-success">رفع طلاب من Excel/CSV</button>
                <?php if ($auth->hasRole('admin')): ?>
                    <button onclick="document.getElementById('deleteAllForm').style.display='block'" class="btn btn-danger">🗑️ حذف جميع الطلاب</button>
                <?php endif; ?>
            </div>
        </div>
        
        <?php echo $message; ?>
        
        <!-- فلتر الكلية والتخصص -->
        <div class="card" style="margin-bottom: 20px;">
            <div class="card-header">تصفية القائمة</div>
            <div style="padding: 15px;">
                <form method="get" action="students.php" id="filterForm" style="display: flex; flex-wrap: wrap; align-items: center; gap: 15px;">
                    <?php if ($auth->hasRole('admin')): ?>
                    <div class="form-group" style="margin: 0;">
                        <label for="filter_college_id">الكلية:</label>
                        <select name="college_id" id="filter_college_id" onchange="document.getElementById('filterForm').submit();">
                            <option value="">-- الكل --</option>
                            <?php foreach ($colleges as $c): ?>
                                <option value="<?php echo (int)$c['id']; ?>"<?php echo ($filter_college_id === (int)$c['id']) ? ' selected' : ''; ?>><?php echo htmlspecialchars($c['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group" style="margin: 0;">
                        <label for="filter_major_id">التخصص:</label>
                        <select name="major_id" id="filter_major_id" onchange="document.getElementById('filterForm').submit();">
                            <option value="">-- الكل --</option>
                            <?php foreach ($filter_majors as $fm): ?>
                                <option value="<?php echo (int)$fm['id']; ?>"<?php echo ($major_id === (int)$fm['id']) ? ' selected' : ''; ?>><?php echo htmlspecialchars($fm['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php else: ?>
                    <div class="form-group" style="margin: 0;">
                        <label for="filter_major_id">التخصص:</label>
                        <select name="major_id" id="filter_major_id" onchange="document.getElementById('filterForm').submit();">
                            <option value="">-- الكل --</option>
                            <?php foreach ($majors as $fm): ?>
                                <option value="<?php echo (int)$fm['id']; ?>"<?php echo ($major_id === (int)$fm['id']) ? ' selected' : ''; ?>><?php echo htmlspecialchars($fm['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    <button type="submit" class="btn btn-primary">تطبيق</button>
                    <?php if ($filter_college_id || $major_id): ?>
                    <a href="students.php" class="btn btn-secondary">إلغاء التصفية</a>
                    <?php endif; ?>
                </form>
            </div>
        </div>
        
        <!-- قسم رفع ملف Excel/CSV -->
        <div id="uploadForm" class="card" style="display:none; margin-bottom: 20px; border: 2px solid #4CAF50;">
            <div class="card-header" style="background-color: #4CAF50; color: white;">
                <h3>📥 رفع طلاب من ملف Excel/CSV</h3>
                <p style="margin:8px 0 0 0; font-size:14px; opacity:0.95;">لتجنب مشكلة «سجل واحد» و«الأحرف العربية»، استخدم <a href="paste_students_list.php" style="color:#fff; text-decoration:underline;">لصق القائمة وتوليد القالب</a> ثم ارفع الملف المُحمَّل من هنا.</p>
            </div>
            <div style="padding: 20px;">
                <form method="POST" enctype="multipart/form-data" style="margin-bottom: 15px;">
                    <input type="hidden" name="action" value="upload_excel">
                    
                    <?php if ($auth->hasRole('admin')): ?>
                    <div class="form-group">
                        <label>الكلية:</label>
                        <select name="college_id" id="upload_college_id" required onchange="loadMajorsForUpload()">
                            <option value="">-- اختر الكلية --</option>
                            <?php foreach ($colleges as $college): ?>
                                <option value="<?php echo $college['id']; ?>"><?php echo htmlspecialchars($college['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php else: ?>
                        <input type="hidden" name="college_id" id="upload_college_id" value="<?php echo $_SESSION['college_id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-group">
                        <label>التخصص (اختياري - سيتم استخدامه إذا لم يكن موجوداً في الملف):</label>
                        <select name="major_id" id="upload_major_id" <?php echo $auth->hasRole('admin') ? 'disabled' : ''; ?>>
                            <option value=""><?php echo $auth->hasRole('admin') ? '-- اختر الكلية أولاً --' : '-- اختر التخصص (اختياري) --'; ?></option>
                            <?php foreach ($majors as $major): ?>
                                <option value="<?php echo $major['id']; ?>"<?php echo (!$auth->hasRole('admin') && count($majors) === 1) ? ' selected' : ''; ?>><?php echo htmlspecialchars($major['name_ar']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="excel_file" style="display: block; margin-bottom: 8px; font-weight: bold;">
                            اختر ملف Excel أو CSV:
                        </label>
                        <input type="file" 
                               name="excel_file" 
                               id="excel_file" 
                               accept=".csv,.xlsx,.xls"
                               required
                               style="padding: 8px; width: 100%; max-width: 400px; border: 1px solid #ddd; border-radius: 4px;">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            الصيغ المدعومة: CSV, Excel (.xlsx, .xls) - الحد الأقصى: 10MB
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-success" style="padding: 10px 20px;">
                        📤 رفع الملف وإضافة الطلاب
                    </button>
                    <button type="button" onclick="document.getElementById('uploadForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
                </form>
                
                <div style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 15px;">
                    <h4 style="margin-top: 0;">📋 تنسيق الملف المطلوب:</h4>
                    <p><strong>الحقول المطلوبة:</strong></p>
                    <ul style="margin-right: 20px;">
                        <li><strong>student_number</strong> أو <strong>الرقم الجامعي</strong> - الرقم الجامعي (مطلوب)</li>
                        <li><strong>full_name_ar</strong> أو <strong>الاسم العربي</strong> - الاسم الكامل بالعربية (مطلوب)</li>
                        <li><strong>full_name_en</strong> أو <strong>الاسم الإنجليزي</strong> - الاسم الكامل بالإنجليزية (اختياري)</li>
                        <li><strong>national_id</strong> أو <strong>الرقم الوطني</strong> - الرقم الوطني (اختياري)</li>
                        <li><strong>email</strong> أو <strong>البريد الإلكتروني</strong> - البريد الإلكتروني (اختياري)</li>
                        <li><strong>phone</strong> أو <strong>الهاتف</strong> - رقم الهاتف (اختياري)</li>
                        <li><strong>address</strong> أو <strong>العنوان</strong> - العنوان (اختياري)</li>
                        <li><strong>enrollment_date</strong> أو <strong>تاريخ التسجيل</strong> - تاريخ التسجيل (صيغة: YYYY-MM-DD، افتراضي: اليوم)</li>
                        <li><strong>graduation_date</strong> أو <strong>تاريخ التخرج</strong> - تاريخ التخرج (اختياري، YYYY-MM-DD)</li>
                        <li><strong>status</strong> أو <strong>الحالة</strong> - الحالة: active، graduated، suspended، withdrawn (افتراضي: active)</li>
                        <li><strong>major_code</strong> أو <strong>رمز التخصص</strong> - رمز التخصص (اختياري)</li>
                        <li><strong>major_id</strong> - معرف التخصص (اختياري)</li>
                    </ul>
                    <p style="margin-top: 10px;"><strong>تحميل قالب CSV (يطابق جدول الطلاب):</strong></p>
                    <p style="margin-top: 8px;">
                        <a href="download_templates.php?type=students" class="btn btn-secondary" style="padding: 8px 15px; margin-left: 10px;">
                            📥 قالب إنجليزي (template_students.csv)
                        </a>
                        <a href="download_templates.php?type=students_ar" class="btn btn-secondary" style="padding: 8px 15px; margin-left: 10px;">
                            📥 قالب بعناوين عربية (قالب_استيراد_الطلاب.csv)
                        </a>
                        <a href="download_templates.php?type=students_law" class="btn btn-secondary" style="padding: 8px 15px; margin-left: 10px;">
                            📥 قالب جاهز كلية القانون (UTF-8)
                        </a>
                        <a href="sample_students_import.csv" download class="btn btn-secondary" style="padding: 8px 15px; margin-left: 10px;">
                            📥 ملف نموذجي قديم (CSV)
                        </a>
                    </p>
                    <p style="margin-top: 10px; color: #666; font-size: 14px;">
                        <strong>ملاحظات:</strong>
                        <ul style="margin-right: 20px; margin-top: 5px;">
                            <li>يمكن استخدام أسماء الأعمدة بالعربية أو الإنجليزية</li>
                            <li>الحقول المطلوبة فقط: <code>student_number</code> و <code>full_name_ar</code></li>
                            <li>إذا لم يتم تحديد التخصص في الملف، سيتم استخدام التخصص المحدد في النموذج</li>
                            <li>الطلاب المكررون (بنفس الرقم الجامعي) سيتم تخطيهم تلقائياً</li>
                        </ul>
                    </p>
                </div>
            </div>
        </div>
        
        <div id="addForm" class="card" style="display:none; margin-bottom: 20px;">
            <div class="card-header">
                <h3>إضافة طالب جديد</h3>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="add">
                <div class="form-group">
                    <label>الرقم الجامعي</label>
                    <input type="text" name="student_number" required>
                </div>
                <div class="form-group">
                    <label>الرقم الوطني</label>
                    <input type="text" name="national_id">
                </div>
                <div class="form-group">
                    <label>الاسم الكامل (عربي)</label>
                    <input type="text" name="full_name_ar" required>
                </div>
                <div class="form-group">
                    <label>الاسم الكامل (إنجليزي)</label>
                    <input type="text" name="full_name_en">
                </div>
                <div class="form-group">
                    <label>البريد الإلكتروني</label>
                    <input type="email" name="email">
                </div>
                <div class="form-group">
                    <label>الهاتف</label>
                    <input type="text" name="phone">
                </div>
                <?php if ($auth->hasRole('admin')): ?>
                <div class="form-group">
                    <label>الكلية</label>
                    <select name="college_id" id="add_college_id" required onchange="loadMajorsForAdd()">
                        <option value="">-- اختر الكلية --</option>
                        <?php foreach ($colleges as $college): ?>
                            <option value="<?php echo $college['id']; ?>"><?php echo htmlspecialchars($college['name_ar']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php else: ?>
                    <input type="hidden" name="college_id" id="add_college_id" value="<?php echo $_SESSION['college_id']; ?>">
                <?php endif; ?>
                <div class="form-group">
                    <label>التخصص</label>
                    <select name="major_id" id="add_major_id" required disabled>
                        <option value=""><?php echo $auth->hasRole('admin') ? 'اختر الكلية أولاً' : 'اختر التخصص'; ?></option>
                        <?php foreach ($majors as $major): ?>
                            <option value="<?php echo $major['id']; ?>"><?php echo htmlspecialchars($major['name_ar']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>تاريخ التسجيل</label>
                    <input type="date" name="enrollment_date" value="<?php echo date('Y-m-d'); ?>" required>
                </div>
                <button type="submit" class="btn btn-success">إضافة</button>
                <button type="button" onclick="document.getElementById('addForm').style.display='none'" class="btn btn-secondary">إلغاء</button>
            </form>
        </div>
        
        <!-- قسم حذف جميع الطلاب -->
        <?php if ($auth->hasRole('admin')): ?>
        <div id="deleteAllForm" class="card" style="display:none; margin-bottom: 20px; border: 3px solid #f44336;">
            <div class="card-header" style="background-color: #f44336; color: white;">
                <h3>⚠️ حذف جميع الطلاب</h3>
            </div>
            <div style="padding: 20px;">
        <?php
        // جلب عدد الطلاب للعرض في النموذج
        try {
            $total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
        } catch (Exception $e) {
            $total_students = 0;
        }
        
        // تعريف المتغير للاستخدام في JavaScript
        $js_total_students = $total_students;
        ?>
                <div style="background: #fff3cd; padding: 15px; border-radius: 5px; margin-bottom: 20px; border: 2px solid #ffc107;">
                    <h4 style="color: #856404; margin-top: 0;">⚠️ تحذير خطير!</h4>
                    <p style="color: #856404; font-weight: bold; margin-bottom: 10px;">
                        هذه العملية ستحذف <strong>جميع الطلاب</strong> من النظام بشكل نهائي ولا يمكن التراجع عنها!
                    </p>
                    <p style="color: #856404; margin-bottom: 5px;"><strong>عدد الطلاب الحالي: <?php echo number_format($total_students); ?> طالب</strong></p>
                    <p style="color: #856404; margin-bottom: 0;">
                        <strong>سيتم حذف:</strong>
                    </p>
                    <ul style="color: #856404; margin-right: 20px; margin-top: 10px;">
                        <li>جميع بيانات الطلاب</li>
                        <li>جميع حسابات المستخدمين المرتبطة بالطلاب</li>
                        <li>جميع تسجيلات الطلاب على المواد</li>
                        <li>جميع الدرجات</li>
                        <li>جميع المعدلات التراكمية والفصلية</li>
                        <li>جميع البيانات المرتبطة الأخرى</li>
                    </ul>
                </div>
                
                <form method="POST" onsubmit="return confirmDeleteAll()">
                    <input type="hidden" name="action" value="delete_all">
                    
                    <div class="form-group">
                        <label style="color: #f44336; font-weight: bold; font-size: 16px;">
                            للتأكيد، يرجى كتابة: <code style="background: #f4f4f4; padding: 5px 10px; border-radius: 3px;">حذف جميع الطلاب</code>
                        </label>
                        <input type="text" 
                               name="confirm_text" 
                               id="confirmDeleteText"
                               placeholder="اكتب: حذف جميع الطلاب"
                               required
                               style="width: 100%; max-width: 400px; padding: 12px; border: 2px solid #f44336; border-radius: 5px; font-size: 16px;"
                               oninput="checkConfirmText()">
                        <small style="display: block; margin-top: 5px; color: #666;">
                            يجب كتابة النص بالضبط كما هو موضح أعلاه
                        </small>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <button type="submit" 
                                id="deleteAllBtn"
                                class="btn btn-danger" 
                                disabled
                                style="padding: 12px 30px; font-size: 16px; font-weight: bold;">
                            🗑️ حذف جميع الطلاب (<?php echo number_format($total_students); ?>)
                        </button>
                        <button type="button" 
                                onclick="document.getElementById('deleteAllForm').style.display='none'; document.getElementById('confirmDeleteText').value=''; checkConfirmText();" 
                                class="btn btn-secondary">
                            إلغاء
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- قسم البحث -->
        <div class="card" style="margin-bottom: 20px;">
            <div class="card-header">
                <h3>🔍 البحث عن طالب</h3>
            </div>
            <div style="padding: 20px;">
                <div class="form-group" style="position: relative; display: inline-block; width: 100%; max-width: 600px;">
                    <label for="searchInput" style="display: block; margin-bottom: 8px; font-weight: bold;">
                        ابحث عن طالب:
                    </label>
                    <div style="position: relative; display: flex; align-items: center;">
                        <input type="text" 
                               id="searchInput" 
                               placeholder="ابحث بالرقم الجامعي، الاسم، الرقم الوطني، البريد الإلكتروني، الكلية، أو التخصص..."
                               style="padding: 10px 40px 10px 10px; width: 100%; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
                               onkeyup="filterStudents()">
                        <button type="button" 
                                id="clearSearchBtn" 
                                onclick="clearSearch()" 
                                style="position: absolute; left: 10px; background: #f44336; color: white; border: none; border-radius: 50%; width: 28px; height: 28px; cursor: pointer; display: none; font-size: 16px; line-height: 1;"
                                title="مسح البحث">
                            ×
                        </button>
                    </div>
                    <small style="display: block; margin-top: 5px; color: #666;">
                        يمكنك البحث في جميع الحقول: الرقم الجامعي، الاسم، الرقم الوطني، البريد الإلكتروني، الكلية، أو التخصص
                    </small>
                </div>
                <div id="searchResults" style="margin-top: 10px; color: #666; font-size: 14px;"></div>
            </div>
        </div>
        
        <p style="margin: 0 0 10px 0; color: #666; font-size: 14px;">
            عرض <strong><?php echo count($students); ?></strong> طالب
            <?php if ($auth->hasRole('admin') && $total_students_in_db !== null): ?>
                <span style="color: #888;">(إجمالي في قاعدة البيانات: <?php echo $total_students_in_db; ?>)</span>
            <?php endif; ?>
        </p>
        <?php if (count($students) === 1 && $total_students_in_db === 1): ?>
        <p style="margin: 0 0 12px 0; padding: 10px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 6px; font-size: 13px;">
            <strong>ملاحظة:</strong> يظهر طالب واحد فقط لأن إجمالي الطلاب المسجلين في قاعدة البيانات هو 1. إذا رفعت ملفاً فيه عدة طلاب وظهرت رسالة «95 طالب موجودون مسبقاً»، فغالباً الملف كان يحتوي على <strong>نفس الرقم الجامعي مكرراً</strong> في عدة صفوف (النظام يضيف الطالب مرة واحدة ويتخطى التكرار). تأكد أن كل صف في الملف له <strong>رقم جامعي مختلف</strong>.
        </p>
        <?php endif; ?>
        <div class="table-container">
            <table id="studentsTable">
                <thead>
                    <tr>
                        <th>الرقم الجامعي</th>
                        <th>الاسم</th>
                        <th>الكلية</th>
                        <th>التخصص</th>
                        <th>الفصل الدراسي</th>
                        <th>السنة الأكاديمية</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                    </tr>
                </thead>
                <tbody id="studentsTableBody">
                    <?php foreach ($students as $student): ?>
                        <tr data-student-number="<?php echo htmlspecialchars($student['student_number']); ?>"
                            data-student-name-ar="<?php echo htmlspecialchars($student['full_name_ar']); ?>"
                            data-student-name-en="<?php echo htmlspecialchars($student['full_name_en'] ?? ''); ?>"
                            data-national-id="<?php echo htmlspecialchars($student['national_id'] ?? ''); ?>"
                            data-email="<?php echo htmlspecialchars($student['email'] ?? ''); ?>"
                            data-college-name="<?php echo htmlspecialchars($student['college_name']); ?>"
                            data-major-name="<?php echo htmlspecialchars($student['major_name']); ?>">
                            <td><?php echo htmlspecialchars($student['student_number']); ?></td>
                            <td><?php echo htmlspecialchars($student['full_name_ar']); ?></td>
                            <td><?php echo htmlspecialchars($student['college_name']); ?></td>
                            <td><?php echo (trim($student['major_name'] ?? '') === 'الكلية') ? '—' : htmlspecialchars($student['major_name']); ?></td>
                            <td><?php 
                                if (!empty($student['semester_number'])) {
                                    echo htmlspecialchars(getSemesterName($student['semester_number']));
                                } else {
                                    echo '-';
                                }
                            ?></td>
                            <td><?php 
                                if (!empty($student['academic_year'])) {
                                    echo htmlspecialchars($student['academic_year']);
                                } else {
                                    echo '-';
                                }
                            ?></td>
                            <td><?php 
                                $status = ['active' => 'نشط', 'graduated' => 'متخرج', 'suspended' => 'معلق', 'withdrawn' => 'منسحب'];
                                echo $status[$student['status']] ?? $student['status'];
                            ?></td>
                            <td>
                                <a href="edit_student.php?id=<?php echo $student['id']; ?>&return=<?php echo urlencode('students.php' . ($filter_college_id || $major_id ? '?college_id=' . (int)$filter_college_id . ($major_id ? '&major_id=' . (int)$major_id : '') : '')); ?>" class="btn btn-primary">تعديل</a>
                                <a href="student_details.php?id=<?php echo $student['id']; ?>" class="btn btn-primary">التفاصيل</a>
                                <a href="grades.php?student_id=<?php echo $student['id']; ?>" class="btn btn-success">الدرجات</a>
                                <button onclick="deleteStudent(<?php echo $student['id']; ?>, '<?php echo htmlspecialchars($student['full_name_ar'], ENT_QUOTES); ?>', '<?php echo htmlspecialchars($student['student_number'], ENT_QUOTES); ?>')" class="btn btn-danger">حذف</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <script>
    // دالة البحث والتصفية
    function filterStudents() {
        const searchInput = document.getElementById('searchInput');
        const clearBtn = document.getElementById('clearSearchBtn');
        const searchTerm = searchInput.value.toLowerCase().trim();
        const tableBody = document.getElementById('studentsTableBody');
        const rows = tableBody.getElementsByTagName('tr');
        const resultsDiv = document.getElementById('searchResults');
        
        // إظهار/إخفاء زر المسح
        if (searchTerm) {
            clearBtn.style.display = 'block';
        } else {
            clearBtn.style.display = 'none';
        }
        
        let visibleCount = 0;
        
        for (let i = 0; i < rows.length; i++) {
            const row = rows[i];
            const studentNumber = (row.getAttribute('data-student-number') || '').toLowerCase();
            const studentNameAr = (row.getAttribute('data-student-name-ar') || '').toLowerCase();
            const studentNameEn = (row.getAttribute('data-student-name-en') || '').toLowerCase();
            const nationalId = (row.getAttribute('data-national-id') || '').toLowerCase();
            const email = (row.getAttribute('data-email') || '').toLowerCase();
            const collegeName = (row.getAttribute('data-college-name') || '').toLowerCase();
            const majorName = (row.getAttribute('data-major-name') || '').toLowerCase();
            
            // البحث في جميع الحقول
            const matches = !searchTerm || 
                studentNumber.includes(searchTerm) ||
                studentNameAr.includes(searchTerm) ||
                studentNameEn.includes(searchTerm) ||
                nationalId.includes(searchTerm) ||
                email.includes(searchTerm) ||
                collegeName.includes(searchTerm) ||
                majorName.includes(searchTerm);
            
            if (matches) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        }
        
        // عرض عدد النتائج
        if (searchTerm) {
            resultsDiv.innerHTML = `تم العثور على <strong>${visibleCount}</strong> طالب`;
            if (visibleCount === 0) {
                resultsDiv.innerHTML = '<span style="color: #d32f2f;">لم يتم العثور على أي نتائج</span>';
            }
        } else {
            resultsDiv.innerHTML = '';
        }
    }
    
    // دالة مسح البحث
    function clearSearch() {
        const searchInput = document.getElementById('searchInput');
        searchInput.value = '';
        filterStudents();
        searchInput.focus();
    }
    
    // دالة لتحديث قائمة التخصصات في نموذج رفع Excel
    function loadMajorsForUpload() {
        const collegeId = document.getElementById('upload_college_id').value;
        const majorSelect = document.getElementById('upload_major_id');
        
        if (!collegeId) {
            majorSelect.innerHTML = '<option value="">-- اختر الكلية أولاً --</option>';
            majorSelect.disabled = true;
            return;
        }
        
        majorSelect.innerHTML = '<option value="">-- اختر التخصص (اختياري) --</option>';
        majorSelect.disabled = false;
        
        fetch(`students.php?action=get_majors&college_id=${collegeId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(major => {
                    const option = document.createElement('option');
                    option.value = major.id;
                    option.textContent = major.name_ar;
                    majorSelect.appendChild(option);
                });
                majorSelect.disabled = false;
                if (data.length === 1) majorSelect.selectedIndex = 1;
            })
            .catch(error => {
                console.error('Error loading majors:', error);
                majorSelect.innerHTML = '<option value="">خطأ في تحميل التخصصات</option>';
            });
    }
    
    // دالة لتحديث قائمة التخصصات في نموذج إضافة طالب
    function loadMajorsForAdd() {
        const collegeId = document.getElementById('add_college_id').value;
        const majorSelect = document.getElementById('add_major_id');
        
        if (!collegeId) {
            majorSelect.innerHTML = '<option value="">اختر الكلية أولاً</option>';
            majorSelect.disabled = true;
            return;
        }
        
        majorSelect.innerHTML = '<option value="">اختر التخصص</option>';
        majorSelect.disabled = false;
        
        fetch(`students.php?action=get_majors&college_id=${collegeId}`)
            .then(response => response.json())
            .then(data => {
                data.forEach(major => {
                    const option = document.createElement('option');
                    option.value = major.id;
                    option.textContent = major.name_ar;
                    majorSelect.appendChild(option);
                });
                majorSelect.disabled = false;
            })
            .catch(error => {
                console.error('Error loading majors:', error);
                majorSelect.innerHTML = '<option value="">خطأ في تحميل التخصصات</option>';
            });
    }
    
    // تحميل التخصصات عند فتح النموذج إذا كانت الكلية محددة
    document.addEventListener('DOMContentLoaded', function() {
        <?php if ($auth->hasRole('admin')): ?>
        // إذا كان admin، نتحقق من وجود كلية محددة مسبقاً
        const uploadCollegeId = document.getElementById('upload_college_id');
        const addCollegeId = document.getElementById('add_college_id');
        
        if (uploadCollegeId && uploadCollegeId.value) {
            loadMajorsForUpload();
        }
        if (addCollegeId && addCollegeId.value) {
            loadMajorsForAdd();
        }
        <?php else: ?>
        // إذا لم يكن admin، الكلية محددة مسبقاً، نحمل التخصصات
        const uploadCollegeId = document.getElementById('upload_college_id');
        const addCollegeId = document.getElementById('add_college_id');
        
        if (uploadCollegeId && uploadCollegeId.value) {
            loadMajorsForUpload();
        }
        if (addCollegeId && addCollegeId.value) {
            loadMajorsForAdd();
        }
        <?php endif; ?>
    });
    
    function deleteStudent(studentId, studentName, studentNumber) {
        if (confirm('هل أنت متأكد من حذف الطالب:\nالاسم: ' + studentName + '\nالرقم الجامعي: ' + studentNumber + '\n\nسيتم حذف جميع البيانات المرتبطة بهذا الطالب (الدرجات، التسجيلات، إلخ).\nهذا الإجراء لا يمكن التراجع عنه!')) {
            // إنشاء نموذج مخفي لإرسال طلب الحذف
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = 'students.php';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'delete';
            form.appendChild(actionInput);
            
            const studentIdInput = document.createElement('input');
            studentIdInput.type = 'hidden';
            studentIdInput.name = 'student_id';
            studentIdInput.value = studentId;
            form.appendChild(studentIdInput);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // دالة للتحقق من نص التأكيد لحذف جميع الطلاب
    function checkConfirmText() {
        const confirmText = document.getElementById('confirmDeleteText');
        const deleteBtn = document.getElementById('deleteAllBtn');
        const expectedText = 'حذف جميع الطلاب';
        
        if (confirmText && deleteBtn) {
            if (confirmText.value.trim() === expectedText) {
                deleteBtn.disabled = false;
                deleteBtn.style.opacity = '1';
            } else {
                deleteBtn.disabled = true;
                deleteBtn.style.opacity = '0.5';
            }
        }
    }
    
    // دالة تأكيد نهائية قبل الحذف
    function confirmDeleteAll() {
        const confirmText = document.getElementById('confirmDeleteText');
        const expectedText = 'حذف جميع الطلاب';
        
        if (!confirmText || confirmText.value.trim() !== expectedText) {
            alert('⚠️ يجب كتابة "حذف جميع الطلاب" بالضبط للتأكيد');
            return false;
        }
        
        <?php
        // جلب عدد الطلاب للاستخدام في JavaScript
        try {
            $js_total_students = $db->fetchOne("SELECT COUNT(*) as count FROM students")['count'];
        } catch (Exception $e) {
            $js_total_students = 0;
        }
        ?>
        const totalStudents = <?php echo $js_total_students; ?>;
        
        if (totalStudents === 0) {
            alert('لا يوجد طلاب لحذفهم');
            return false;
        }
        
        const finalConfirm = confirm(
            '⚠️⚠️⚠️ تحذير نهائي ⚠️⚠️⚠️\n\n' +
            'أنت على وشك حذف جميع الطلاب (' + totalStudents + ' طالب) من النظام!\n\n' +
            'هذه العملية:\n' +
            '• لا يمكن التراجع عنها\n' +
            '• ستحذف جميع بيانات الطلاب\n' +
            '• ستحذف جميع الدرجات والتسجيلات\n' +
            '• ستحذف جميع المعدلات\n' +
            '• ستحذف جميع البيانات المرتبطة\n\n' +
            'هل أنت متأكد تماماً من رغبتك في المتابعة؟'
        );
        
        if (!finalConfirm) {
            return false;
        }
        
        // تأكيد إضافي
        const secondConfirm = confirm(
            '⚠️ تأكيد نهائي أخير ⚠️\n\n' +
            'أنت على وشك حذف ' + totalStudents + ' طالب بشكل نهائي!\n\n' +
            'اضغط "موافق" فقط إذا كنت متأكداً 100%'
        );
        
        return secondConfirm;
    }
    </script>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

