<?php
/**
 * نقل محتوى فصل دراسي (المواد وتسجيلات الطلاب) إلى فصل آخر
 * نفس الكلية ونفس التخصص، سنة دراسية مختلفة
 * Transfer semester content (courses + student registrations) to another semester
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasRole('admin') && !$auth->hasRole('college_admin'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$message = '';
$message_type = '';

// طلبات AJAX: جلب التخصصات حسب الكلية، وجلب الفصول حسب التخصص
if (isset($_GET['action'])) {
    if ($_GET['action'] === 'get_majors' && isset($_GET['college_id'])) {
        header('Content-Type: application/json');
        $cid = (int)$_GET['college_id'];
        if ($cid <= 0) {
            echo json_encode([]);
            exit;
        }
        if (!$auth->belongsToCollege($cid) && !$auth->hasRole('admin')) {
            echo json_encode([]);
            exit;
        }
        $majors = $db->fetchAll(
            "SELECT id, name_ar FROM majors WHERE college_id = ? ORDER BY name_ar",
            [$cid]
        );
        echo json_encode($majors);
        exit;
    }
    if ($_GET['action'] === 'get_semesters' && isset($_GET['major_id'])) {
        header('Content-Type: application/json');
        $mid = (int)$_GET['major_id'];
        if ($mid <= 0) {
            echo json_encode([]);
            exit;
        }
        $major = $db->fetchOne("SELECT college_id FROM majors WHERE id = ?", [$mid]);
        if (!$major || (!$auth->belongsToCollege($major['college_id']) && !$auth->hasRole('admin'))) {
            echo json_encode([]);
            exit;
        }
        $semesters = $db->fetchAll(
            "SELECT s.id, s.semester_number, s.academic_year, s.major_id,
                    m.name_ar as major_name, c.name_ar as college_name
             FROM semesters s
             INNER JOIN majors m ON s.major_id = m.id
             INNER JOIN colleges c ON m.college_id = c.id
             WHERE s.major_id = ?
             ORDER BY s.academic_year DESC, s.semester_number DESC",
            [$mid]
        );
        echo json_encode($semesters);
        exit;
    }
}

// معالجة النقل
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transfer'])) {
    $source_semester_id = (int)($_POST['source_semester_id'] ?? 0);
    $target_academic_year = trim($_POST['target_academic_year'] ?? '');
    $target_semester_number = (int)($_POST['target_semester_number'] ?? 0);

    if ($source_semester_id <= 0 || $target_semester_number < 1 || $target_semester_number > 20 || empty($target_academic_year)) {
        $message = 'يرجى اختيار الفصل المصدر وتحديد رقم الفصل المستهدف والعام الدراسي.';
        $message_type = 'error';
    } else {
        $source = $db->fetchOne(
            "SELECT s.*, m.college_id, m.name_ar as major_name, c.name_ar as college_name
             FROM semesters s
             JOIN majors m ON s.major_id = m.id
             JOIN colleges c ON m.college_id = c.id
             WHERE s.id = ?",
            [$source_semester_id]
        );

        if (!$source) {
            $message = 'الفصل المصدر غير موجود.';
            $message_type = 'error';
        } elseif (!$auth->belongsToCollege($source['college_id']) && !$auth->hasRole('admin')) {
            $message = 'ليس لديك صلاحية للوصول إلى هذا الفصل.';
            $message_type = 'error';
        } else {
            $major_id = (int)$source['major_id'];

            // البحث عن الفصل المستهدف أو إنشاؤه
            $target = $db->fetchOne(
                "SELECT id FROM semesters
                 WHERE major_id = ? AND semester_number = ? AND academic_year = ?",
                [$major_id, $target_semester_number, $target_academic_year]
            );

            if (!$target) {
                try {
                    $db->query(
                        "INSERT INTO semesters (major_id, semester_number, academic_year, is_active)
                         VALUES (?, ?, ?, 1)",
                        [$major_id, $target_semester_number, $target_academic_year]
                    );
                    $target_semester_id = (int)$db->getConnection()->lastInsertId();
                } catch (Exception $e) {
                    $message = 'فشل إنشاء الفصل المستهدف: ' . $e->getMessage();
                    $message_type = 'error';
                    $target_semester_id = 0;
                }
            } else {
                $target_semester_id = (int)$target['id'];
            }

            if ($target_semester_id > 0) {
                if ($target_semester_id === $source_semester_id) {
                    $message = 'الفصل المستهدف لا يمكن أن يكون نفس الفصل المصدر.';
                    $message_type = 'error';
                } else {
                    $db->getConnection()->beginTransaction();
                    $errors = [];
                    $courses_copied = 0;
                    $course_id_map = []; // old_course_id => new_course_id (أو نفس الـ id إذا لم تُنسخ المادة)

                    try {
                        // 1) نسخ المواد المرتبطة بالفصل المصدر (semester_id = source) إلى الفصل المستهدف
                        $source_courses = $db->fetchAll(
                            "SELECT id, code, name_ar, name_en, credit_hours, is_required, semester_id
                             FROM courses
                             WHERE major_id = ? AND semester_id = ?",
                            [$major_id, $source_semester_id]
                        );

                        $target_code_suffix = '_' . str_replace('-', '', $target_academic_year);

                        foreach ($source_courses as $c) {
                            $old_id = (int)$c['id'];
                            $code = $c['code'];
                            $new_code = $code . $target_code_suffix;
                            $existing = $db->fetchOne(
                                "SELECT id FROM courses WHERE major_id = ? AND code = ?",
                                [$major_id, $new_code]
                            );
                            if ($existing) {
                                $course_id_map[$old_id] = (int)$existing['id'];
                            } else {
                                $db->query(
                                    "INSERT INTO courses (major_id, semester_id, code, name_ar, name_en, credit_hours, is_required)
                                     VALUES (?, ?, ?, ?, ?, ?, ?)",
                                    [
                                        $major_id,
                                        $target_semester_id,
                                        $new_code,
                                        $c['name_ar'],
                                        $c['name_en'] ?? null,
                                        (int)$c['credit_hours'],
                                        !empty($c['is_required']) ? 1 : 0
                                    ]
                                );
                                $course_id_map[$old_id] = (int)$db->getConnection()->lastInsertId();
                                $courses_copied++;
                            }
                        }

                        // مواد تظهر في تسجيلات الفصل المصدر لكن لم تُنسخ بعد (semester_id NULL أو مختلف)
                        $used_course_ids = $db->fetchAll(
                            "SELECT DISTINCT course_id FROM student_courses WHERE semester_id = ?",
                            [$source_semester_id]
                        );
                        foreach ($used_course_ids as $uc) {
                            $cid = (int)$uc['course_id'];
                            if (isset($course_id_map[$cid])) {
                                continue;
                            }
                            $course_row = $db->fetchOne(
                                "SELECT id, code, name_ar, name_en, credit_hours, is_required, semester_id FROM courses WHERE id = ? AND major_id = ?",
                                [$cid, $major_id]
                            );
                            if (!$course_row) {
                                $course_id_map[$cid] = $cid;
                                continue;
                            }
                            // إذا المادة عامة (semester_id NULL) ننسخها للفصل المستهدف أيضاً
                            if (empty($course_row['semester_id'])) {
                                $new_code = $course_row['code'] . $target_code_suffix;
                                $existing = $db->fetchOne(
                                    "SELECT id FROM courses WHERE major_id = ? AND code = ?",
                                    [$major_id, $new_code]
                                );
                                if ($existing) {
                                    $course_id_map[$cid] = (int)$existing['id'];
                                } else {
                                    $db->query(
                                        "INSERT INTO courses (major_id, semester_id, code, name_ar, name_en, credit_hours, is_required)
                                         VALUES (?, ?, ?, ?, ?, ?, ?)",
                                        [
                                            $major_id,
                                            $target_semester_id,
                                            $new_code,
                                            $course_row['name_ar'],
                                            $course_row['name_en'] ?? null,
                                            (int)$course_row['credit_hours'],
                                            !empty($course_row['is_required']) ? 1 : 0
                                        ]
                                    );
                                    $course_id_map[$cid] = (int)$db->getConnection()->lastInsertId();
                                    $courses_copied++;
                                }
                            } else {
                                $course_id_map[$cid] = $cid;
                            }
                        }

                        // 2) نسخ تسجيلات الطلاب في المواد إلى الفصل المستهدف
                        $registrations = $db->fetchAll(
                            "SELECT student_id, course_id FROM student_courses
                             WHERE semester_id = ?
                             ORDER BY student_id, course_id",
                            [$source_semester_id]
                        );

                        $added = 0;
                        $skipped = 0;

                        foreach ($registrations as $row) {
                            $target_course_id = isset($course_id_map[(int)$row['course_id']]) ? $course_id_map[(int)$row['course_id']] : (int)$row['course_id'];
                            $exists = $db->fetchOne(
                                "SELECT id FROM student_courses
                                 WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                                [$row['student_id'], $target_course_id, $target_semester_id]
                            );
                            if ($exists) {
                                $skipped++;
                                continue;
                            }
                            try {
                                $db->query(
                                    "INSERT INTO student_courses (student_id, course_id, semester_id, registration_date)
                                     VALUES (?, ?, ?, CURDATE())",
                                    [$row['student_id'], $target_course_id, $target_semester_id]
                                );
                                $added++;
                            } catch (Exception $e) {
                                $errors[] = $e->getMessage();
                            }
                        }

                        $db->getConnection()->commit();

                        $message = 'تم نقل كل محتوى الفصل المصدر إلى الفصل المستهدف بنجاح.<br>';
                        $message .= '- عدد المواد المنقولة/المضافة للفصل المستهدف: ' . $courses_copied . '<br>';
                        $message .= '- عدد التسجيلات المنقولة: ' . $added . '<br>';
                        if ($skipped > 0) {
                            $message .= '- تسجيلات موجودة مسبقاً (تم تخطيها): ' . $skipped . '<br>';
                        }
                        $message .= '- الفصل المستهدف: الفصل ' . $target_semester_number . ' - ' . htmlspecialchars($target_academic_year);
                        if (!empty($errors)) {
                            $message .= '<br><br>تحذيرات: ' . implode('<br>', array_slice($errors, 0, 5));
                        }
                        $message_type = 'success';

                    } catch (Exception $e) {
                        $db->getConnection()->rollBack();
                        $message = 'حدث خطأ أثناء النقل: ' . $e->getMessage();
                        $message_type = 'error';
                    }
                }
            }
        }
    }
}

// جلب الكليات
$user = $auth->getCurrentUser();
$college_id = $user['college_id'] ?? null;

if ($auth->hasRole('admin')) {
    $colleges = $db->fetchAll("SELECT id, name_ar FROM colleges ORDER BY name_ar");
} else {
    $colleges = $college_id ? $db->fetchAll("SELECT id, name_ar FROM colleges WHERE id = ? ORDER BY name_ar", [$college_id]) : [];
}

include __DIR__ . '/includes/header.php';
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نقل محتوى فصل دراسي</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .transfer-form { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); margin-top: 20px; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; margin-bottom: 8px; font-weight: bold; color: #333; }
        .form-group input, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 16px; box-sizing: border-box; }
        .info-box { background: #e8f5e9; padding: 15px; border-radius: 8px; margin: 20px 0; border-right: 4px solid #4caf50; }
        .section-title { font-size: 1.1em; color: #1976d2; margin: 25px 0 12px 0; padding-bottom: 6px; border-bottom: 1px solid #e0e0e0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>نقل محتوى فصل دراسي</h1>

        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?>" style="margin-bottom: 20px;">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="info-box">
            <h3>ℹ️ الوظيفة</h3>
            <p>نقل <strong>كل محتوى الفصل المصدر</strong> (المواد + تسجيلات الطلاب) إلى فصل مستهدف في <strong>نفس الكلية ونفس التخصص</strong> مع <strong>اختلاف السنة الدراسية</strong>.</p>
            <ul style="margin-right: 20px;">
                <li><strong>المواد:</strong> تُنسخ المواد المرتبطة بالفصل المصدر (semester_id = الفصل المصدر) إلى الفصل المستهدف بكواد جديدة (مثلاً: LAW101 → LAW101_20252026).</li>
                <li><strong>التسجيلات:</strong> يُنسخ كل طالب مسجّل في كل مادة من الفصل المصدر إلى الفصل المستهدف (مع ربط الطلاب بالمواد الجديدة عند الحاجة).</li>
                <li>إذا لم يكن الفصل المستهدف موجوداً، يتم إنشاؤه تلقائياً.</li>
                <li>الدرجات لا تُنقل؛ يتم نسخ المواد وتسجيل الطلاب فقط.</li>
            </ul>
        </div>

        <div class="transfer-form">
            <form method="POST">
                <div class="section-title">1. اختر الفصل المصدر (الكلية ← التخصص ← الفصل)</div>
                <div class="form-group">
                    <label>الكلية:</label>
                    <select id="college_id" required onchange="onCollegeChange()">
                        <option value="">-- اختر الكلية أولاً --</option>
                        <?php foreach ($colleges as $c): ?>
                            <option value="<?php echo (int)$c['id']; ?>"><?php echo htmlspecialchars($c['name_ar']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>التخصص:</label>
                    <select id="major_id" required onchange="onMajorChange()" disabled>
                        <option value="">-- اختر التخصص --</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>الفصل الدراسي المصدر:</label>
                    <select name="source_semester_id" id="source_semester" required disabled>
                        <option value="">-- اختر الفصل الدراسي --</option>
                    </select>
                </div>

                <div class="section-title">2. حدد الفصل المستهدف (نفس التخصص، سنة مختلفة)</div>
                <div class="form-group">
                    <label>رقم الفصل المستهدف:</label>
                    <input type="number" name="target_semester_number" id="target_semester_number" min="1" max="20" required placeholder="مثال: 2">
                    <small>مثال: 1 أو 2 أو 3 … (نفس رقم الفصل أو مختلف)</small>
                </div>
                <div class="form-group">
                    <label>العام الدراسي المستهدف:</label>
                    <input type="text" name="target_academic_year" id="target_academic_year" required placeholder="مثال: 2025-2026">
                    <small>يجب أن يختلف عن عام الفصل المصدر (سنة أخرى)</small>
                </div>

                <div class="form-group" style="margin-top: 25px;">
                    <button type="submit" name="transfer" class="btn btn-primary"
                            onclick="return confirm('هل أنت متأكد من نقل تسجيلات الطلاب من الفصل المصدر إلى الفصل المستهدف؟');">
                        تنفيذ النقل
                    </button>
                    <a href="semesters.php" class="btn btn-secondary" style="margin-right:10px;">العودة إلى الفصول</a>
                </div>
            </form>
        </div>
    </div>

    <script>
        function onCollegeChange() {
            var collegeSelect = document.getElementById('college_id');
            var majorSelect = document.getElementById('major_id');
            var semesterSelect = document.getElementById('source_semester');
            var collegeId = collegeSelect.value;

            majorSelect.innerHTML = '<option value="">-- اختر التخصص --</option>';
            majorSelect.disabled = !collegeId;
            semesterSelect.innerHTML = '<option value="">-- اختر الفصل الدراسي --</option>';
            semesterSelect.disabled = true;

            if (!collegeId) return;
            fetch('transfer_semester_content.php?action=get_majors&college_id=' + encodeURIComponent(collegeId))
                .then(function(r) { return r.json(); })
                .then(function(majors) {
                    majors.forEach(function(m) {
                        var opt = document.createElement('option');
                        opt.value = m.id;
                        opt.textContent = m.name_ar;
                        majorSelect.appendChild(opt);
                    });
                })
                .catch(function() { majorSelect.innerHTML = '<option value="">-- خطأ في التحميل --</option>'; });
        }

        function onMajorChange() {
            var majorSelect = document.getElementById('major_id');
            var semesterSelect = document.getElementById('source_semester');
            var majorId = majorSelect.value;

            semesterSelect.innerHTML = '<option value="">-- اختر الفصل الدراسي --</option>';
            semesterSelect.disabled = !majorId;

            if (!majorId) return;
            fetch('transfer_semester_content.php?action=get_semesters&major_id=' + encodeURIComponent(majorId))
                .then(function(r) { return r.json(); })
                .then(function(semesters) {
                    semesters.forEach(function(s) {
                        var opt = document.createElement('option');
                        opt.value = s.id;
                        opt.setAttribute('data-academic-year', s.academic_year || '');
                        opt.setAttribute('data-semester-number', s.semester_number);
                        opt.textContent = 'الفصل ' + s.semester_number + ' - ' + (s.academic_year || '');
                        semesterSelect.appendChild(opt);
                    });
                    semesterSelect.disabled = false;
                })
                .catch(function() {
                    semesterSelect.innerHTML = '<option value="">-- خطأ في التحميل --</option>';
                });
        }

        (function() {
            var collegeSelect = document.getElementById('college_id');
            if (collegeSelect && collegeSelect.options.length === 2) {
                collegeSelect.selectedIndex = 1;
                onCollegeChange();
            }
        })();
    </script>
    <?php include __DIR__ . '/includes/footer.php'; ?>
</body>
</html>
