<?php
/**
 * رفع واستعادة قاعدة البيانات
 * Upload and Restore Database
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

// التحقق من الصلاحيات (فقط للمديرين)
$user = $auth->getCurrentUser();
if ($user['role'] !== 'admin' && $user['role'] !== 'super_admin') {
    die('ليس لديك صلاحية للوصول إلى هذه الصفحة');
}

$db = Database::getInstance();
$message = '';
$message_type = '';

// معالجة رفع الملف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload') {
    if (!isset($_FILES['sql_file']) || $_FILES['sql_file']['error'] !== UPLOAD_ERR_OK) {
        $message = 'حدث خطأ أثناء رفع الملف. يرجى المحاولة مرة أخرى.';
        $message_type = 'error';
    } else {
        $file = $_FILES['sql_file'];
        $file_name = $file['name'];
        $file_tmp = $file['tmp_name'];
        $file_size = $file['size'];
        $file_type = $file['type'];
        
        // التحقق من نوع الملف
        $allowed_extensions = ['sql', 'txt'];
        $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_extensions)) {
            $message = 'نوع الملف غير مدعوم. يرجى رفع ملف SQL فقط (.sql أو .txt)';
            $message_type = 'error';
        } elseif ($file_size > 100 * 1024 * 1024) { // 100 MB
            $message = 'حجم الملف كبير جداً. الحد الأقصى 100 MB';
            $message_type = 'error';
        } else {
            // قراءة محتوى الملف
            $sql_content = file_get_contents($file_tmp);
            
            if (empty($sql_content)) {
                $message = 'الملف فارغ أو لا يمكن قراءته.';
                $message_type = 'error';
            } else {
                try {
                    // التحقق من وجود نسخة احتياطية قبل الاستعادة
                    $backup_created = false;
                    $backup_file = '';
                    
                    // إنشاء نسخة احتياطية تلقائياً قبل الاستعادة
                    $backup_dir = __DIR__ . '/backups/';
                    if (!file_exists($backup_dir)) {
                        @mkdir($backup_dir, 0777, true);
                    }
                    
                    $timestamp = date('Y-m-d_H-i-s');
                    $backup_file = $backup_dir . 'backup_before_restore_' . $timestamp . '.sql';
                    
                    // محاولة إنشاء نسخة احتياطية باستخدام mysqldump
                    $password_arg = !empty(DB_PASS) ? '-p' . escapeshellarg(DB_PASS) : '';
                    $backup_command = sprintf(
                        'mysqldump -h %s -u %s %s %s > %s 2>&1',
                        escapeshellarg(DB_HOST),
                        escapeshellarg(DB_USER),
                        $password_arg,
                        escapeshellarg(DB_NAME),
                        escapeshellarg($backup_file)
                    );
                    
                    exec($backup_command, $backup_output, $backup_return);
                    
                    if ($backup_return === 0 && file_exists($backup_file) && filesize($backup_file) > 0) {
                        $backup_created = true;
                    } else {
                        // إذا فشل mysqldump، إنشاء نسخة احتياطية يدوياً
                        $backup_created = createManualBackup($backup_file);
                    }
                    
                    // بدء المعاملة
                    $db->getConnection()->beginTransaction();
                    
                    // تقسيم SQL إلى أوامر منفصلة
                    // إزالة التعليقات والأوامر غير الضرورية
                    $sql_content = preg_replace('/--.*$/m', '', $sql_content); // إزالة التعليقات
                    $sql_content = preg_replace('/\/\*.*?\*\//s', '', $sql_content); // إزالة التعليقات متعددة الأسطر
                    $sql_content = preg_replace('/^USE\s+.*?;$/mi', '', $sql_content); // إزالة USE statements
                    $sql_content = preg_replace('/^SET\s+.*?;$/mi', '', $sql_content); // إزالة SET statements
                    
                    // تقسيم إلى أوامر
                    $statements = array_filter(
                        array_map('trim', explode(';', $sql_content)),
                        function($stmt) {
                            $stmt = trim($stmt);
                            return !empty($stmt) && 
                                   !preg_match('/^(SET|USE|\/\*|--)/i', $stmt) &&
                                   strlen($stmt) > 5;
                        }
                    );
                    
                    $success_count = 0;
                    $error_count = 0;
                    $errors = [];
                    
                    // تنفيذ الأوامر
                    foreach ($statements as $statement) {
                        $statement = trim($statement);
                        if (empty($statement)) continue;
                        
                        try {
                            $db->query($statement);
                            $success_count++;
                        } catch (PDOException $e) {
                            // تجاهل بعض الأخطاء الشائعة
                            $error_msg = $e->getMessage();
                            if (strpos($error_msg, 'Duplicate entry') === false &&
                                strpos($error_msg, 'already exists') === false &&
                                strpos($error_msg, 'Unknown table') === false) {
                                $error_count++;
                                $errors[] = $error_msg;
                                
                                // إذا كان الخطأ خطيراً، إيقاف التنفيذ
                                if (strpos($error_msg, 'syntax error') !== false ||
                                    strpos($error_msg, 'parse error') !== false) {
                                    throw $e;
                                }
                            }
                        }
                    }
                    
                    // تأكيد المعاملة
                    $db->getConnection()->commit();
                    
                    $message = "✓ تم استعادة قاعدة البيانات بنجاح!\n";
                    $message .= "عدد الأوامر المنفذة: $success_count\n";
                    if ($error_count > 0) {
                        $message .= "عدد الأخطاء (تم تجاهلها): $error_count\n";
                    }
                    if ($backup_created) {
                        $message .= "\n✓ تم إنشاء نسخة احتياطية تلقائياً قبل الاستعادة: " . basename($backup_file);
                    }
                    $message_type = 'success';
                    
                } catch (Exception $e) {
                    // إرجاع المعاملة في حالة الخطأ
                    $db->getConnection()->rollBack();
                    $message = '✗ فشل استعادة قاعدة البيانات: ' . $e->getMessage();
                    $message_type = 'error';
                }
            }
        }
    }
}

/**
 * إنشاء نسخة احتياطية يدوياً
 */
function createManualBackup($backup_file) {
    try {
        $db = Database::getInstance();
        $fp = fopen($backup_file, 'w');
        if (!$fp) {
            return false;
        }
        
        fwrite($fp, "-- نسخة احتياطية قبل الاستعادة\n");
        fwrite($fp, "-- Backup before restore\n");
        fwrite($fp, "-- Generated: " . date('Y-m-d H:i:s') . "\n\n");
        
        $tables = $db->fetchAll("SHOW TABLES");
        $table_key = 'Tables_in_' . DB_NAME;
        
        foreach ($tables as $table_row) {
            $table_name = $table_row[$table_key];
            
            $create_table = $db->fetchOne("SHOW CREATE TABLE `{$table_name}`");
            fwrite($fp, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($fp, $create_table['Create Table'] . ";\n\n");
            
            $rows = $db->fetchAll("SELECT * FROM `{$table_name}`");
            if (count($rows) > 0) {
                $columns = array_keys($rows[0]);
                $column_names = '`' . implode('`, `', $columns) . '`';
                
                foreach ($rows as $row) {
                    $values = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $values[] = 'NULL';
                        } else {
                            $escaped = addslashes($value);
                            $escaped = str_replace(["\n", "\r", "\t"], ["\\n", "\\r", "\\t"], $escaped);
                            $values[] = "'" . $escaped . "'";
                        }
                    }
                    $values_str = implode(', ', $values);
                    fwrite($fp, "INSERT INTO `{$table_name}` ({$column_names}) VALUES ({$values_str});\n");
                }
                fwrite($fp, "\n");
            }
        }
        
        fclose($fp);
        return true;
    } catch (Exception $e) {
        if (isset($fp) && is_resource($fp)) {
            fclose($fp);
        }
        return false;
    }
}

/**
 * تنسيق حجم الملف
 */
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>رفع واستعادة قاعدة البيانات - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            min-height: 100vh;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2em;
            margin-bottom: 10px;
        }
        
        .content {
            padding: 30px;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-right: 4px solid;
            white-space: pre-line;
        }
        
        .alert-success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        
        .alert-error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        
        .alert-warning {
            background: #fff3cd;
            border-color: #ffc107;
            color: #856404;
        }
        
        .alert-info {
            background: #d1ecf1;
            border-color: #17a2b8;
            color: #0c5460;
        }
        
        .upload-form {
            background: #f8f9fa;
            border: 2px dashed #667eea;
            border-radius: 10px;
            padding: 30px;
            text-align: center;
            margin: 20px 0;
        }
        
        .upload-form input[type="file"] {
            display: block;
            margin: 20px auto;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            background: white;
            width: 100%;
            max-width: 400px;
        }
        
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            font-size: 16px;
            transition: transform 0.2s, box-shadow 0.2s;
            margin: 5px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .btn-danger {
            background: #dc3545;
        }
        
        .btn-secondary {
            background: #6c757d;
        }
        
        .info-box {
            background: #e7f3ff;
            border-right: 4px solid #2196F3;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .info-box h3 {
            color: #1976D2;
            margin-bottom: 10px;
        }
        
        .info-box ul {
            margin-right: 20px;
            line-height: 1.8;
        }
        
        .warning-box {
            background: #fff3cd;
            border-right: 4px solid #ffc107;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .warning-box h3 {
            color: #856404;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="header">
            <h1>📤 رفع واستعادة قاعدة البيانات</h1>
            <p>Upload and Restore Database</p>
        </div>
        
        <div class="content">
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <div class="warning-box">
                <h3>⚠️ تحذير مهم</h3>
                <p><strong>استعادة قاعدة البيانات ستحذف جميع البيانات الحالية وتستبدلها بالبيانات من الملف المرفوع!</strong></p>
                <p>سيتم إنشاء نسخة احتياطية تلقائياً قبل الاستعادة، ولكن يُنصح بإنشاء نسخة احتياطية يدوياً أيضاً.</p>
            </div>
            
            <div class="info-box">
                <h3>📋 متطلبات الملف</h3>
                <ul>
                    <li>نوع الملف: ملف SQL (.sql أو .txt)</li>
                    <li>الحد الأقصى للحجم: 100 MB</li>
                    <li>التنسيق: يجب أن يكون الملف بصيغة SQL صالحة</li>
                </ul>
            </div>
            
            <form method="POST" enctype="multipart/form-data" class="upload-form" onsubmit="return confirm('هل أنت متأكد من استعادة قاعدة البيانات؟ سيتم حذف جميع البيانات الحالية!');">
                <input type="hidden" name="action" value="upload">
                
                <h3>اختر ملف قاعدة البيانات</h3>
                <input type="file" name="sql_file" accept=".sql,.txt" required>
                
                <div style="margin-top: 20px;">
                    <button type="submit" class="btn btn-danger">
                        🔄 استعادة قاعدة البيانات
                    </button>
                    <a href="backup.php" class="btn btn-secondary">
                        💾 إنشاء نسخة احتياطية
                    </a>
                    <a href="dashboard.php" class="btn btn-secondary">
                        🏠 العودة للوحة التحكم
                    </a>
                </div>
            </form>
            
            <div class="info-box">
                <h3>💡 نصائح مهمة</h3>
                <ul>
                    <li>تأكد من أن الملف المرفوع هو نسخة احتياطية صالحة من قاعدة البيانات</li>
                    <li>يُنصح بإنشاء نسخة احتياطية قبل الاستعادة</li>
                    <li>يمكنك إنشاء نسخة احتياطية من صفحة <a href="backup.php">النسخ الاحتياطي</a></li>
                    <li>بعد الاستعادة، قد تحتاج إلى تسجيل الدخول مرة أخرى</li>
                    <li>تأكد من أن الملف يحتوي على جميع الجداول والبيانات المطلوبة</li>
                </ul>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

