<?php
/**
 * رفع الدرجات من ملف Excel/CSV
 * Upload Grades from Excel/CSV File
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';
require_once __DIR__ . '/classes/GradeCalculator.php';
require_once __DIR__ . '/classes/ExcelReader.php';

$auth = new Auth();
if (!$auth->isLoggedIn() || (!$auth->hasPermission('enter_grades') && !$auth->hasPermission('manage_grades'))) {
    header('Location: index.php');
    exit;
}

$db = Database::getInstance();
$gradeCalculator = new GradeCalculator();
$excelReader = new ExcelReader();
$message = '';
$errors = [];
$success_count = 0;
$error_count = 0;
$skipped_count = 0;
$preview_data = [];

// جلب الفصول الدراسية
$semesters = [];
if ($auth->hasRole('admin')) {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         ORDER BY s.academic_year DESC, s.semester_number DESC"
    );
} else {
    $semesters = $db->fetchAll(
        "SELECT s.*, m.name_ar as major_name 
         FROM semesters s 
         JOIN majors m ON s.major_id = m.id 
         WHERE m.college_id = ?
         ORDER BY s.academic_year DESC, s.semester_number DESC",
        [$_SESSION['college_id']]
    );
}

// معالجة رفع الملف
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'upload_grades') {
        $semester_id = isset($_POST['semester_id']) ? (int)$_POST['semester_id'] : 0;
        
        if (!$semester_id) {
            $message = '<div class="alert alert-error">✗ يجب اختيار الفصل الدراسي</div>';
        } elseif (!isset($_FILES['grades_file']) || $_FILES['grades_file']['error'] !== UPLOAD_ERR_OK) {
            $message = '<div class="alert alert-error">✗ حدث خطأ في رفع الملف. تأكد من اختيار ملف صحيح.</div>';
        } else {
            $file = $_FILES['grades_file'];
            $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if (!in_array($file_ext, ['csv', 'xlsx', 'xls'])) {
                $message = '<div class="alert alert-error">✗ نوع الملف غير مدعوم. يجب أن يكون CSV أو Excel (.xlsx, .xls)</div>';
            } else {
                // قراءة الملف
                $temp_file = $file['tmp_name'];
                $data = $excelReader->readFile($temp_file, $file_ext);
                
                if ($data === false || empty($data)) {
                    $message = '<div class="alert alert-error">✗ فشل قراءة الملف أو الملف فارغ</div>';
                } else {
                    // معالجة البيانات
                    try {
                        $db->getConnection()->beginTransaction();
                        
                        foreach ($data as $row_index => $row) {
                            $row_num = $row_index + 2; // +2 لأن الصف الأول هو العناوين
                            
                            // البحث عن رقم الطالب أو معرف الطالب
                            $student_number = null;
                            $student_id = null;
                            
                            // البحث في جميع المفاتيح المحتملة
                            $student_keys = ['student_number', 'student_id', 'id', 'رقم_جامعي', 'رقم جامعي', 'الرقم الجامعي', 'معرف', 'معرّف', 'رقم', 'الرقم'];
                            foreach ($student_keys as $key) {
                                if (isset($row[$key]) && !empty($row[$key])) {
                                    $value = trim($row[$key]);
                                    if (is_numeric($value) && strlen($value) <= 10) {
                                        // إذا كان رقم صغير، قد يكون معرف
                                        $student_id = (int)$value;
                                    } else {
                                        // وإلا فهو رقم جامعي
                                        $student_number = $value;
                                    }
                                    break;
                                }
                            }
                            
                            // البحث عن رمز المادة أو معرف المادة
                            $course_code = null;
                            $course_id = null;
                            
                            $course_keys = ['course_code', 'course_id', 'رمز_المادة', 'رمز المادة', 'رمز', 'الرمز', 'معرف_المادة', 'معرف المادة'];
                            foreach ($course_keys as $key) {
                                if (isset($row[$key]) && !empty($row[$key])) {
                                    $value = trim($row[$key]);
                                    if (is_numeric($value) && strlen($value) <= 10) {
                                        $course_id = (int)$value;
                                    } else {
                                        $course_code = $value;
                                    }
                                    break;
                                }
                            }
                            
                            // الحصول على الدرجة
                            $marks_input = null;
                            $marks_keys = ['marks', 'grade', 'degree', 'mark', 'درجة', 'الدرجة'];
                            foreach ($marks_keys as $key) {
                                if (isset($row[$key]) && $row[$key] !== '' && $row[$key] !== null) {
                                    $marks_input = trim($row[$key]);
                                    break;
                                }
                            }
                            
                            // التحقق من البيانات المطلوبة
                            if (!$student_number && !$student_id) {
                                $errors[] = "الصف $row_num: لم يتم العثور على رقم الطالب أو معرف الطالب";
                                $error_count++;
                                continue;
                            }
                            
                            if (!$course_code && !$course_id) {
                                $errors[] = "الصف $row_num: لم يتم العثور على رمز المادة أو معرف المادة";
                                $error_count++;
                                continue;
                            }
                            
                            if ($marks_input === null || $marks_input === '') {
                                $errors[] = "الصف $row_num: لم يتم العثور على الدرجة";
                                $error_count++;
                                continue;
                            }
                            
                            // البحث عن الطالب
                            $student = null;
                            if ($student_id) {
                                $student = $db->fetchOne("SELECT * FROM students WHERE id = ?", [$student_id]);
                            } elseif ($student_number) {
                                $student = $db->fetchOne("SELECT * FROM students WHERE student_number = ?", [$student_number]);
                            }
                            
                            if (!$student) {
                                $errors[] = "الصف $row_num: الطالب غير موجود (رقم: " . ($student_number ?: $student_id) . ")";
                                $error_count++;
                                continue;
                            }
                            
                            // التحقق من الصلاحيات
                            if (!$auth->belongsToCollege($student['college_id']) && !$auth->hasRole('admin')) {
                                $errors[] = "الصف $row_num: ليس لديك صلاحية لإدخال درجات لهذا الطالب";
                                $error_count++;
                                continue;
                            }
                            
                            // البحث عن المادة
                            $course = null;
                            if ($course_id) {
                                $course = $db->fetchOne("SELECT * FROM courses WHERE id = ?", [$course_id]);
                            } elseif ($course_code) {
                                $course = $db->fetchOne("SELECT * FROM courses WHERE code = ?", [$course_code]);
                            }
                            
                            if (!$course) {
                                $errors[] = "الصف $row_num: المادة غير موجودة (رمز: " . ($course_code ?: $course_id) . ")";
                                $error_count++;
                                continue;
                            }
                            
                            // التحقق من أن الطالب مسجل في المادة في هذا الفصل
                            $student_course = $db->fetchOne(
                                "SELECT * FROM student_courses 
                                 WHERE student_id = ? AND course_id = ? AND semester_id = ?",
                                [$student['id'], $course['id'], $semester_id]
                            );
                            
                            if (!$student_course) {
                                $errors[] = "الصف $row_num: الطالب " . htmlspecialchars($student['full_name_ar']) . " غير مسجل في المادة " . htmlspecialchars($course['name_ar']) . " في هذا الفصل";
                                $error_count++;
                                continue;
                            }
                            
                            // معالجة الدرجة
                            $grade_data = $gradeCalculator->processGradeInput($marks_input);
                            
                            if (!$grade_data) {
                                $errors[] = "الصف $row_num: درجة غير صحيحة: " . htmlspecialchars($marks_input);
                                $error_count++;
                                continue;
                            }
                            
                            // التحقق من نطاق الدرجة الرقمية
                            if (!$grade_data['is_text'] && ($grade_data['marks'] < 0 || $grade_data['marks'] > 100)) {
                                $errors[] = "الصف $row_num: الدرجة يجب أن تكون بين 0 و 100: " . $grade_data['marks'];
                                $error_count++;
                                continue;
                            }
                            
                            // إعداد القيم للحفظ
                            if ($grade_data['is_text']) {
                                $marks = $marks_input; // حفظ القيمة النصية الأصلية
                            } else {
                                $marks = $grade_data['marks'] !== null ? $grade_data['marks'] : 0;
                            }
                            $letter_grade = $grade_data['letter_grade'];
                            $points = $grade_data['points'];
                            
                            // التحقق من وجود درجة سابقة
                            $existing = $db->fetchOne(
                                "SELECT * FROM grades WHERE student_course_id = ?",
                                [$student_course['id']]
                            );
                            
                            if ($existing) {
                                // تحديث الدرجة
                                $db->query(
                                    "UPDATE grades SET marks = ?, letter_grade = ?, points = ?, updated_at = CURRENT_TIMESTAMP, entered_by = ?
                                     WHERE student_course_id = ?",
                                    [$marks, $letter_grade, $points, $_SESSION['user_id'], $student_course['id']]
                                );
                            } else {
                                // إضافة درجة جديدة
                                $db->query(
                                    "INSERT INTO grades (student_course_id, student_id, course_id, semester_id, marks, letter_grade, points, entered_by) 
                                     VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
                                    [
                                        $student_course['id'],
                                        $student['id'],
                                        $course['id'],
                                        $semester_id,
                                        $marks,
                                        $letter_grade,
                                        $points,
                                        $_SESSION['user_id']
                                    ]
                                );
                            }
                            
                            // تحديث المعدلات
                            try {
                                $gradeCalculator->updateAllGPAs($student['id']);
                            } catch (Exception $e) {
                                error_log("Error updating GPA for student {$student['id']}: " . $e->getMessage());
                            }
                            
                            $success_count++;
                        }
                        
                        $db->getConnection()->commit();
                        
                        if ($success_count > 0) {
                            $message = '<div class="alert alert-success">✓ تم رفع ' . $success_count . ' درجة بنجاح</div>';
                        }
                        if ($error_count > 0) {
                            $message .= '<div class="alert alert-warning">⚠ فشل رفع ' . $error_count . ' درجة</div>';
                        }
                        if ($skipped_count > 0) {
                            $message .= '<div class="alert alert-info">ℹ تم تخطي ' . $skipped_count . ' صف</div>';
                        }
                        
                    } catch (Exception $e) {
                        $db->getConnection()->rollBack();
                        $message = '<div class="alert alert-error">✗ خطأ في معالجة البيانات: ' . htmlspecialchars($e->getMessage()) . '</div>';
                        error_log("Error uploading grades: " . $e->getMessage());
                    }
                }
            }
        }
    } elseif ($_POST['action'] === 'preview') {
        // معاينة البيانات قبل الحفظ
        $semester_id = isset($_POST['semester_id']) ? (int)$_POST['semester_id'] : 0;
        
        if (!$semester_id) {
            $message = '<div class="alert alert-error">✗ يجب اختيار الفصل الدراسي</div>';
        } elseif (!isset($_FILES['grades_file']) || $_FILES['grades_file']['error'] !== UPLOAD_ERR_OK) {
            $message = '<div class="alert alert-error">✗ حدث خطأ في رفع الملف</div>';
        } else {
            $file = $_FILES['grades_file'];
            $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if (!in_array($file_ext, ['csv', 'xlsx', 'xls'])) {
                $message = '<div class="alert alert-error">✗ نوع الملف غير مدعوم</div>';
            } else {
                $temp_file = $file['tmp_name'];
                $data = $excelReader->readFile($temp_file, $file_ext);
                
                if ($data === false || empty($data)) {
                    $message = '<div class="alert alert-error">✗ فشل قراءة الملف</div>';
                } else {
                    // معاينة أول 20 صف
                    $preview_data = array_slice($data, 0, 20);
                    $message = '<div class="alert alert-info">ℹ معاينة أول ' . count($preview_data) . ' صف من ' . count($data) . ' صف</div>';
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>رفع الدرجات من Excel</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .upload-container {
            max-width: 1000px;
            margin: 30px auto;
            padding: 20px;
        }
        .info-box {
            background: #e3f2fd;
            border-right: 5px solid #2196F3;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
        .warning-box {
            background: #fff3cd;
            border-right: 5px solid #ffc107;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
        .errors-list {
            background: #ffebee;
            border-right: 5px solid #f44336;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
            max-height: 300px;
            overflow-y: auto;
        }
        .errors-list ul {
            margin: 10px 0;
            padding-right: 20px;
        }
        .preview-table {
            margin-top: 20px;
        }
        .file-upload-area {
            border: 2px dashed #ddd;
            border-radius: 8px;
            padding: 30px;
            text-align: center;
            background: #f9f9f9;
            margin: 20px 0;
        }
        .file-upload-area:hover {
            border-color: #667eea;
            background: #f0f4ff;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="page-header">
            <h2>رفع الدرجات من ملف Excel/CSV</h2>
            <a href="paste_grades_list.php" class="btn btn-success" style="margin-left: 10px;">📋 لصق القائمة</a>
            <a href="grades.php" class="btn btn-secondary">العودة لإدارة الدرجات</a>
        </div>
        
        <div class="alert alert-info" style="margin-bottom: 20px; padding: 14px 18px; border: 2px solid #0d6efd; border-radius: 8px; background: #e7f1ff;">
            <strong>💡 لتجنب مشاكل الترميز:</strong> استخدم 
            <a href="paste_grades_list.php" style="color: #0d6efd; font-weight: 600; text-decoration: underline;">لصق القائمة</a>
            لإنشاء ملف جاهز ثم ارفعه من هنا.
        </div>
        
        <?php echo $message; ?>
        
        <?php if (!empty($errors)): ?>
            <div class="errors-list">
                <h4>الأخطاء:</h4>
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-header">
                <h3>📤 رفع ملف الدرجات</h3>
            </div>
            <div style="padding: 20px;">
                <div style="background: #fff3cd; border: 2px solid #ffc107; border-radius: 8px; padding: 14px; margin-bottom: 20px;">
                    <strong style="color: #856404;">📋 لصق القائمة:</strong>
                    <a href="paste_grades_list.php" style="color: #0066cc; font-weight: 600; margin-right: 8px;">لصق القائمة وتوليد القالب</a>
                    — يتجنب مشاكل الترميز والأحرف العربية.
                </div>
                <div class="info-box">
                    <strong>📋 تنسيق الملف المطلوب:</strong>
                    <p>يجب أن يحتوي الملف على الأعمدة التالية:</p>
                    <ul style="margin-right: 20px; margin-top: 10px;">
                        <li><strong>student_number</strong> أو <strong>student_id</strong>: رقم الطالب الجامعي أو معرف الطالب</li>
                        <li><strong>course_code</strong> أو <strong>course_id</strong>: رمز المادة أو معرف المادة</li>
                        <li><strong>marks</strong> أو <strong>grade</strong> أو <strong>degree</strong>: الدرجة (رقم من 0-100 أو نص مثل: بد، غ، غش)</li>
                    </ul>
                </div>
                
                <div class="warning-box">
                    <strong>⚠️ ملاحظات مهمة:</strong>
                    <ul style="margin-right: 20px; margin-top: 10px;">
                        <li>يجب أن يكون الطالب مسجل في المادة في الفصل الدراسي المختار</li>
                        <li>إذا كانت الدرجة موجودة مسبقاً، سيتم تحديثها</li>
                        <li>الدرجات النصية المدعومة: "بد"، "غ"، "غياب"، "غش"، "غشش"</li>
                        <li>يجب أن يكون الملف بصيغة CSV أو Excel (.xlsx, .xls)</li>
                    </ul>
                </div>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="upload_grades">
                    
                    <div class="form-group">
                        <label><strong>الفصل الدراسي:</strong> <span style="color: red;">*</span></label>
                        <select name="semester_id" required style="width: 100%; padding: 10px; margin-top: 5px;">
                            <option value="">-- اختر الفصل الدراسي --</option>
                            <?php foreach ($semesters as $sem): ?>
                                <option value="<?php echo $sem['id']; ?>">
                                    فصل <?php echo $sem['semester_number']; ?> - <?php echo htmlspecialchars($sem['academic_year']); ?>
                                    (<?php echo htmlspecialchars($sem['major_name']); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group" style="margin-top: 20px;">
                        <label><strong>ملف الدرجات:</strong> <span style="color: red;">*</span></label>
                        <div class="file-upload-area">
                            <input type="file" name="grades_file" accept=".csv,.xlsx,.xls" required 
                                   style="display: block; margin: 0 auto; padding: 10px;">
                            <p style="margin-top: 15px; color: #666;">
                                اختر ملف Excel أو CSV يحتوي على الدرجات
                            </p>
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <button type="submit" class="btn btn-primary" style="padding: 12px 30px; font-size: 16px;">
                            ✅ رفع وحفظ الدرجات
                        </button>
                        <button type="submit" name="action" value="preview" class="btn btn-info" style="padding: 12px 30px; font-size: 16px; margin-right: 10px;">
                            👁️ معاينة البيانات
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <?php if (!empty($preview_data)): ?>
            <div class="card preview-table">
                <div class="card-header">
                    <h3>معاينة البيانات</h3>
                </div>
                <div style="padding: 20px;">
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>رقم الطالب</th>
                                    <th>اسم الطالب</th>
                                    <th>رمز المادة</th>
                                    <th>اسم المادة</th>
                                    <th>الدرجة</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($preview_data as $index => $row): ?>
                                    <tr>
                                        <td><?php echo $index + 1; ?></td>
                                        <td><?php echo htmlspecialchars($row['student_number'] ?? $row['student_id'] ?? '-'); ?></td>
                                        <td><?php echo htmlspecialchars($row['student_name'] ?? '-'); ?></td>
                                        <td><?php echo htmlspecialchars($row['course_code'] ?? $row['course_id'] ?? '-'); ?></td>
                                        <td>-</td>
                                        <td><?php echo htmlspecialchars($row['marks'] ?? $row['grade'] ?? $row['degree'] ?? '-'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="card" style="margin-top: 20px;">
            <div class="card-header">
                <h3>📝 مثال على تنسيق الملف</h3>
            </div>
            <div style="padding: 20px;">
                <p><strong>مثال على ملف CSV:</strong></p>
                <pre style="background: #f5f5f5; padding: 15px; border-radius: 5px; overflow-x: auto;">
student_number,course_code,marks
2021001,CS101,85
2021002,CS101,75
2021003,CS101,65
2021001,MATH101,90
2021002,MATH101,80
                </pre>
                
                <p style="margin-top: 20px;"><strong>مثال مع درجات نصية:</strong></p>
                <pre style="background: #f5f5f5; padding: 15px; border-radius: 5px; overflow-x: auto;">
student_number,course_code,marks
2021001,CS101,85
2021002,CS101,بد
2021003,CS101,غ
2021004,CS101,غش
                </pre>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>

